# 機能設計書 24-ヘッダー削除

## 概要

本ドキュメントは、HorseフレームワークにおけるHTTPレスポンスヘッダー削除機能の設計を記述する。THorseResponse.RemoveHeaderメソッドを使用して、既存のカスタムHTTPヘッダーをレスポンスから削除する機能について詳述する。

### 本機能の処理概要

**業務上の目的・背景**：HTTPレスポンスヘッダーの動的な制御において、特定のヘッダーを削除する必要が生じることがある。ミドルウェアで追加されたヘッダーを条件に応じて削除する、デバッグ用ヘッダーを本番環境で除去するなど、柔軟なヘッダー管理が求められる。本機能により、開発者は任意のHTTPヘッダーをレスポンスから削除できる。

**機能の利用シーン**：ミドルウェアで追加されたヘッダーの条件付き削除、開発環境専用ヘッダーの本番環境での除去、不要なセキュリティヘッダーの削除、レスポンスサイズ最適化のための不要ヘッダー除去などで使用される。

**主要な処理内容**：
1. 削除対象のヘッダー名を受け取る
2. CustomHeadersリストからヘッダー名のインデックスを検索する
3. 該当するヘッダーが存在すれば削除する
4. 自身のインスタンスを返却してメソッドチェーンを可能にする

**関連システム・外部連携**：本機能はHTTPプロトコルの標準的なヘッダー仕様に従う。削除されたヘッダーはクライアントに送信されない。

**権限による制御**：本機能自体に権限制御機能は含まれない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面に直接関連しない（APIレスポンス機能） |

## 機能種別

HTTP制御機能 / ヘッダー操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AName | string | Yes | 削除対象のヘッダー名 | - |

### 入力データソース

アプリケーションコードからの直接指定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | - | ヘッダー削除の結果、該当ヘッダーがレスポンスから除去される |

### 出力先

HTTPレスポンスヘッダーへの影響（該当ヘッダーの除去）

## 処理フロー

### 処理シーケンス

```
1. RemoveHeaderメソッド呼び出し
   └─ ヘッダー名を受け取る
2. ヘッダーのインデックス検索
   └─ FWebResponse.CustomHeaders.IndexOfNameで検索
3. ヘッダー存在確認と削除
   └─ インデックスが-1でなければDeleteで削除
4. 自身のインスタンス返却
   └─ メソッドチェーンを可能にする
```

### フローチャート

```mermaid
flowchart TD
    A[RemoveHeader呼び出し] --> B[AName受け取り]
    B --> C[IndexOfNameで検索]
    C --> D{インデックス != -1}
    D -->|Yes| E[CustomHeaders.Delete呼び出し]
    D -->|No| F[何もしない]
    E --> G[Self返却]
    F --> G
    G --> H[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-24-01 | フルーエントインターフェース | メソッドは自身のインスタンスを返却しメソッドチェーンを可能にする | 常時 |
| BR-24-02 | 存在しないヘッダーの無視 | 指定されたヘッダーが存在しない場合はエラーなく処理を終了 | ヘッダー削除時 |
| BR-24-03 | インデックスベースの削除 | ヘッダー名からインデックスを特定して削除を実行 | ヘッダー削除時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能ではデータベース操作は発生しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 本機能では明示的なエラー処理なし | - |

### リトライ仕様

リトライ機能は実装されていない。

## トランザクション仕様

本機能ではトランザクション処理は発生しない。

## パフォーマンス要件

- CustomHeadersリスト内の線形検索のため、ヘッダー数に応じた処理時間
- 通常のヘッダー数であれば処理は即座に完了する

## セキュリティ考慮事項

- 必要なセキュリティヘッダーを誤って削除しないよう注意が必要
- 削除対象のヘッダー名の検証をアプリケーション層で実装することを推奨

## 備考

- 存在しないヘッダーを削除しようとしてもエラーは発生しない（安全な操作）
- 標準ヘッダー（Content-Type等）の削除は別途考慮が必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

レスポンスクラスの基本構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Response.pas | `src/Horse.Response.pas` | THorseResponseクラスの定義（24-50行目）でFWebResponseフィールドを確認 |

**読解のコツ**: FWebResponseのCustomHeadersプロパティがヘッダーリストを保持している。IndexOfNameとDeleteメソッドはTStrings系クラスの標準メソッドである。

#### Step 2: エントリーポイントを理解する

RemoveHeader機能の宣言を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Response.pas | `src/Horse.Response.pas` | RemoveHeaderメソッドの宣言（43行目） |

**主要処理フロー**:
1. **43行目**: `function RemoveHeader(const AName: string): THorseResponse; virtual;`

#### Step 3: RemoveHeader実装の詳細を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Response.pas | `src/Horse.Response.pas` | RemoveHeaderメソッドの実装部分（131-139行目） |

**主要処理フロー**:
- **131-139行目**: RemoveHeader実装

```pascal
function THorseResponse.RemoveHeader(const AName: string): THorseResponse;
var
  I: Integer;
begin
  I := FWebResponse.CustomHeaders.IndexOfName(AName);
  if I <> -1 then
    FWebResponse.CustomHeaders.Delete(I);
  Result := Self;
end;
```

**実装の詳細説明**:
- **132行目**: ローカル変数Iでインデックスを保持
- **135行目**: `IndexOfName`でヘッダー名からインデックスを検索（見つからなければ-1を返す）
- **136-137行目**: インデックスが有効な場合のみ`Delete`でヘッダーを削除
- **138行目**: メソッドチェーン用に自身を返却

### プログラム呼び出し階層図

```
THorseResponse.RemoveHeader(Name)
    │
    ├─ FWebResponse.CustomHeaders.IndexOfName(Name)
    │      └─ ヘッダー名からインデックスを検索
    │              └─ 見つからない場合は -1 を返す
    │
    └─ FWebResponse.CustomHeaders.Delete(Index)
           └─ インデックス位置のヘッダーを削除
                  └─ I = -1 の場合は呼び出されない
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ヘッダー名 (AName) ───▶ IndexOfName検索
                              │
                              ▼
                        インデックス取得
                              │
                        ┌─────┴─────┐
                        │           │
                    I != -1      I == -1
                        │           │
                        ▼           │
                  Delete(I)         │
                        │           │
                        └─────┬─────┘
                              │
                              ▼
                        Self返却 ───▶ メソッドチェーン継続
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | RemoveHeaderメソッドの実装 |
| Web.HTTPApp | Delphi標準ライブラリ | 外部ライブラリ | TWebResponse.CustomHeaders（Delphi環境） |
| HTTPDefs | FPC標準ライブラリ | 外部ライブラリ | TResponse.CustomHeaders（FPC環境） |
