# 機能設計書 27-サーバー停止

## 概要

本ドキュメントは、HorseフレームワークにおけるHTTPサーバー停止機能の設計を記述する。THorseProvider.StopListenメソッドを使用して、実行中のHTTPサーバーを停止する機能について詳述する。

### 本機能の処理概要

**業務上の目的・背景**：Webアプリケーションの運用において、サーバーを安全に停止する機能は不可欠である。メンテナンス時のサーバー停止、アプリケーション終了時のリソース解放、設定変更のための再起動などで必要となる。本機能により、開発者は実行中のHTTPサーバーを適切に停止できる。

**機能の利用シーン**：GUIアプリケーションでのストップボタン押下時、Windowsサービス停止時、Linuxデーモン停止時、アプリケーション終了時のクリーンアップ処理などで使用される。

**主要な処理内容**：
1. サーバーの実行状態を確認する
2. TIdHTTPWebBrokerBridgeのリスニングを停止する
3. サーバーを非アクティブ化する
4. 実行状態フラグを更新する
5. OnStopListenコールバックを実行する

**関連システム・外部連携**：Indy（TIdHTTPWebBrokerBridge）のStopListening/Active設定を使用。Console環境ではイベントをシグナルしてブロッキングループを解除する。

**権限による制御**：本機能自体に権限制御機能は含まれない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | VCLメインフォーム | 主機能 | Stopボタン押下時にTHorse.StopListenを呼び出してHTTPサーバーを停止 |
| 3 | Windowsサービスメイン | 主機能 | ServiceStop時にTHorse.StopListenを呼び出してHTTPサーバーを停止 |
| 4 | LCLメインフォーム | 主機能 | Stopボタン押下時にTHorse.StopListenを呼び出してHTTPサーバーを停止 |
| 5 | デーモンメイン | 主機能 | DataModuleShutDown時にTHorse.StopListenを呼び出してHTTPサーバーを停止 |

## 機能種別

サーバー制御機能 / 停止処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| - | - | - | 引数なし | - |

### 入力データソース

なし（内部状態のみを参照）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| FRunning | Boolean | サーバー実行状態フラグ（False: 停止） |

### 出力先

内部状態の更新、OnStopListenコールバックの実行

## 処理フロー

### 処理シーケンス

```
1. サーバー実行状態の確認
   └─ HTTPWebBrokerがnilでないことを確認
2. リスニング停止
   └─ StopListening呼び出し（Console版）またはActive := False（VCL版）
3. サーバー非アクティブ化
   └─ Active := False
4. 状態更新
   └─ FRunning := False
5. コールバック実行
   └─ DoOnStopListen
6. イベントシグナル（Console版）
   └─ FEvent.SetEventでブロッキングループ解除
```

### フローチャート

```mermaid
flowchart TD
    A[StopListen呼び出し] --> B{HTTPWebBroker != nil?}
    B -->|No| C[例外発生: Horse not listen]
    B -->|Yes| D[StopListening呼び出し]
    D --> E[Active := False]
    E --> F[FRunning := False]
    F --> G[DoOnStopListen]
    G --> H{FEvent != nil?}
    H -->|Yes| I[FEvent.SetEvent]
    H -->|No| J[終了]
    I --> J
    C --> K[エラー終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-27-01 | 起動確認 | サーバーが起動していない場合は例外を発生 | HTTPWebBroker = nil の場合 |
| BR-27-02 | 順序保証 | StopListening → Active := False → FRunning := False の順序で実行 | 常時 |
| BR-27-03 | イベント解除 | Console版ではFEvent.SetEventでブロッキングを解除 | Console環境 |
| BR-27-04 | コールバック実行 | DoOnStopListenは状態更新後に実行 | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能ではデータベース操作は発生しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | サーバーが起動していない状態でStopListen呼び出し | IsRunningで事前確認する |

### リトライ仕様

リトライ機能は実装されていない。

## トランザクション仕様

本機能ではトランザクション処理は発生しない。

## パフォーマンス要件

- サーバー停止は通常即座に完了する
- 接続中のリクエストの処理完了を待機する可能性がある

## セキュリティ考慮事項

- サーバー停止操作は適切な権限を持つ者のみが実行すべき
- 不正な停止操作はサービス拒否攻撃につながる可能性

## 備考

- Console版とVCL版で処理順序に若干の違いがある
- 二重停止を防ぐためにIsRunningで事前確認することを推奨

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

プロバイダー抽象クラスのStopListen定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Provider.Abstract.pas | `src/Horse.Provider.Abstract.pas` | THorseProviderAbstractクラスのStopListen宣言（34行目） |
| 1-2 | Horse.Provider.Abstract.pas | `src/Horse.Provider.Abstract.pas` | StopListenのデフォルト実装（71-74行目） |

**読解のコツ**: 抽象クラスではStopListenが例外を発生させるデフォルト実装となっており、具体的なプロバイダーでオーバーライドされる。

#### Step 2: Consoleプロバイダーの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | StopListenメソッドの宣言（56行目） |
| 2-2 | Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | InternalStopListenメソッドの実装（240-253行目） |

**主要処理フロー**:
- **240-253行目**: InternalStopListen実装

```pascal
class procedure THorseProvider.InternalStopListen;
begin
  if not HTTPWebBrokerIsNil then
  begin
    GetDefaultHTTPWebBroker.StopListening;
    GetDefaultHTTPWebBroker.Active := False;
    DoOnStopListen;
    FRunning := False;
    if FEvent <> nil then
      GetDefaultEvent.SetEvent;
  end
  else
    raise Exception.Create('Horse not listen');
end;
```

#### Step 3: VCLプロバイダーの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Provider.VCL.pas | `src/Horse.Provider.VCL.pas` | InternalStopListen実装（215-226行目） |

**VCL版の特徴**:
- Active := Falseを先に実行
- FRunning := Falseの後にDoOnStopListen
- StopListeningは最後に実行

```pascal
class procedure THorseProvider.InternalStopListen;
begin
  if not HTTPWebBrokerIsNil then
  begin
    GetDefaultHTTPWebBroker.Active := False;
    FRunning := False;
    DoOnStopListen;
    GetDefaultHTTPWebBroker.StopListening;
  end
  else
    raise Exception.Create('Horse not listen');
end;
```

#### Step 4: サンプル実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Main.Form.pas | `samples/delphi/vcl/src/Main.Form.pas` | VCLサンプルでのStopListen呼び出し（61-64行目、33-37行目） |

```pascal
procedure TFrmVCL.Stop;
begin
  THorse.StopListen;
end;

procedure TFrmVCL.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  if THorse.IsRunning then
    Stop;
end;
```

### プログラム呼び出し階層図

```
THorse.StopListen
    │
    └─ THorseProvider.StopListen（オーバーライド）
           │
           └─ InternalStopListen
                  │
                  ├─ HTTPWebBrokerIsNil チェック
                  │      └─ 未起動なら例外発生
                  │
                  ├─ GetDefaultHTTPWebBroker.StopListening
                  │
                  ├─ GetDefaultHTTPWebBroker.Active := False
                  │
                  ├─ DoOnStopListen
                  │      └─ FOnStopListen コールバック実行
                  │
                  ├─ FRunning := False
                  │
                  └─ GetDefaultEvent.SetEvent（Console版）
                         └─ Listenのブロッキングループ解除
```

### データフロー図

```
[入力]                    [処理]                         [出力]

StopListen呼び出し ───▶ HTTPWebBroker確認
                              │
                              ▼
                    ┌─────────┴─────────┐
                    │                   │
              nil以外               nil
                    │                   │
                    ▼                   ▼
            StopListening          例外発生
                    │
                    ▼
            Active := False
                    │
                    ▼
            FRunning := False ───▶ IsRunning = False
                    │
                    ▼
            DoOnStopListen ───▶ コールバック実行
                    │
                    ▼
            SetEvent（Console）───▶ ループ解除
```

### Console版とVCL版の違い

| 処理順序 | Console版 | VCL版 |
|---------|-----------|-------|
| 1 | StopListening | Active := False |
| 2 | Active := False | FRunning := False |
| 3 | DoOnStopListen | DoOnStopListen |
| 4 | FRunning := False | StopListening |
| 5 | SetEvent | - |

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Provider.Abstract.pas | `src/Horse.Provider.Abstract.pas` | ソース | プロバイダー抽象基底クラス |
| Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | ソース | ConsoleプロバイダーのStopListen実装 |
| Horse.Provider.VCL.pas | `src/Horse.Provider.VCL.pas` | ソース | VCLプロバイダーのStopListen実装 |
| Main.Form.pas | `samples/delphi/vcl/src/Main.Form.pas` | サンプル | VCLサンプル実装 |
| Views.Main.pas | `samples/lazarus/lcl/src/views.main.pas` | サンプル | LCLサンプル実装 |
