# 機能設計書 29-ホスト設定

## 概要

本ドキュメントは、HorseフレームワークにおけるHTTPサーバーのリスニングホスト設定機能の設計を記述する。THorseProvider.Hostプロパティを使用して、サーバーがバインドするネットワークインターフェース（IPアドレス）を設定する機能について詳述する。

### 本機能の処理概要

**業務上の目的・背景**：HTTPサーバーを起動する際、どのネットワークインターフェースでリクエストを受け付けるかは重要なセキュリティ設定である。全インターフェース（0.0.0.0）でリスニングするか、特定のIPアドレスのみでリスニングするかを制御できる。本機能により、開発者はサーバーのバインドアドレスを柔軟に設定できる。

**機能の利用シーン**：ローカルホストのみでのリスニング（開発環境）、特定のネットワークインターフェースのみでのリスニング（マルチホーミング環境）、全インターフェースでのリスニング（本番環境）などで使用される。

**主要な処理内容**：
1. ホストアドレス（string値）を受け取る
2. 入力値をトリムしてクラス変数FHostに格納する
3. 次回のListen呼び出し時に設定されたホストが使用される

**関連システム・外部連携**：設定されたホストはListen時にTIdHTTPWebBrokerBridgeのBindingsに反映される。

**権限による制御**：特定のネットワークインターフェースへのバインドには適切なネットワーク権限が必要な場合がある。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面との直接関連なし（プログラムからの設定が主） |

## 機能種別

サーバー設定機能 / プロパティ操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AValue | string | Yes | ホストアドレス（IPアドレスまたはホスト名） | トリム処理のみ |

### 入力データソース

アプリケーションコードからの直接指定、設定ファイル、環境変数など

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| FHost | string | 設定されたホストアドレス |

### 出力先

クラス変数FHostへの格納

## 処理フロー

### 処理シーケンス

```
1. Hostプロパティへの値設定
   └─ SetHostメソッドが呼び出される
2. 入力値のトリム
   └─ AValue.Trimで空白を除去
3. FHost変数への代入
   └─ クラス変数FHostに値を格納
4. Listen時にホストが使用される
   └─ InternalListenでFHost値を参照
```

### フローチャート

```mermaid
flowchart TD
    A[Host := Value] --> B[SetHost呼び出し]
    B --> C[FHost := AValue.Trim]
    C --> D[設定完了]
    D --> E[Listen呼び出し時]
    E --> F{FHost.IsEmpty?}
    F -->|Yes| G[DEFAULT_HOST使用]
    F -->|No| H{FHost != DEFAULT_HOST?}
    H -->|Yes| I[Bindingsに設定]
    H -->|No| J[DefaultPort設定のみ]
    G --> J
    I --> K[サーバー起動]
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-29-01 | デフォルトホスト | ホストが空の場合はDEFAULT_HOST(0.0.0.0)を使用 | Listen時 |
| BR-29-02 | トリム処理 | 入力値は自動的にトリムされる | SetHost時 |
| BR-29-03 | Bindings設定 | ホストがデフォルト以外の場合のみBindingsを設定 | Listen時 |
| BR-29-04 | 静的プロパティ | Hostはクラスプロパティとして実装されている | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能ではデータベース操作は発生しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | 指定IPアドレスへのバインド失敗 | 有効なIPアドレスを指定 |
| - | Exception | 存在しないネットワークインターフェース指定 | システムのIPアドレスを確認 |

### リトライ仕様

リトライ機能は実装されていない。

## トランザクション仕様

本機能ではトランザクション処理は発生しない。

## パフォーマンス要件

- プロパティ設定のみのため、処理は即座に完了する

## セキュリティ考慮事項

- 0.0.0.0は全インターフェースでリスニングするため、意図しない公開に注意
- 開発環境では127.0.0.1を使用してローカルアクセスのみに制限することを推奨
- 本番環境では適切なファイアウォール設定と組み合わせて使用

## 備考

- Hostプロパティはクラスプロパティのため、インスタンス化不要で設定可能
- Listenメソッドの引数でもホスト指定可能（その場合はプロパティ設定は不要）
- IPv6アドレスも指定可能（環境による）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

定数とクラス変数の定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Constants.pas | `src/Horse.Constants.pas` | DEFAULT_HOST定数の定義（10行目） |
| 1-2 | Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | FHostクラス変数の定義（18行目） |

**読解のコツ**: DEFAULT_HOST = '0.0.0.0'が定義されており、全インターフェースでリスニングすることを意味する。

#### Step 2: プロパティ宣言を理解する

Hostプロパティの宣言を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | Hostプロパティの宣言（50行目） |

**主要処理フロー**:
1. **50行目**: `class property Host: string read GetHost write SetHost;`

#### Step 3: Getter/Setter実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | GetDefaultHost/GetHost/SetHost実装 |

**主要処理フロー**:
- **113-116行目**: GetDefaultHost実装

```pascal
class function THorseProvider.GetDefaultHost: string;
begin
  Result := DEFAULT_HOST;
end;
```

- **123-126行目**: GetHost実装

```pascal
class function THorseProvider.GetHost: string;
begin
  Result := FHost;
end;
```

- **294-297行目**: SetHost実装

```pascal
class procedure THorseProvider.SetHost(const AValue: string);
begin
  FHost := AValue.Trim;
end;
```

#### Step 4: Listen時のホスト使用を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | InternalListenでのホスト参照（183-184行目、202-208行目） |

```pascal
if FHost.IsEmpty then
  FHost := GetDefaultHost;

// ...

LIdHTTPWebBrokerBridge.Bindings.Clear;
if FHost <> GetDefaultHost then
begin
  LIdHTTPWebBrokerBridge.Bindings.Add;
  LIdHTTPWebBrokerBridge.Bindings.Items[0].IP := FHost;
  LIdHTTPWebBrokerBridge.Bindings.Items[0].Port := FPort;
end;
```

### プログラム呼び出し階層図

```
THorse.Host := '127.0.0.1'
    │
    └─ THorseProvider.SetHost('127.0.0.1')
           │
           └─ FHost := '127.0.0.1'.Trim

THorse.Host（取得）
    │
    └─ THorseProvider.GetHost
           │
           └─ Result := FHost

THorse.Listen
    │
    └─ InternalListen
           │
           ├─ FHost.IsEmpty チェック
           │      └─ GetDefaultHost（'0.0.0.0'）
           │
           ├─ FHost <> DEFAULT_HOST チェック
           │      │
           │      └─ True の場合
           │             ├─ Bindings.Add
           │             ├─ Bindings[0].IP := FHost
           │             └─ Bindings[0].Port := FPort
           │
           └─ サーバー起動
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ホストアドレス ───────▶ SetHost
                         │
                         ▼
                   AValue.Trim ───▶ トリム処理
                         │
                         ▼
                   FHost := Value ───▶ クラス変数に格納
                                               │
                                               ▼
                                         Listen呼び出し
                                               │
                                               ▼
                                    ┌──────────┴──────────┐
                                    │                     │
                              FHost = DEFAULT       FHost != DEFAULT
                                    │                     │
                                    ▼                     ▼
                              DefaultPort設定       Bindings設定
                                    │                     │
                                    └──────────┬──────────┘
                                               │
                                               ▼
                                         サーバー起動
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Constants.pas | `src/Horse.Constants.pas` | ソース | DEFAULT_HOST定数定義 |
| Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | ソース | ConsoleプロバイダーのHost実装 |
| Horse.Provider.VCL.pas | `src/Horse.Provider.VCL.pas` | ソース | VCLプロバイダーのHost実装 |
