# 機能設計書 30-最大接続数設定

## 概要

本ドキュメントは、HorseフレームワークにおけるHTTPサーバーの最大同時接続数設定機能の設計を記述する。THorseProvider.MaxConnectionsプロパティを使用して、サーバーが同時に受け付ける接続の最大数を制限する機能について詳述する。

### 本機能の処理概要

**業務上の目的・背景**：HTTPサーバーの安定運用において、同時接続数の制限は重要なリソース管理手段である。無制限の接続を許可するとサーバーリソース（メモリ、CPU、ファイルディスクリプタ等）が枯渇する可能性がある。本機能により、開発者はサーバーの負荷を制御し、安定したサービス提供を実現できる。

**機能の利用シーン**：本番環境でのサーバー安定化、負荷制限の実装、リソース保護、DoS攻撃への対策（一部）、開発環境でのリソース制限などで使用される。

**主要な処理内容**：
1. 最大接続数（Integer値）を受け取る
2. クラス変数FMaxConnectionsに値を格納する
3. 次回のListen呼び出し時にWebRequestHandlerとHTTPWebBrokerに設定が適用される

**関連システム・外部連携**：設定された値はListen時にWebRequestHandler.MaxConnectionsおよびTIdHTTPWebBrokerBridge.MaxConnectionsに反映される。

**権限による制御**：本機能自体に権限制御機能は含まれない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面との直接関連なし（プログラムからの設定が主） |

## 機能種別

サーバー設定機能 / プロパティ操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AValue | Integer | Yes | 最大同時接続数 | 正の整数（0は無制限または未設定） |

### 入力データソース

アプリケーションコードからの直接指定、設定ファイル、環境変数など

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| FMaxConnections | Integer | 設定された最大接続数 |

### 出力先

クラス変数FMaxConnectionsへの格納

## 処理フロー

### 処理シーケンス

```
1. MaxConnectionsプロパティへの値設定
   └─ SetMaxConnectionsメソッドが呼び出される
2. FMaxConnections変数への代入
   └─ クラス変数FMaxConnectionsに値を格納
3. Listen時に設定が適用される
   └─ InternalListenでFMaxConnections値を参照
4. WebRequestHandlerとHTTPWebBrokerに設定
   └─ 両方のMaxConnectionsプロパティに値を設定
```

### フローチャート

```mermaid
flowchart TD
    A[MaxConnections := Value] --> B[SetMaxConnections呼び出し]
    B --> C[FMaxConnections := AValue]
    C --> D[設定完了]
    D --> E[Listen呼び出し時]
    E --> F{FMaxConnections > 0?}
    F -->|Yes| G[WebRequestHandler.MaxConnections設定]
    F -->|No| H[設定スキップ]
    G --> I[HTTPWebBroker.MaxConnections設定]
    I --> J[サーバー起動]
    H --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-30-01 | 条件付き適用 | MaxConnectionsが0より大きい場合のみ設定を適用 | Listen時 |
| BR-30-02 | 二重設定 | WebRequestHandlerとHTTPWebBrokerの両方に設定 | MaxConnections > 0 |
| BR-30-03 | 静的プロパティ | MaxConnectionsはクラスプロパティとして実装されている | 常時 |
| BR-30-04 | デフォルト値なし | MaxConnectionsにはデフォルト値が設定されていない（0） | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能ではデータベース操作は発生しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 本機能では明示的なエラー処理なし | - |

### リトライ仕様

リトライ機能は実装されていない。

## トランザクション仕様

本機能ではトランザクション処理は発生しない。

## パフォーマンス要件

- プロパティ設定のみのため、処理は即座に完了する
- 設定値はサーバーの同時接続処理能力に直接影響する

## セキュリティ考慮事項

- 適切なMaxConnections設定によりDoS攻撃の影響を軽減可能
- 値が小さすぎると正当なリクエストも拒否される可能性
- 値が大きすぎるとサーバーリソースが枯渇する可能性
- 本番環境ではサーバースペックに応じた適切な値を設定することを推奨

## 備考

- MaxConnectionsプロパティはクラスプロパティのため、インスタンス化不要で設定可能
- 0または負の値の場合は設定がスキップされ、無制限（またはサーバーデフォルト）となる
- ListenQueueプロパティと組み合わせてより詳細な接続制御が可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

クラス変数の定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | FMaxConnectionsクラス変数の定義（21行目） |

**読解のコツ**: FMaxConnectionsはクラス変数として宣言されており、デフォルト値は0（未設定/無制限）となる。

#### Step 2: プロパティ宣言を理解する

MaxConnectionsプロパティの宣言を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | MaxConnectionsプロパティの宣言（52行目） |

**主要処理フロー**:
1. **52行目**: `class property MaxConnections: Integer read GetMaxConnections write SetMaxConnections;`

#### Step 3: Getter/Setter実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | GetMaxConnections/SetMaxConnections実装 |

**主要処理フロー**:
- **148-151行目**: GetMaxConnections実装

```pascal
class function THorseProvider.GetMaxConnections: Integer;
begin
  Result := FMaxConnections;
end;
```

- **314-317行目**: SetMaxConnections実装

```pascal
class procedure THorseProvider.SetMaxConnections(const AValue: Integer);
begin
  FMaxConnections := AValue;
end;
```

#### Step 4: Listen時のMaxConnections適用を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | InternalListenでのMaxConnections適用（189-193行目） |

```pascal
if FMaxConnections > 0 then
begin
  WebRequestHandler.MaxConnections := FMaxConnections;
  GetDefaultHTTPWebBroker.MaxConnections := FMaxConnections;
end;
```

### プログラム呼び出し階層図

```
THorse.MaxConnections := 100
    │
    └─ THorseProvider.SetMaxConnections(100)
           │
           └─ FMaxConnections := 100

THorse.MaxConnections（取得）
    │
    └─ THorseProvider.GetMaxConnections
           │
           └─ Result := FMaxConnections

THorse.Listen
    │
    └─ InternalListen
           │
           ├─ FMaxConnections > 0 チェック
           │      │
           │      └─ True の場合
           │             ├─ WebRequestHandler.MaxConnections := FMaxConnections
           │             └─ HTTPWebBroker.MaxConnections := FMaxConnections
           │
           └─ サーバー起動
```

### データフロー図

```
[入力]                    [処理]                         [出力]

最大接続数 ───────▶ SetMaxConnections
                         │
                         ▼
                   FMaxConnections := AValue ───▶ クラス変数に格納
                                                        │
                                                        ▼
                                                  Listen呼び出し
                                                        │
                                                        ▼
                                              FMaxConnections > 0?
                                                        │
                                              ┌─────────┴─────────┐
                                              │                   │
                                            True               False
                                              │                   │
                                              ▼                   │
                                    WebRequestHandler設定         │
                                              │                   │
                                              ▼                   │
                                    HTTPWebBroker設定             │
                                              │                   │
                                              └─────────┬─────────┘
                                                        │
                                                        ▼
                                                  サーバー起動
```

### WebRequestHandlerとHTTPWebBrokerの役割

| コンポーネント | 役割 | MaxConnectionsの効果 |
|---------------|------|---------------------|
| WebRequestHandler | Webリクエストの処理を管理 | 処理キューの制限 |
| HTTPWebBroker | HTTP接続の管理 | 同時接続数の制限 |

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Provider.Console.pas | `src/Horse.Provider.Console.pas` | ソース | ConsoleプロバイダーのMaxConnections実装 |
| Horse.Provider.VCL.pas | `src/Horse.Provider.VCL.pas` | ソース | VCLプロバイダーのMaxConnections実装 |
| Web.WebReq | Delphi標準ライブラリ | 外部ライブラリ | WebRequestHandler定義 |
| IdHTTPWebBrokerBridge | Indyライブラリ | 外部ライブラリ | TIdHTTPWebBrokerBridge定義 |
