# 機能設計書 35-パスパラメータ対応

## 概要

本ドキュメントは、Horseフレームワークにおけるパスパラメータ対応機能の詳細設計を記載したものである。THorseRouterTree クラスにおいて、`:id` 形式のパスパラメータをサポートし、動的なURLパスからパラメータ値を抽出してリクエストオブジェクトに格納する機能について説明する。

### 本機能の処理概要

**業務上の目的・背景**：REST APIの設計において、リソースの識別子をURLパスに含めることが一般的である（例：/users/123, /products/abc）。本機能は、ルート定義時に `:パラメータ名` 形式でパスパラメータを指定し、実際のリクエスト時にそのパラメータ値を抽出してコールバック内で利用できるようにする。これにより、動的なリソース識別が可能となる。

**機能の利用シーン**：本機能は、以下のようなシーンで利用される。
- リソースIDによる個別取得（/users/:id → /users/123）
- 階層的なリソースアクセス（/users/:userId/posts/:postId）
- カテゴリやスラッグによるコンテンツ取得（/articles/:slug）
- 複数パラメータの組み合わせ（/api/:version/resources/:id）

**主要な処理内容**：
1. ルート登録時にパスパラメータの検出（`:` で始まるセグメント）
2. パラメータ名（タグ）の抽出と保存（FTag, FIsParamsKey）
3. リクエスト実行時のパスマッチングとパラメータ値の抽出
4. 抽出したパラメータ値のリクエストオブジェクトへの格納

**関連システム・外部連携**：パスパラメータは THorseRouterTree のルーティングツリーで処理され、抽出された値は THorseRequest.Params でアクセス可能となる。

**権限による制御**：パスパラメータ機能自体に権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面に直接関連しない（API設計時に使用） |

## 機能種別

ルーティング / パラメータ抽出

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| APath | string | Yes | パスパラメータを含むルートパス（例：/users/:id） | ':' で始まるセグメントがパラメータとして認識 |
| HTTPリクエストパス | string | Yes | 実際のリクエストURL（例：/users/123） | なし |

### 入力データソース

- ルート定義時のパス文字列
- HTTPリクエストのURL

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| FTag | string | パラメータ名（':'を除いた部分） |
| FIsParamsKey | Boolean | パスパラメータかどうかのフラグ |
| Req.Params | THorseCoreParam | 抽出されたパラメータ値（キー・バリュー） |

### 出力先

THorseRequest.Params プロパティを通じてコールバック内でアクセス可能

## 処理フロー

### 処理シーケンス

```
1. ルート登録時（RegisterInternal）
   └─ パスの各セグメントを処理
      └─ FPart.StartsWith(':') でパスパラメータを検出
         └─ FIsParamsKey := True を設定
         └─ FTag := FPart.Substring(1, Length(FPart) - 1) でパラメータ名を抽出
2. リクエスト実行時（ExecuteInternal）
   └─ パスのマッチング処理
      └─ FIsParamsKeyがTrueの場合、任意の値にマッチ
         └─ リクエストの対応するセグメントをパラメータ値として抽出
3. TNextCallerでの処理
   └─ SetIsParamsKey(FIsParamsKey) でフラグを渡す
   └─ SetTag(FTag) でパラメータ名を渡す
   └─ パラメータ値をリクエストオブジェクトに格納
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[ルート登録]
    B --> C[パスをセグメントに分割]
    C --> D{セグメントが : で始まる?}
    D -->|Yes| E[FIsParamsKey := True]
    E --> F[FTag := パラメータ名抽出]
    F --> G[FRegexedKeysに追加]
    D -->|No| H[通常セグメントとして登録]
    G --> I[次のセグメント]
    H --> I
    I --> J{最後のセグメント?}
    J -->|No| C
    J -->|Yes| K[登録完了]

    subgraph リクエスト実行時
    L[HTTPリクエスト受信] --> M[パスマッチング]
    M --> N{FIsParamsKey?}
    N -->|Yes| O[任意の値にマッチ]
    O --> P[パラメータ値を抽出]
    P --> Q[Req.Paramsに格納]
    N -->|No| R[完全一致チェック]
    R --> S{マッチ?}
    S -->|Yes| Q
    S -->|No| T[次のルートを試行]
    end
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-35-01 | パラメータ識別 | ':' で始まるパスセグメントはパスパラメータとして認識される | ルート登録時 |
| BR-35-02 | パラメータ名抽出 | ':' を除いた残りの文字列がパラメータ名（タグ）となる | パラメータ検出時 |
| BR-35-03 | 任意値マッチ | パスパラメータ位置は任意の非空文字列にマッチする | リクエストマッチング時 |
| BR-35-04 | 優先順位 | 完全一致ルートはパラメータルートより優先される | 複数ルートが該当する場合 |
| BR-35-05 | FRegexedKeys登録 | パスパラメータを含むルートはFRegexedKeysリストに追加される | ルート登録時 |

### 計算ロジック

**パスパラメータ検出（RegisterInternal）**:
```pascal
FIsParamsKey := FPart.StartsWith(':');
FTag := FPart.Substring(1, Length(FPart) - 1);
```

**例**:
- パス `/users/:id/posts/:postId` の場合
  - セグメント 'users' → FIsParamsKey=False
  - セグメント ':id' → FIsParamsKey=True, FTag='id'
  - セグメント 'posts' → FIsParamsKey=False
  - セグメント ':postId' → FIsParamsKey=True, FTag='postId'

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

本機能ではデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 404 Not Found | パスパラメータを含むルートにマッチしない場合 | ルート定義を確認 |

### リトライ仕様

リトライは不要。

## トランザクション仕様

トランザクション処理は行わない。

## パフォーマンス要件

- パスパラメータルートはFRegexedKeysリストに追加され、完全一致しない場合に順次チェックされる
- 多数のパスパラメータルートがある場合、マッチング処理に若干のオーバーヘッドが発生する可能性がある

## セキュリティ考慮事項

- パスパラメータの値はユーザー入力であるため、適切なバリデーションが必要
- SQLインジェクションやパストラバーサルを防ぐため、パラメータ値の検証を行うこと
- 数値IDが期待される場合は型変換時のエラーハンドリングを実装すること

## 備考

- パスパラメータ名は大文字・小文字が区別される
- 同一パス位置に複数のパラメータ名を定義することはできない
- 正規表現ルーティング（機能No.36）と組み合わせることで、より複雑なパターンマッチングが可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

パスパラメータ関連のフィールド定義から始める。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | THorseRouterTreeクラスのパスパラメータ関連フィールドを確認 |

**読解のコツ**:
- FIsParamsKey: パスパラメータかどうかを示すブールフラグ
- FTag: パラメータ名（':'を除いた部分）
- FRegexedKeys: パスパラメータを含むルートキーのリスト

**主要処理フロー**:
- **34行目**: `FPart: string;` - パスの各セグメント
- **35行目**: `FTag: string;` - パラメータ名
- **36行目**: `FIsParamsKey: Boolean;` - パスパラメータフラグ
- **40行目**: `FRegexedKeys: TList<string>;` - パラメータルートのキーリスト

#### Step 2: ルート登録処理を理解する

パスパラメータの検出と登録ロジックを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | RegisterInternalメソッドを確認 |

**主要処理フロー**:
- **274-315行目**: RegisterInternal メソッド全体
- **280-291行目**: 初期化処理（初回のみ実行）
  - **282行目**: `FPart := APath.Dequeue;` - パスセグメントを取得
  - **284行目**: `FIsParamsKey := FPart.StartsWith(':');` - パスパラメータ検出
  - **285行目**: `FTag := FPart.Substring(1, Length(FPart) - 1);` - パラメータ名抽出
- **312-313行目**: FRegexedKeysへの追加
  ```pascal
  if LForceRouter.FIsParamsKey or LForceRouter.FIsRouterRegex then
    FRegexedKeys.Add(LNextPart);
  ```

**読解のコツ**:
- `FPart.StartsWith(':')` でコロンで始まるかを判定
- `Substring(1, Length - 1)` でコロンを除いた部分を抽出
- FIsInitialized フラグで初回のみパラメータ検出を行う

#### Step 3: パスマッチング処理を理解する

リクエスト実行時のパスマッチングロジックを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | CallNextPathメソッドとHasNextメソッドを確認 |

**主要処理フロー**:
- **83-117行目**: CallNextPath メソッド
  - **94行目**: `FRoute.TryGetValue(LCurrent, LAcceptable);` - 完全一致チェック
  - **102-113行目**: 完全一致しない場合、FRegexedKeysを順次チェック
    ```pascal
    for LKey in FRegexedKeys do
    begin
      FRoute.TryGetValue(LKey, LAcceptable);
      if LAcceptable.HasNext(AHTTPType, APath.ToArray) then
      begin
        LFound := LAcceptable.ExecuteInternal(...);
        Break;
      end;
    end;
    ```
- **240-272行目**: HasNext メソッド
  - **248行目**: `(APaths[AIndex] = FPart) or (FIsParamsKey)` - パスパラメータは任意値にマッチ

#### Step 4: TNextCallerでのパラメータ処理を理解する

パラメータ値のリクエストオブジェクトへの格納を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ExecuteInternalメソッドを確認 |

**主要処理フロー**:
- **168-194行目**: ExecuteInternal メソッド
  - **184行目**: `LNextCaller.SetTag(FTag);` - パラメータ名を設定
  - **185行目**: `LNextCaller.SetIsParamsKey(FIsParamsKey);` - パラメータフラグを設定

### プログラム呼び出し階層図

```
THorseRouterTree.RegisterRoute(AHTTPType, APath, ACallback)
    │
    └─ RegisterInternal(AHTTPType, APath, ACallback)
           │
           ├─ FPart := APath.Dequeue （セグメント取得）
           │
           ├─ FPart.StartsWith(':') でパスパラメータ検出
           │      │
           │      ├─ True: FIsParamsKey := True
           │      │         FTag := パラメータ名抽出
           │      │
           │      └─ False: 通常セグメント
           │
           └─ FRegexedKeys.Add (パラメータルートの場合)

リクエスト実行時:
THorseRouterTree.Execute(ARequest, AResponse)
    │
    └─ ExecuteInternal(APath, AHTTPType, ARequest, AResponse)
           │
           ├─ CallNextPath で次のルートを探索
           │      │
           │      ├─ 完全一致チェック
           │      │
           │      └─ 不一致の場合、FRegexedKeysを探索
           │             │
           │             └─ HasNext で FIsParamsKey を考慮したマッチング
           │
           └─ TNextCaller で処理
                  │
                  ├─ SetTag(FTag)
                  ├─ SetIsParamsKey(FIsParamsKey)
                  └─ パラメータ値を Req.Params に格納
```

### データフロー図

```
[ルート登録時]

パス文字列 ─────────────▶ パスセグメント分割
 例: '/users/:id'              │
                               ▼
                        セグメント毎に処理
                               │
                               ▼
                        ':' で始まる? ──▶ No: 通常セグメント
                               │
                               ▼ Yes
                        FIsParamsKey := True
                        FTag := 'id'
                               │
                               ▼
                        FRegexedKeys に追加


[リクエスト実行時]

リクエストURL ──────────▶ パスマッチング
 例: '/users/123'              │
                               ▼
                        完全一致チェック
                               │
                               ▼ 不一致
                        FRegexedKeys 探索
                               │
                               ▼
                        FIsParamsKey=True で任意値マッチ
                               │
                               ▼
                        パラメータ値 '123' を抽出
                               │
                               ▼
                        Req.Params['id'] = '123'
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ソース | ルーティングツリー実装（パスパラメータ検出・マッチング） |
| Horse.Core.RouterTree.NextCaller.pas | `src/Horse.Core.RouterTree.NextCaller.pas` | ソース | TNextCaller クラス（パラメータ値のリクエストへの格納） |
| Horse.Request.pas | `src/Horse.Request.pas` | ソース | THorseRequest クラス（Paramsプロパティ） |
| Horse.Core.Param.pas | `src/Horse.Core.Param.pas` | ソース | THorseCoreParam クラス（パラメータアクセス） |
