# 機能設計書 36-正規表現ルーティング

## 概要

本ドキュメントは、Horseフレームワークにおける正規表現ルーティング機能の詳細設計を記載したものである。THorseRouterTree クラスにおいて、`(正規表現)` 形式のパスセグメントをサポートし、正規表現によるパスマッチングを実現する機能について説明する。

### 本機能の処理概要

**業務上の目的・背景**：REST APIの設計において、パスパラメータ（:id形式）だけでは対応できない複雑なパターンマッチングが必要な場合がある。例えば、数値のみを許可するID、特定の形式を持つスラッグ、ファイル拡張子を含むパスなどである。本機能は、正規表現を使用した柔軟なパスマッチングを提供し、より詳細なルーティング制御を可能にする。

**機能の利用シーン**：本機能は、以下のようなシーンで利用される。
- 数値IDのみを許可するルート（`/users/(\d+)`）
- 特定形式のスラッグ（`/articles/([a-z0-9-]+)`）
- ファイル名パターン（`/(.*\.json)`）
- 複数形式に対応するルート（`/(v1|v2)/api`）

**主要な処理内容**：
1. ルート登録時に正規表現パターンの検出（`(` で始まり `)` で終わるセグメント）
2. 正規表現パターンの保存（FRouterRegex, FIsRouterRegex）
3. リクエスト実行時の正規表現マッチング
4. FRegexedKeysへの登録による特殊ルートとしての管理

**関連システム・外部連携**：正規表現マッチングには Delphi の System.RegularExpressions または Free Pascal の RegExpr ユニットを使用する。

**権限による制御**：正規表現ルーティング機能自体に権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面に直接関連しない（API設計時に使用） |

## 機能種別

ルーティング / パターンマッチング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| APath | string | Yes | 正規表現を含むルートパス（例：/users/(\d+)） | '(' で始まり ')' で終わるセグメントが正規表現として認識 |
| HTTPリクエストパス | string | Yes | 実際のリクエストURL（例：/users/123） | なし |

### 入力データソース

- ルート定義時のパス文字列（正規表現パターン含む）
- HTTPリクエストのURL

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| FRouterRegex | string | 正規表現パターン文字列 |
| FIsRouterRegex | Boolean | 正規表現ルートかどうかのフラグ |

### 出力先

THorseRouterTree のルーティングツリーに正規表現ルートとして登録

## 処理フロー

### 処理シーケンス

```
1. ルート登録時（RegisterInternal）
   └─ パスの各セグメントを処理
      └─ FPart.StartsWith('(') and FPart.EndsWith(')') で正規表現を検出
         └─ FIsRouterRegex := True を設定
         └─ FRouterRegex := FPart で正規表現パターンを保存
2. リクエスト実行時（HasNext）
   └─ FIsRouterRegex が True の場合
      └─ TRegEx.IsMatch で正規表現マッチングを実行
         └─ パターン: '^' + FRouterRegex + '$'
3. マッチ成功時
   └─ ルートのコールバックを実行
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[ルート登録]
    B --> C[パスをセグメントに分割]
    C --> D{セグメントが括弧で囲まれている?}
    D -->|Yes| E[FIsRouterRegex := True]
    E --> F[FRouterRegex := パターン]
    F --> G[FRegexedKeysに追加]
    D -->|No| H{: で始まる?}
    H -->|Yes| I[パスパラメータ処理]
    H -->|No| J[通常セグメントとして登録]
    G --> K[次のセグメント]
    I --> K
    J --> K
    K --> L{最後のセグメント?}
    L -->|No| C
    L -->|Yes| M[登録完了]

    subgraph リクエスト実行時
    N[HTTPリクエスト受信] --> O[パスマッチング]
    O --> P{FIsRouterRegex?}
    P -->|Yes| Q[TRegEx.IsMatch 実行]
    Q --> R{マッチ?}
    R -->|Yes| S[コールバック実行]
    R -->|No| T[次のルートを試行]
    P -->|No| U[通常のマッチング処理]
    end
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-36-01 | 正規表現識別 | '(' で始まり ')' で終わるパスセグメントは正規表現として認識される | ルート登録時 |
| BR-36-02 | パターン保存 | 正規表現パターンはそのままFRouterRegexに保存される | 正規表現検出時 |
| BR-36-03 | 完全一致マッチ | マッチング時は '^パターン$' 形式で完全一致を判定 | リクエストマッチング時 |
| BR-36-04 | FRegexedKeys登録 | 正規表現ルートはFRegexedKeysリストに追加される | ルート登録時 |
| BR-36-05 | Delphi専用 | 正規表現マッチングはDelphi環境でのみ完全サポート | HasNextメソッド内 |

### 計算ロジック

**正規表現パターン検出（RegisterInternal）**:
```pascal
FIsRouterRegex := FPart.StartsWith('(') and FPart.EndsWith(')');
FRouterRegex := FPart;
```

**正規表現マッチング（HasNext - Delphi版）**:
```pascal
if FIsRouterRegex then
begin
  Result := TRegEx.IsMatch(APaths[AIndex], Format('^%s$', [FRouterRegex]));
  Exit;
end;
```

**例**:
- パス `/users/(\d+)` の場合
  - セグメント 'users' → FIsRouterRegex=False
  - セグメント '(\d+)' → FIsRouterRegex=True, FRouterRegex='(\d+)'
- リクエスト `/users/123` の場合
  - '123' を '^(\d+)$' でマッチング → True

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

本機能ではデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 正規表現エラー | 不正な正規表現パターンを指定した場合 | 正しい正規表現構文を使用 |
| - | 404 Not Found | 正規表現にマッチしない場合 | ルート定義を確認 |

### リトライ仕様

リトライは不要。

## トランザクション仕様

トランザクション処理は行わない。

## パフォーマンス要件

- 正規表現マッチングは文字列比較より計算コストが高い
- 多数の正規表現ルートを使用する場合、パフォーマンスへの影響に注意
- 可能な限り、パスパラメータ（:id形式）で代替できる場合はそちらを使用

## セキュリティ考慮事項

- 正規表現のReDoS（正規表現サービス拒否）攻撃に注意
- 複雑すぎる正規表現パターンは避けること
- 正規表現でマッチした値もユーザー入力として扱い、適切なバリデーションを行うこと

## 備考

- Free Pascal（FPC）環境では、HasNext内の正規表現マッチング処理は `{$IFNDEF FPC}` で囲まれており、直接的なTRegEx使用は行われない
- パスパラメータ（:id形式）と正規表現ルーティングは同じFRegexedKeysで管理される
- 完全一致チェックが先に行われ、一致しない場合にのみ正規表現/パスパラメータルートが探索される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

正規表現ルーティング関連のフィールド定義から始める。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | THorseRouterTreeクラスの正規表現関連フィールドを確認 |

**読解のコツ**:
- FRouterRegex: 正規表現パターン文字列
- FIsRouterRegex: 正規表現ルートかどうかのフラグ
- FRegexedKeys: パスパラメータおよび正規表現ルートのキーリスト

**主要処理フロー**:
- **37行目**: `FRouterRegex: string;` - 正規表現パターン
- **38行目**: `FIsRouterRegex: Boolean;` - 正規表現フラグ
- **40行目**: `FRegexedKeys: TList<string>;` - 特殊ルートのキーリスト

#### Step 2: ルート登録処理を理解する

正規表現パターンの検出と登録ロジックを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | RegisterInternalメソッドを確認 |

**主要処理フロー**:
- **274-315行目**: RegisterInternal メソッド全体
- **287-288行目**: 正規表現パターン検出と保存
  ```pascal
  FIsRouterRegex := FPart.StartsWith('(') and FPart.EndsWith(')');
  FRouterRegex := FPart;
  ```
- **312-313行目**: FRegexedKeysへの追加
  ```pascal
  if LForceRouter.FIsParamsKey or LForceRouter.FIsRouterRegex then
    FRegexedKeys.Add(LNextPart);
  ```

**読解のコツ**:
- `FPart.StartsWith('(') and FPart.EndsWith(')')` で括弧で囲まれたセグメントを検出
- FRouterRegexには括弧を含めたパターン全体が保存される

#### Step 3: マッチング処理を理解する

リクエスト実行時の正規表現マッチングロジックを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | HasNextメソッドを確認 |

**主要処理フロー**:
- **240-272行目**: HasNext メソッド全体
- **251-256行目**: 正規表現マッチング（Delphi専用）
  ```pascal
  {$IFNDEF FPC}
    if FIsRouterRegex then
    begin
      Result := TRegEx.IsMatch(APaths[AIndex], Format('^%s$', [FRouterRegex]));
      Exit;
    end;
  {$ENDIF}
  ```

**読解のコツ**:
- `{$IFNDEF FPC}` でFPC環境では実行されない
- `Format('^%s$', [FRouterRegex])` で完全一致パターンを生成
- TRegEx.IsMatch で正規表現マッチングを実行

#### Step 4: ルート探索の流れを理解する

CallNextPathでのFRegexedKeys探索を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | CallNextPathメソッドを確認 |

**主要処理フロー**:
- **83-117行目**: CallNextPath メソッド
- **102-113行目**: FRegexedKeys探索
  ```pascal
  if (not LFound) and (FRegexedKeys.Count > 0) then
  begin
    for LKey in FRegexedKeys do
    begin
      FRoute.TryGetValue(LKey, LAcceptable);
      if LAcceptable.HasNext(AHTTPType, APath.ToArray) then
      begin
        LFound := LAcceptable.ExecuteInternal(APath, ...);
        Break;
      end;
    end;
  end
  ```

### プログラム呼び出し階層図

```
THorseRouterTree.RegisterRoute(AHTTPType, APath, ACallback)
    │
    └─ RegisterInternal(AHTTPType, APath, ACallback)
           │
           ├─ FPart := APath.Dequeue （セグメント取得）
           │
           ├─ FPart.StartsWith('(') and FPart.EndsWith(')') で正規表現検出
           │      │
           │      ├─ True: FIsRouterRegex := True
           │      │         FRouterRegex := FPart
           │      │
           │      └─ False: パスパラメータまたは通常セグメント
           │
           └─ FRegexedKeys.Add (正規表現ルートの場合)

リクエスト実行時:
THorseRouterTree.Execute(ARequest, AResponse)
    │
    └─ ExecuteInternal → CallNextPath
           │
           ├─ 完全一致チェック（FRoute.TryGetValue）
           │
           └─ 不一致の場合、FRegexedKeys探索
                  │
                  └─ HasNext で FIsRouterRegex を考慮
                         │
                         └─ TRegEx.IsMatch(パス, '^パターン$')
```

### データフロー図

```
[ルート登録時]

パス文字列 ─────────────▶ パスセグメント分割
 例: '/users/(\d+)'            │
                               ▼
                        セグメント毎に処理
                               │
                               ▼
                        '(' で始まり ')' で終わる? ──▶ No: 他の処理へ
                               │
                               ▼ Yes
                        FIsRouterRegex := True
                        FRouterRegex := '(\d+)'
                               │
                               ▼
                        FRegexedKeys に追加


[リクエスト実行時]

リクエストURL ──────────▶ パスマッチング
 例: '/users/123'              │
                               ▼
                        完全一致チェック
                               │
                               ▼ 不一致
                        FRegexedKeys 探索
                               │
                               ▼
                        HasNext: FIsRouterRegex=True
                               │
                               ▼
                        TRegEx.IsMatch('123', '^(\d+)$')
                               │
                               ▼ マッチ成功
                        コールバック実行
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ソース | ルーティングツリー実装（正規表現検出・マッチング） |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | MatchRoute関数（正規表現によるルートマッチング補助） |
