# 機能設計書 37-フィールド型変換

## 概要

本ドキュメントは、Horseフレームワークにおけるフィールド型変換機能の詳細設計を記載したものである。THorseCoreParamField クラスを使用して、HTTPリクエストのパラメータ（クエリ、パス、ボディ等）を各種データ型に変換する機能について説明する。

### 本機能の処理概要

**業務上の目的・背景**：HTTPリクエストで受け取るパラメータはすべて文字列形式である。しかし、アプリケーションロジックでは数値、日付、ブール値などの型で処理する必要がある。本機能は、リクエストパラメータを各種データ型に安全に変換し、型変換エラー時には適切なHTTPエラーレスポンスを返す機能を提供する。

**機能の利用シーン**：本機能は、以下のようなシーンで利用される。
- IDパラメータの整数変換（/users/:id → Integer）
- 金額パラメータの通貨型変換（price → Currency）
- 日付パラメータの日時変換（startDate → TDateTime）
- フラグパラメータのブール変換（isActive → Boolean）
- ファイルパラメータのストリーム取得（file → TStream）
- リスト形式パラメータの配列変換（ids=1,2,3 → TList<T>）

**主要な処理内容**：
1. 文字列から整数への変換（AsInteger, AsInt64）
2. 文字列から浮動小数点への変換（AsFloat, AsExtended, AsCurrency）
3. 文字列から日付・時刻への変換（AsDate, AsTime, AsDateTime, AsISO8601DateTime）
4. 文字列からブール値への変換（AsBoolean）
5. ストリームデータの取得（AsStream）
6. 文字列のまま取得（AsString）
7. カンマ区切り文字列からリストへの変換（AsList, AsList<T>）
8. ファイルへの保存（SaveToFile）

**関連システム・外部連携**：パラメータ変換はリクエストオブジェクト（THorseRequest）の Params、Query、Body 等のメソッドを通じてアクセスされる。

**権限による制御**：型変換機能自体に権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面に直接関連しない（API処理時に使用） |

## 機能種別

パラメータ処理 / データ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AParams | TDictionary<string, string> | Yes | パラメータ辞書 | なし |
| AFieldName | string | Yes | フィールド名 | なし |
| AStream | TStream | No | ストリームデータ（ファイルアップロード時） | なし |

### 入力データソース

- HTTPリクエストのクエリパラメータ
- HTTPリクエストのパスパラメータ
- HTTPリクエストのボディ（フォームデータ、マルチパートデータ）
- アップロードファイルのストリーム

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| AsString | string | 文字列として取得 |
| AsInteger | Integer | 整数として取得 |
| AsInt64 | Int64 | 64ビット整数として取得 |
| AsFloat | Double | 倍精度浮動小数点として取得 |
| AsExtended | Extended | 拡張精度浮動小数点として取得 |
| AsCurrency | Currency | 通貨型として取得 |
| AsBoolean | Boolean | ブール値として取得 |
| AsDate | TDateTime | 日付として取得 |
| AsTime | TTime | 時刻として取得 |
| AsDateTime | TDateTime | 日時として取得 |
| AsISO8601DateTime | TDateTime | ISO8601形式日時として取得 |
| AsStream | TStream | ストリームとして取得 |
| AsList | TList<string> | 文字列リストとして取得 |
| AsList<T> | TList<T> | 型付きリストとして取得（Delphi専用） |

### 出力先

コールバック関数内で使用

## 処理フロー

### 処理シーケンス

```
1. THorseCoreParam.Field(フィールド名)でTHorseCoreParamFieldインスタンス取得
   └─ フィールドが存在するかチェック（FContainsフラグ設定）
2. オプション設定（Required, DateFormat等）
   └─ メソッドチェーンでフルーエントに設定
3. 型変換メソッド呼び出し（AsInteger, AsString等）
   └─ 値が存在しない場合
      └─ Requiredがtrueなら例外発生（EHorseException with BadRequest）
      └─ Requiredがfalseならデフォルト値を返却
   └─ 値が存在する場合
      └─ 変換処理を実行
      └─ 変換失敗時は例外発生（EHorseException with BadRequest）
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[Field メソッド呼び出し]
    B --> C[THorseCoreParamField 生成]
    C --> D{フィールド存在?}
    D -->|Yes| E[FContains := True, FValue設定]
    D -->|No| F[FContains := False]
    E --> G[オプション設定]
    F --> G
    G --> H[型変換メソッド呼び出し]
    H --> I{FContains?}
    I -->|No| J{FRequired?}
    J -->|Yes| K[EHorseException 発生]
    J -->|No| L[デフォルト値を返却]
    I -->|Yes| M[型変換実行]
    M --> N{変換成功?}
    N -->|Yes| O[変換結果を返却]
    N -->|No| P[EHorseException 発生]
    O --> Q[終了]
    L --> Q
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-37-01 | 必須チェック | Required設定時、値がなければBadRequest例外を発生 | Required(True)設定時 |
| BR-37-02 | 変換エラー | 型変換に失敗した場合、BadRequest例外を発生 | 変換メソッド呼び出し時 |
| BR-37-03 | 空値処理 | 空文字列は各型のデフォルト値（0, 0.0, False, 0（日時）等）を返却 | 値が空の場合 |
| BR-37-04 | ブール変換 | TrueValueで指定された文字列（デフォルト: 'true'）と一致すればTrue | AsBoolean呼び出し時 |
| BR-37-05 | 日付フォーマット | DateFormatで指定された形式で日付を解析（デフォルト: 'yyyy-MM-dd'） | AsDate/AsDateTime呼び出し時 |
| BR-37-06 | リスト変換 | カンマで区切られた文字列をリストに変換 | AsList呼び出し時 |

### 計算ロジック

**ブール変換（AsBoolean）**:
```pascal
Result := LowerCase(LStrParam) = LowerCase(FTrueValue);
```

**日付変換（AsDate）**:
```pascal
LFormat := GetFormatSettings; // FDateFormat, FTimeFormatを使用
Result := StrToDate(Copy(LStrParam, 1, Length(FDateFormat)), LFormat);
```

**リスト変換（AsList）**:
```pascal
ValueArray := FValue.Split([',']);
for i := 0 to High(ValueArray) do
  Result.Add(ValueArray[i]);
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

本機能ではデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | 必須パラメータが存在しない | パラメータを指定する |
| 400 | BadRequest | 整数変換失敗 | 有効な整数値を指定する |
| 400 | BadRequest | 浮動小数点変換失敗 | 有効な数値を指定する |
| 400 | BadRequest | 日付変換失敗 | 正しい日付形式を指定する |
| 400 | BadRequest | ISO8601日時変換失敗 | ISO8601形式の日時を指定する |

### リトライ仕様

リトライは不要（クライアント側でパラメータを修正して再送信）。

## トランザクション仕様

トランザクション処理は行わない。

## パフォーマンス要件

- 型変換処理は軽量で、パフォーマンスへの影響は軽微
- 大量のパラメータを処理する場合も、各変換は独立して実行

## セキュリティ考慮事項

- 変換されたパラメータ値もユーザー入力として扱い、追加のバリデーションを行うこと
- SQLインジェクションやXSS攻撃を防ぐため、変換後の値も適切にエスケープすること

## 備考

- THorseCoreParamFieldはTHorseCoreParam.Fieldメソッドから取得
- 各変換メソッドはフルーエントインターフェースでオプション設定可能
- AsList<T>はDelphi専用（FPCではAsList: TList<string>のみ使用可能）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

THorseCoreParamFieldクラスの構造から始める。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | THorseCoreParamFieldクラスのフィールドとメソッド定義を確認 |

**読解のコツ**:
- FContains: パラメータが存在するかどうかのフラグ
- FValue: パラメータ値（文字列）
- FRequired: 必須フラグ
- FDateFormat, FTimeFormat: 日付・時刻フォーマット設定

**主要処理フロー**:
- **23-73行目**: THorseCoreParamFieldクラス定義
- **25-35行目**: プライベートフィールド
  - **25行目**: `FContains: Boolean;` - 存在フラグ
  - **27行目**: `FRequired: Boolean;` - 必須フラグ
  - **29行目**: `FDateFormat: string;` - 日付フォーマット
  - **30行目**: `FTimeFormat: string;` - 時刻フォーマット
  - **32行目**: `FTrueValue: string;` - ブール値のtrue文字列
  - **33行目**: `FValue: string;` - パラメータ値

#### Step 2: コンストラクタを理解する

インスタンス生成時の初期化処理を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | Createコンストラクタを確認 |

**主要処理フロー**:
- **313-335行目**: Create(AParams, AFieldName) コンストラクタ
  - **317-318行目**: フィールド初期化（FContains=False, FValue=EmptyStr）
  - **322-329行目**: パラメータ辞書からフィールド値を検索（大文字小文字を区別しない）
  - **331-332行目**: 値が見つかった場合、FContains=True, FValue設定
- **304-311行目**: Create(AStream, AFieldName) コンストラクタ（ストリーム用）

#### Step 3: 型変換メソッドを理解する

各型への変換処理を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | 各As*メソッドを確認 |

**主要処理フロー**:
- **277-284行目**: AsString メソッド
  - FContainsがTrueならFValueを返却
  - FalseかつFRequiredがTrueなら例外発生
- **176-189行目**: AsInteger メソッド
  - 空文字は0を返却
  - StrToIntで変換、失敗時はEHorseException発生
- **161-174行目**: AsInt64 メソッド
- **144-159行目**: AsFloat メソッド
  - 小数点セパレータを正規化してStrToFloat
- **88-96行目**: AsBoolean メソッド
  - LowerCase比較でFTrueValueと一致すればTrue
- **103-119行目**: AsDate メソッド
  - GetFormatSettingsでフォーマット取得
  - StrToDateで変換
- **121-138行目**: AsDateTime メソッド
- **286-302行目**: AsTime メソッド
- **191-201行目**: AsISO8601DateTime メソッド
  - ISO8601ToDate関数を使用
- **203-220行目**: AsList メソッド
  - Split([','])でカンマ区切り分割
- **264-275行目**: AsStream メソッド
  - FStreamを返却（Position=0にリセット）

#### Step 4: オプション設定を理解する

フルーエントインターフェースによるオプション設定を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | Required, DateFormat等のメソッドを確認 |

**主要処理フロー**:
- **400-404行目**: Required(AValue) メソッド
- **406-410行目**: Required() メソッド（引数なしはTrue設定）
- **343-347行目**: DateFormat(AValue) メソッド
- **441-445行目**: TimeFormat(AValue) メソッド
- **447-451行目**: TrueValue(AValue) メソッド

### プログラム呼び出し階層図

```
THorseCoreParam.Field(フィールド名)
    │
    └─ THorseCoreParamField.Create(AParams, AFieldName)
           │
           ├─ パラメータ辞書を検索（大文字小文字区別なし）
           │
           ├─ FContains, FValue を設定
           │
           └─ THorseCoreParamConfig からデフォルト設定を適用
                  │
                  ├─ DateFormat
                  ├─ TimeFormat
                  ├─ TrueValue
                  ├─ RequiredMessage
                  └─ InvalidFormatMessage

THorseCoreParamField.As* メソッド
    │
    ├─ FContains チェック
    │      │
    │      ├─ False かつ FRequired=True → EHorseException発生
    │      │
    │      └─ False かつ FRequired=False → デフォルト値返却
    │
    └─ 型変換実行
           │
           ├─ 成功 → 変換結果を返却
           │
           └─ 失敗 → EHorseException発生（BadRequest）
```

### データフロー図

```
[入力]                    [処理]                         [出力]

パラメータ辞書 ─────────▶ THorseCoreParamField.Create
                              │
                              ▼
                        フィールド検索
                              │
                              ▼
                        FContains, FValue 設定
                              │
オプション設定 ─────────────────┤
  - Required                   │
  - DateFormat                 │
  - TrueValue                  │
                              ▼
型変換メソッド呼び出し ───▶ AsInteger / AsString / AsDate / ...
                              │
                              ├─ 存在チェック
                              │
                              ├─ 変換処理
                              │
                              ▼
                        変換結果 or 例外
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | ソース | THorseCoreParamField 実装（型変換メソッド群） |
| Horse.Core.Param.pas | `src/Horse.Core.Param.pas` | ソース | THorseCoreParam 実装（Fieldメソッド） |
| Horse.Core.Param.Config.pas | `src/Horse.Core.Param.Config.pas` | ソース | THorseCoreParamConfig（デフォルト設定） |
| Horse.Exception.pas | `src/Horse.Exception.pas` | ソース | EHorseException（変換エラー時の例外） |
| Horse.Core.Param.Field.Brackets.pas | `src/Horse.Core.Param.Field.Brackets.pas` | ソース | LHS Brackets 機能（フィルタ演算子） |
