# 機能設計書 38-必須パラメータ検証

## 概要

本ドキュメントは、Horseフレームワークにおける必須パラメータ検証機能の詳細設計を記載したものである。THorseCoreParam.Required メソッドおよび THorseCoreParamField.Required メソッドを使用して、HTTPリクエストのパラメータが必須であることを宣言し、存在しない場合に適切なエラーレスポンスを返す機能について説明する。

### 本機能の処理概要

**業務上の目的・背景**：APIエンドポイントにおいて、特定のパラメータが必須である場合、そのパラメータが存在しないリクエストを適切にエラー処理する必要がある。本機能は、パラメータの必須チェックを宣言的に定義し、存在しない場合に HTTP 400 Bad Request エラーを自動的に返す機能を提供する。

**機能の利用シーン**：本機能は、以下のようなシーンで利用される。
- ユーザー登録時の必須フィールド検証（name, email, password）
- 検索APIの必須パラメータ検証（keyword）
- リソース更新時の必須フィールド検証
- フォームデータの必須入力チェック

**主要な処理内容**：
1. THorseCoreParam.Required(True) でパラメータ全体の必須設定
2. THorseCoreParamField.Required() または Required(True) で個別フィールドの必須設定
3. 型変換メソッド（AsString, AsInteger等）呼び出し時の存在チェック
4. パラメータが存在しない場合の EHorseException 発生

**関連システム・外部連携**：必須チェックは THorseCoreParam および THorseCoreParamField で実行され、エラー発生時は EHorseException として処理される。

**権限による制御**：必須パラメータ検証機能自体に権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面に直接関連しない（API処理時に使用） |

## 機能種別

パラメータ処理 / バリデーション

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AValue | Boolean | No | 必須フラグ（True: 必須、False: 任意） | なし |

### 入力データソース

- コールバック関数内でのメソッド呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| THorseCoreParam | THorseCoreParam | 設定後のパラメータオブジェクト（フルーエントインターフェース） |
| THorseCoreParamField | THorseCoreParamField | 設定後のフィールドオブジェクト（フルーエントインターフェース） |

### 出力先

パラメータ値取得時にエラーがある場合、HTTP 400 Bad Request レスポンスを返却

## 処理フロー

### 処理シーケンス

```
1. THorseCoreParam.Required(True) または THorseCoreParamField.Required() を呼び出し
   └─ FRequired フラグを True に設定
2. Field(フィールド名).AsString() 等の型変換メソッドを呼び出し
   └─ FContains をチェック（パラメータが存在するか）
      └─ 存在する場合: 値を返却
      └─ 存在しない場合かつ FRequired が True: EHorseException 発生
         └─ ステータス: THTTPStatus.BadRequest (400)
         └─ メッセージ: FRequiredMessage（デフォルト: "The %s param is required."）
      └─ 存在しない場合かつ FRequired が False: デフォルト値を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{Required設定?}
    B -->|Param.Required| C[THorseCoreParam.FRequired := True]
    B -->|Field.Required| D[THorseCoreParamField.FRequired := True]
    C --> E[Field取得時にFRequiredを継承]
    D --> F[型変換メソッド呼び出し]
    E --> F
    F --> G{FContains?}
    G -->|Yes| H[値を返却]
    G -->|No| I{FRequired?}
    I -->|Yes| J[EHorseException 発生]
    I -->|No| K[デフォルト値を返却]
    J --> L[400 Bad Request]
    H --> M[終了]
    K --> M
    L --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-38-01 | Param必須設定 | THorseCoreParam.Required(True)で以降のField取得時にFRequiredを継承 | Required設定後のField取得時 |
| BR-38-02 | Field必須設定 | THorseCoreParamField.Required()で個別フィールドを必須設定 | Required呼び出し時 |
| BR-38-03 | 必須エラー | FRequiredがTrueかつFContainsがFalseの場合、BadRequest例外を発生 | 型変換メソッド呼び出し時 |
| BR-38-04 | カスタムメッセージ | RequiredMessage()で必須エラー時のメッセージをカスタマイズ可能 | エラー発生時 |
| BR-38-05 | デフォルトメッセージ | FRequiredMessageのデフォルトは "The %s param is required." | RequiredMessage未設定時 |

### 計算ロジック

**必須チェック（AsString等のメソッド内）**:
```pascal
if FContains then
  Result := FValue
else if FRequired then
  RaiseHorseException(FRequiredMessage, [FFieldName]);
```

**THorseCoreParam.Required からの継承（Field メソッド内）**:
```pascal
Result
  .Required(FRequired)
  .DateFormat(...)
  .InvalidFormatMessage(...)
  ...
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

本機能ではデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | 必須パラメータが存在しない | パラメータを指定する |

### リトライ仕様

リトライは不要（クライアント側でパラメータを追加して再送信）。

## トランザクション仕様

トランザクション処理は行わない。

## パフォーマンス要件

- 必須チェックはブールフラグの比較のみであり、パフォーマンスへの影響は軽微

## セキュリティ考慮事項

- 必須チェックはバリデーションの一部であり、追加のセキュリティチェックは別途必要
- エラーメッセージにパラメータ名が含まれるため、機密性の高いパラメータ名は避けること

## 備考

- Required(True)とRequired()は同じ動作（フラグをTrueに設定）
- Required(False)でフラグをFalseに戻すことも可能
- THorseCoreParam.Required(True)は以降の全Field取得に影響
- THorseCoreParamField.Required()は個別フィールドのみに影響

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: THorseCoreParam の Required を理解する

パラメータ全体の必須設定から始める。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Core.Param.pas | `src/Horse.Core.Param.pas` | THorseCoreParam.Requiredメソッドを確認 |

**読解のコツ**:
- THorseCoreParam.Required はパラメータオブジェクト全体に影響
- 以降の Field 取得時に FRequired が継承される

**主要処理フロー**:
- **31行目**: `FRequired: Boolean;` - 必須フラグ フィールド
- **209-213行目**: Required メソッド
  ```pascal
  function THorseCoreParam.Required(const AValue: Boolean): THorseCoreParam;
  begin
    Result := Self;
    FRequired := AValue;
  end;
  ```

#### Step 2: Field メソッドでの Required 継承を理解する

Field 取得時に Required 設定がどう継承されるかを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.Param.pas | `src/Horse.Core.Param.pas` | Fieldメソッドを確認 |

**主要処理フロー**:
- **99-126行目**: Field メソッド
  - **112-119行目**: THorseCoreParamField 生成時にオプション設定
    ```pascal
    Result
      .Required(FRequired)  // ← ここで THorseCoreParam.FRequired を継承
      .DateFormat(THorseCoreParamConfig.GetInstance.DateFormat)
      ...
    ```

#### Step 3: THorseCoreParamField の Required を理解する

個別フィールドの必須設定を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | THorseCoreParamField.Requiredメソッドを確認 |

**主要処理フロー**:
- **27行目**: `FRequired: Boolean;` - 必須フラグ フィールド
- **400-404行目**: Required(AValue) メソッド
  ```pascal
  function THorseCoreParamField.Required(const AValue: Boolean): THorseCoreParamField;
  begin
    Result := Self;
    FRequired := AValue;
  end;
  ```
- **406-410行目**: Required() メソッド（引数なしバージョン）
  ```pascal
  function THorseCoreParamField.Required: THorseCoreParamField;
  begin
    Result := Self;
    FRequired := True;
  end;
  ```

#### Step 4: 必須チェックの実行を理解する

型変換メソッド内での必須チェック実行を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | AsString等の型変換メソッドを確認 |

**主要処理フロー**:
- **277-284行目**: AsString メソッド
  ```pascal
  function THorseCoreParamField.AsString: string;
  begin
    Result := EmptyStr;
    if FContains then
      Result := FValue
    else if FRequired then
      RaiseHorseException(FRequiredMessage, [FFieldName]);
  end;
  ```

#### Step 5: エラーメッセージのカスタマイズを理解する

必須エラー時のメッセージ設定を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | RequiredMessageメソッドを確認 |
| 5-2 | Horse.Core.Param.Config.pas | `src/Horse.Core.Param.Config.pas` | デフォルトメッセージを確認 |

**主要処理フロー**:
- **412-416行目**: RequiredMessage メソッド
  ```pascal
  function THorseCoreParamField.RequiredMessage(const AValue: string): THorseCoreParamField;
  begin
    Result := Self;
    FRequiredMessage := AValue;
  end;
  ```
- E-03 **55行目**: デフォルトメッセージ
  ```pascal
  FRequiredMessage := 'The %s param is required.';
  ```

### プログラム呼び出し階層図

```
THorseCoreParam.Required(True)
    │
    └─ FRequired := True
           │
           └─ Field(フィールド名)
                  │
                  └─ THorseCoreParamField.Create
                         │
                         └─ .Required(FRequired) ← 継承
                                │
                                └─ AsString() / AsInteger() / etc.
                                       │
                                       ├─ FContains? → Yes → 値を返却
                                       │
                                       └─ No → FRequired? → Yes → RaiseHorseException
                                                         │
                                                         └─ No → デフォルト値

THorseCoreParamField.Required()
    │
    └─ FRequired := True
           │
           └─ AsString() / AsInteger() / etc.
                  │
                  └─ （上記と同様の処理）
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Required(True) ─────────▶ FRequired := True
                              │
                              ▼
Field(フィールド名) ────▶ THorseCoreParamField.Create
                              │
                              ▼
                        .Required(FRequired) 継承
                              │
                              ▼
AsString() ─────────────▶ FContains チェック
                              │
                              ├─ Yes: 値を返却 ───────▶ 正常レスポンス
                              │
                              └─ No: FRequired チェック
                                      │
                                      ├─ Yes ─────────▶ 400 Bad Request
                                      │                 "The xxx param is required."
                                      │
                                      └─ No ──────────▶ デフォルト値返却
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.Param.pas | `src/Horse.Core.Param.pas` | ソース | THorseCoreParam 実装（Required メソッド） |
| Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | ソース | THorseCoreParamField 実装（Required, AsString 等） |
| Horse.Core.Param.Config.pas | `src/Horse.Core.Param.Config.pas` | ソース | THorseCoreParamConfig（デフォルトメッセージ） |
| Horse.Exception.pas | `src/Horse.Exception.pas` | ソース | EHorseException（エラー例外） |
