# 機能設計書 39-日付フォーマット設定

## 概要

本ドキュメントは、Horseフレームワークにおける日付フォーマット設定機能の詳細設計を記載したものである。THorseCoreParamConfig.DateFormat メソッドを使用して、HTTPリクエストパラメータの日付変換時に使用するフォーマット文字列を設定する機能について説明する。

### 本機能の処理概要

**業務上の目的・背景**：異なるシステムやクライアントは、様々な日付形式を使用する（例：yyyy-MM-dd, dd/MM/yyyy, MM-dd-yyyy など）。本機能は、アプリケーション全体またはフィールド単位で日付のパース形式を設定し、多様な日付形式に対応できるようにする。

**機能の利用シーン**：本機能は、以下のようなシーンで利用される。
- 国際化対応（地域ごとの日付形式サポート）
- レガシーシステム連携（既存システムの日付形式に合わせる）
- APIバージョン間の日付形式互換性維持
- フロントエンドの日付ピッカーからの値受け取り

**主要な処理内容**：
1. グローバル設定（THorseCoreParamConfig.DateFormat）
2. フィールド単位設定（THorseCoreParamField.DateFormat）
3. AsDate, AsDateTime メソッドでのフォーマット適用
4. TFormatSettings への DateSeparator, ShortDateFormat 設定

**関連システム・外部連携**：日付フォーマット設定は THorseCoreParamConfig（グローバル）および THorseCoreParamField（個別）で管理され、StrToDate/StrToDateTime 関数で使用される。

**権限による制御**：日付フォーマット設定機能自体に権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面に直接関連しない（API処理時に使用） |

## 機能種別

パラメータ処理 / 設定

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AValue | string | Yes | 日付フォーマット文字列（例：'yyyy-MM-dd'） | なし |

### 入力データソース

- アプリケーション初期化時の設定コード
- コールバック関数内でのフィールド単位設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| THorseCoreParamConfig | THorseCoreParamConfig | 設定後のコンフィグオブジェクト（フルーエントインターフェース） |
| THorseCoreParamField | THorseCoreParamField | 設定後のフィールドオブジェクト（フルーエントインターフェース） |

### 出力先

AsDate, AsDateTime メソッドでの日付パース時に使用

## 処理フロー

### 処理シーケンス

```
1. グローバル設定（オプション）
   └─ THorseCoreParamConfig.GetInstance.DateFormat('dd/MM/yyyy')
      └─ FDateFormat := 'dd/MM/yyyy'
2. Field 取得時にグローバル設定を継承
   └─ THorseCoreParam.Field(フィールド名)
      └─ THorseCoreParamField.DateFormat(THorseCoreParamConfig.GetInstance.DateFormat)
3. フィールド単位でのオーバーライド（オプション）
   └─ .DateFormat('yyyy/MM/dd')
      └─ FDateFormat := 'yyyy/MM/dd'
4. AsDate / AsDateTime 呼び出し時
   └─ GetFormatSettings でフォーマット設定を取得
      └─ FDateFormat に '-' が含まれる場合、DateSeparator := '-'
      └─ ShortDateFormat := FDateFormat
   └─ StrToDate / StrToDateTime でパース
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{グローバル設定?}
    B -->|Yes| C[THorseCoreParamConfig.DateFormat設定]
    B -->|No| D[デフォルト: yyyy-MM-dd]
    C --> E[Field取得]
    D --> E
    E --> F[グローバル設定を継承]
    F --> G{フィールド単位でオーバーライド?}
    G -->|Yes| H[THorseCoreParamField.DateFormat設定]
    G -->|No| I[AsDate/AsDateTime呼び出し]
    H --> I
    I --> J[GetFormatSettings]
    J --> K{フォーマットに - 含む?}
    K -->|Yes| L[DateSeparator := -]
    K -->|No| M[デフォルトセパレータ]
    L --> N[ShortDateFormat設定]
    M --> N
    N --> O[StrToDate/StrToDateTimeでパース]
    O --> P[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-39-01 | デフォルトフォーマット | FDateFormat のデフォルト値は 'yyyy-MM-dd' | 初期化時 |
| BR-39-02 | グローバル設定継承 | Field 取得時に THorseCoreParamConfig.DateFormat の値を継承 | Field呼び出し時 |
| BR-39-03 | フィールド単位オーバーライド | 個別の THorseCoreParamField.DateFormat で上書き可能 | DateFormat呼び出し時 |
| BR-39-04 | セパレータ自動検出 | フォーマットに '-' が含まれる場合、DateSeparator を '-' に設定 | GetFormatSettings時 |
| BR-39-05 | パース長制限 | AsDate では Copy(LStrParam, 1, Length(FDateFormat)) でフォーマット長分のみパース | AsDate呼び出し時 |

### 計算ロジック

**GetFormatSettings メソッド**:
```pascal
function THorseCoreParamField.GetFormatSettings: TFormatSettings;
begin
  {$IF DEFINED(FPC)}
    Result := DefaultFormatSettings;
  {$ELSE}
    Result := TFormatSettings.Create;
  {$ENDIF}
  if FDateFormat.IndexOf('-') > 0 then
    Result.DateSeparator := '-';
  Result.ShortDateFormat := FDateFormat;
  Result.ShortTimeFormat := FTimeFormat;
end;
```

**AsDate メソッドでの使用**:
```pascal
LFormat := GetFormatSettings;
Result := StrToDate(Copy(LStrParam, 1, Length(FDateFormat)), LFormat);
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

本機能ではデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | 指定されたフォーマットで日付をパースできない | 正しい日付形式を指定する |

### リトライ仕様

リトライは不要（クライアント側で日付形式を修正して再送信）。

## トランザクション仕様

トランザクション処理は行わない。

## パフォーマンス要件

- フォーマット設定はフィールドごとに1回のみ実行
- TFormatSettings の生成はパース時に毎回行われるが、軽量処理

## セキュリティ考慮事項

- 日付フォーマット設定自体にはセキュリティ上の考慮事項はない
- 日付パース後の値の範囲チェックは別途必要

## 備考

- TimeFormat も同様の機能を持ち、AsTime メソッドで使用される
- ISO8601形式の日付は AsISO8601DateTime メソッドで個別に処理される
- FDateFormat のデフォルト 'yyyy-MM-dd' は ISO 8601 の日付部分に準拠

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: グローバル設定を理解する

THorseCoreParamConfig での日付フォーマットグローバル設定から始める。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Core.Param.Config.pas | `src/Horse.Core.Param.Config.pas` | THorseCoreParamConfig.DateFormatメソッドを確認 |

**読解のコツ**:
- THorseCoreParamConfig はシングルトンパターン（GetInstance）
- FDateFormat のデフォルト値はコンストラクタで設定

**主要処理フロー**:
- **15行目**: `FDateFormat: string;` - 日付フォーマット フィールド
- **49-58行目**: コンストラクタでデフォルト値設定
  - **52行目**: `FDateFormat := 'yyyy-MM-dd';` - デフォルトフォーマット
- **60-64行目**: DateFormat(AValue) セッターメソッド
  ```pascal
  function THorseCoreParamConfig.DateFormat(const AValue: string): THorseCoreParamConfig;
  begin
    Result := Self;
    FDateFormat := AValue;
  end;
  ```
- **66-69行目**: DateFormat ゲッターメソッド
  ```pascal
  function THorseCoreParamConfig.DateFormat: string;
  begin
    Result := FDateFormat;
  end;
  ```

#### Step 2: フィールド単位設定を理解する

THorseCoreParamField での個別フォーマット設定を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | THorseCoreParamField.DateFormatメソッドを確認 |

**主要処理フロー**:
- **29行目**: `FDateFormat: string;` - 日付フォーマット フィールド
- **343-347行目**: DateFormat メソッド
  ```pascal
  function THorseCoreParamField.DateFormat(const AValue: string): THorseCoreParamField;
  begin
    Result := Self;
    FDateFormat := AValue;
  end;
  ```

#### Step 3: グローバル設定の継承を理解する

THorseCoreParam.Field でグローバル設定がどう継承されるかを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.Param.pas | `src/Horse.Core.Param.pas` | Fieldメソッドでの設定継承を確認 |

**主要処理フロー**:
- **99-126行目**: Field メソッド
  - **114行目**: グローバル設定継承
    ```pascal
    .DateFormat(THorseCoreParamConfig.GetInstance.DateFormat)
    ```

#### Step 4: GetFormatSettings を理解する

日付パース時のフォーマット設定生成を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | GetFormatSettingsメソッドを確認 |

**主要処理フロー**:
- **349-360行目**: GetFormatSettings メソッド
  ```pascal
  function THorseCoreParamField.GetFormatSettings: TFormatSettings;
  begin
    {$IF DEFINED(FPC)}
      Result := DefaultFormatSettings;
    {$ELSE}
      Result := TFormatSettings.Create;
    {$ENDIF}
    if FDateFormat.IndexOf('-') > 0 then
      Result.DateSeparator := '-';
    Result.ShortDateFormat := FDateFormat;
    Result.ShortTimeFormat := FTimeFormat;
  end;
  ```

#### Step 5: AsDate での使用を理解する

日付パース処理でのフォーマット適用を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | AsDateメソッドを確認 |

**主要処理フロー**:
- **103-119行目**: AsDate メソッド
  ```pascal
  function THorseCoreParamField.AsDate: TDateTime;
  var
    LStrParam: string;
    LFormat: TFormatSettings;
  begin
    Result := 0;
    LStrParam := Trim(AsString);
    try
      if LStrParam = EmptyStr then
        Exit;
      LFormat := GetFormatSettings;
      Result := StrToDate(Copy(LStrParam, 1, Length(FDateFormat)), LFormat);
    except
      on E: EConvertError do
        RaiseHorseException(FInvalidFormatMessage, [FFieldName, LStrParam, 'date']);
    end;
  end;
  ```

### プログラム呼び出し階層図

```
THorseCoreParamConfig.GetInstance.DateFormat('dd/MM/yyyy')
    │
    └─ FDateFormat := 'dd/MM/yyyy' （グローバル設定）

THorseCoreParam.Field(フィールド名)
    │
    └─ THorseCoreParamField.Create
           │
           └─ .DateFormat(THorseCoreParamConfig.GetInstance.DateFormat)
                  │
                  └─ FDateFormat := 'dd/MM/yyyy' （継承）

THorseCoreParamField.DateFormat('yyyy/MM/dd')  （オプション）
    │
    └─ FDateFormat := 'yyyy/MM/dd' （オーバーライド）

THorseCoreParamField.AsDate
    │
    └─ GetFormatSettings
           │
           ├─ TFormatSettings.Create
           │
           ├─ FDateFormat に '-' 含む? → DateSeparator := '-'
           │
           └─ ShortDateFormat := FDateFormat
                  │
                  └─ StrToDate(パラメータ値, LFormat)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

DateFormat('dd/MM/yyyy') ▶ THorseCoreParamConfig
                              │
                              ▼
                        FDateFormat 設定
                              │
                              ▼
Field(フィールド名) ────▶ THorseCoreParamField
                              │
                              ▼
                        グローバル設定継承
                              │
                              ▼
DateFormat('yyyy/MM/dd') ──▶ FDateFormat オーバーライド（オプション）
                              │
                              ▼
AsDate() ────────────────▶ GetFormatSettings
                              │
                              ▼
                        TFormatSettings 生成
                              │
                              ▼
                        StrToDate(値, LFormat)
                              │
                              ▼
                        TDateTime 値
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.Param.Config.pas | `src/Horse.Core.Param.Config.pas` | ソース | THorseCoreParamConfig（グローバル設定） |
| Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | ソース | THorseCoreParamField（DateFormat, GetFormatSettings, AsDate） |
| Horse.Core.Param.pas | `src/Horse.Core.Param.pas` | ソース | THorseCoreParam（Field メソッドでの継承） |
