# 機能設計書 4-DELETE リクエスト処理

## 概要

本ドキュメントは、Horseフレームワークにおける HTTP DELETE リクエストを処理するルート登録機能の設計を記述する。

### 本機能の処理概要

HTTP DELETEリクエストを受け付けるエンドポイントをルーティングテーブルに登録し、リクエスト受信時に対応するコールバック関数を実行する機能を提供する。

**業務上の目的・背景**：WebアプリケーションやRESTful APIにおいて、DELETEリクエストはリソースの削除に使用されるHTTPメソッドである。DELETE リクエスト処理機能は、クライアントからのリソース削除要求を受け付け、サーバー側で指定されたリソースを削除する操作を行うためのルーティング基盤を提供する。DELETEはべき等（idempotent）な操作であり、同じリクエストを複数回実行しても結果は同じになる特性を持つ（既に削除されていれば404を返す等）。

**機能の利用シーン**：
- ユーザー削除API（例: `/users/:id`）
- リソースの削除操作（例: `/documents/:id`）
- セッションの終了（ログアウト）（例: `/sessions/:id`）
- キャッシュのクリア（例: `/cache/:key`）

**主要な処理内容**：
1. `THorseCore.Delete` メソッドを呼び出し、パスとコールバック関数を指定
2. パスの正規化（先頭・末尾のスラッシュ処理）
3. 事前登録されたコールバック（AddCallback）があれば先に登録
4. `THorseRouterTree.RegisterRoute` でルーティングテーブルに DELETE ルートを追加
5. リクエスト受信時、パスマッチングを行いコールバックを実行

**関連システム・外部連携**：本機能は単独で動作するルーティング機能であり、外部システムとの直接的な連携はない。ただし、登録されたコールバック内でデータベースからのレコード削除や外部サービスへの削除通知を行うことが一般的である。

**権限による制御**：本機能自体には権限制御は含まれない。権限制御はミドルウェア機能（機能No.8）と組み合わせて実装する。削除操作は特に権限チェックが重要となる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングにDELETEリクエスト処理に直接関連する画面は定義されていない |

## 機能種別

HTTPルーティング / リクエスト処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| APath | string | Yes | ルーティングパス（例: '/users/:id', '/documents/:id'） | 空文字不可、先頭・末尾のスラッシュは自動正規化 |
| ACallback | THorseCallback | Yes | リクエスト処理を行うコールバック関数 | nil不可 |

### コールバック型のバリエーション

| 型名 | シグネチャ | 説明 |
|-----|-----------|------|
| THorseCallback | `procedure(AReq: THorseRequest; ARes: THorseResponse; ANext: TNextProc)` | 完全な形式（Next呼び出し可能） |
| THorseCallbackRequestResponse | `procedure(AReq: THorseRequest; ARes: THorseResponse)` | リクエストとレスポンスを使用 |
| THorseCallbackRequest | `procedure(AReq: THorseRequest)` | リクエストのみ使用 |
| THorseCallbackResponse | `procedure(ARes: THorseResponse)` | レスポンスのみ使用（Delphi専用） |

### 入力データソース

アプリケーション起動時のルート登録コード（ソースコード内で静的に定義）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | THorseCore | メソッドチェーン用のTHorseCoreインスタンス |

### 出力先

- 内部データ構造: `THorseRouterTree.FCallBack` ディクショナリ（TMethodType.mtDelete キーで格納）
- 内部データ構造: `THorseRouterTree.FRoute` ディクショナリ（パスセグメントをキーとしたツリー構造）

## 処理フロー

### 処理シーケンス

```
1. THorseCore.Delete(APath, ACallback) 呼び出し
   └─ 複数のオーバーロードメソッドが存在し、適切な形式に変換

2. GetCallback によるコールバック正規化
   └─ THorseCallbackRequestResponse → THorseCallback への変換
   └─ THorseCallbackRequest → THorseCallback への変換（Status=NoContent設定）

3. RegisterCallbacksRoute(mtDelete, APath) 実行
   └─ AddCallback で事前登録されたコールバックを同じパスに登録

4. RegisterRoute(mtDelete, APath, ACallback) 実行
   └─ TrimPath でパス正規化（'/' + APath.Trim(['/'])）
   └─ THorseRouterTree.RegisterRoute 呼び出し

5. THorseRouterTree.RegisterRoute 内部処理
   └─ GetQueuePath でパスをキュー化
   └─ RegisterInternal でツリー構造に登録
```

### フローチャート

```mermaid
flowchart TD
    A[THorseCore.Delete 呼び出し] --> B{コールバック型チェック}
    B -->|THorseCallbackRequestResponse| C[GetCallback で変換]
    B -->|THorseCallbackRequest| D[GetCallback で変換<br/>Status=NoContent設定]
    B -->|THorseCallback| E[そのまま使用]
    C --> F[RegisterCallbacksRoute]
    D --> F
    E --> F
    F --> G{事前登録コールバックあり?}
    G -->|Yes| H[事前コールバックを先に登録]
    G -->|No| I[RegisterRoute]
    H --> I
    I --> J[TrimPath でパス正規化]
    J --> K[THorseRouterTree.RegisterRoute]
    K --> L[GetQueuePath でパスをキュー化]
    L --> M[RegisterInternal でツリーに登録]
    M --> N[終了: THorseCore を返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | パス正規化 | パスは先頭にスラッシュを付与し、末尾のスラッシュは除去 | 全てのルート登録時 |
| BR-02 | パスパラメータ | `:param` 形式でパスパラメータを定義可能 | パス内に`:` で始まるセグメントがある場合 |
| BR-03 | 正規表現ルート | `(regex)` 形式で正規表現パターンを定義可能 | パス内に`()`で囲まれたセグメントがある場合 |
| BR-04 | コールバック蓄積 | 同一パス・メソッドに複数のコールバック登録可能 | AddCallback使用時 |
| BR-05 | べき等性 | DELETEは同じリクエストを複数回実行しても同じ結果となるべき等な操作 | DELETE API設計時 |
| BR-06 | コンパイラ制約 | Delphi XE6以前では条件コンパイルにより利用不可 | CompilerVersion <= 27.0 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | THorseCoreインスタンスが既に作成済みの状態でCreate呼び出し | シングルトンパターンのため、GetInstanceを使用 |
| - | コンパイルエラー | Delphi XE6以前でDeleteメソッドを使用 | コンパイラバージョンを確認、または別の方法で実装 |

### リトライ仕様

リトライは不要（同期的なルート登録処理）

## トランザクション仕様

トランザクション管理は不要（メモリ上のデータ構造操作のみ）

## パフォーマンス要件

- ルート登録はアプリケーション起動時に1回のみ実行されるため、厳密な性能要件はない
- ルートマッチングはO(n)（nはパスセグメント数）で完了

## セキュリティ考慮事項

- DELETE リクエストは破壊的操作のため、適切な認証・認可が必須
- 削除前に所有者確認を行い、他ユーザーのリソースを削除できないようにすること
- 論理削除（soft delete）と物理削除（hard delete）を適切に使い分けること
- 削除操作のログ記録を推奨

## 備考

- DELETEメソッドはDelphi XE7（CompilerVersion > 27.0）以降およびFree Pascal環境で利用可能
- `{$IF (defined(fpc) or (CompilerVersion > 27.0))}` 条件コンパイルにより制御されている
- GET/POST/PUTと同様の実装パターンであり、HTTPメソッドタイプ（mtDelete）のみが異なる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

コールバック型の定義を理解することで、ルート登録の引数パターンを把握できる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Callback.pas | `src/Horse.Callback.pas` | コールバック型（THorseCallback等）の定義を確認 |
| 1-2 | Horse.Commons.pas | `src/Horse.Commons.pas` | TMethodType列挙型でHTTPメソッド種別を確認（mtDelete） |

**読解のコツ**: TMethodType列挙型には mtDelete が含まれており、これがDELETEメソッドを識別するキーとなる。

#### Step 2: エントリーポイントを理解する

`THorseCore.Delete` メソッドがエントリーポイントとなる。条件コンパイルに注意。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.pas | `src/Horse.Core.pas` | THorseCoreクラスのDeleteメソッド群を確認 |

**主要処理フロー**:
1. **116-123行目**: Delete メソッドのオーバーロード宣言（4種類）※条件コンパイル内
2. **268-292行目**: Delete メソッド実装群 ※`{$IF (defined(fpc) or (CompilerVersion > 27.0))}`内
3. **287-291行目**: 基本形式の Delete メソッド実装（THorseCallback型）

#### Step 3: 条件コンパイルを理解する

Delete/Patchメソッドは古いDelphiバージョンでは利用不可。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.pas | `src/Horse.Core.pas` | 条件コンパイル `{$IF (defined(fpc) or (CompilerVersion > 27.0))}` |

**読解のコツ**:
- `CompilerVersion > 27.0` はDelphi XE7以降を意味する
- FPC（Free Pascal Compiler）環境では常に利用可能
- 古いDelphiバージョンでは該当コードがコンパイルから除外される

#### Step 4: ルート登録処理を理解する

内部的なルート登録の流れを確認する。他のHTTPメソッドと同様。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.pas | `src/Horse.Core.pas` | RegisterRoute/RegisterCallbacksRoute メソッド |
| 4-2 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | THorseRouterTree.RegisterRoute/RegisterInternal |

**主要処理フロー**:
- **186-193行目（Horse.Core.pas）**: RegisterCallbacksRoute で事前コールバック登録
- **205-212行目（Horse.Core.pas）**: RegisterRoute でルーティングテーブルに登録
- **71-81行目（Horse.Core.RouterTree.pas）**: THorseRouterTree.RegisterRoute 実装

### プログラム呼び出し階層図

```
THorseCore.Delete(APath, ACallback)
    │
    ├─ GetCallback(ACallback)  ※型変換が必要な場合
    │      └─ 無名メソッドでラップしてTHorseCallback形式に統一
    │
    ├─ RegisterCallbacksRoute(mtDelete, APath)
    │      └─ GetCallbacks() で事前登録コールバック取得
    │      └─ RegisterRoute(mtDelete, APath, LCallback) を各コールバックに対して実行
    │
    └─ RegisterRoute(mtDelete, APath, ACallback)
           └─ GetInstance() でシングルトンインスタンス取得
           └─ TrimPath(APath) でパス正規化
           └─ THorseRouterTree.RegisterRoute(AHTTPType, APath, ACallback)
                  └─ GetQueuePath(APath) でパスをセグメント化
                  └─ RegisterInternal(AHTTPType, APath, ACallback)
                         └─ ForcePath() で中間ノード作成
                         └─ FCallBack.Add() でコールバック登録
```

### データフロー図

```
[入力]                      [処理]                           [出力]

パス文字列 ─────────────────┐
("/users/:id")              │
                            ▼
コールバック関数 ──────► THorseCore.Delete ──► TrimPath ──► "/' + path.Trim(['/'])"
                            │                               │
                            ▼                               ▼
                    GetCallback         THorseRouterTree.RegisterRoute
                    (型変換)                    │
                            │                   ▼
                            │           GetQueuePath
                            │           (["", "users", ":id"])
                            │                   │
                            ▼                   ▼
               THorseCallback形式 ──────► RegisterInternal ──► FCallBack[mtDelete]
                                                               FRoute["users"]
                                                                 └─FRoute[":id"]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.pas | `src/Horse.Core.pas` | ソース | THorseCoreクラス - Deleteメソッド等HTTPルーティングのエントリーポイント |
| Horse.Callback.pas | `src/Horse.Callback.pas` | ソース | コールバック型定義（THorseCallback等） |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | TMethodType列挙型（mtDelete）、THTTPStatus等の共通定義 |
| Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ソース | THorseRouterTreeクラス - ルーティングテーブルのツリー構造管理 |
| Horse.Request.pas | `src/Horse.Request.pas` | ソース | THorseRequestクラス - リクエスト情報のラッパー |
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponseクラス - レスポンス処理のラッパー |
