# 機能設計書 40-カスタム例外

## 概要

本ドキュメントは、Horseフレームワークにおけるカスタム例外機能の詳細設計を記載したものである。EHorseException クラスを使用して、HTTP例外を構造化データとして定義し、エラーレスポンスをJSON形式で返却する機能について説明する。

### 本機能の処理概要

**業務上の目的・背景**：RESTful APIでは、エラー発生時に適切なHTTPステータスコードとともに、構造化されたエラー情報をクライアントに返却する必要がある。本機能は、EHorseException クラスを通じて、エラーメッセージ、HTTPステータス、エラーコード、ヒント情報などを一元管理し、JSON形式で出力する仕組みを提供する。

**機能の利用シーン**：本機能は、以下のようなシーンで利用される。
- バリデーションエラー発生時（400 Bad Request）
- 認証エラー発生時（401 Unauthorized）
- リソース未発見時（404 Not Found）
- サーバー内部エラー発生時（500 Internal Server Error）
- ビジネスロジックエラー発生時（カスタムエラーコード付き）

**主要な処理内容**：
1. EHorseException インスタンスの生成（New または Create）
2. フルーエントインターフェースによるエラー情報設定
3. ToJSON / ToJSONObject によるJSON形式への変換
4. 例外発生（raise）によるエラーレスポンス返却

**関連システム・外部連携**：EHorseException は Horse フレームワークの例外ハンドラと連携し、HTTPレスポンスとしてクライアントに返却される。

**権限による制御**：カスタム例外機能自体に権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面に直接関連しない（API処理時に使用） |

## 機能種別

例外処理 / エラーハンドリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Error | string | No | エラーメッセージ | なし |
| Status | THTTPStatus | No | HTTPステータスコード | なし |
| Type | TMessageType | No | メッセージ種別（Error, Warning, Information） | なし |
| Title | string | No | エラータイトル | なし |
| Code | Integer | No | アプリケーション固有のエラーコード | なし |
| Hint | string | No | 解決のヒント | なし |
| Unit | string | No | エラー発生元ユニット名 | なし |
| Detail | string | No | 詳細情報 | なし |

### 入力データソース

- アプリケーションコード内での例外生成

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| EHorseException | EHorseException | 設定後の例外オブジェクト（フルーエントインターフェース） |
| JSON | string | ToJSON メソッドによるJSON文字列 |
| TJSONObject | TJSONObject | ToJSONObject メソッドによるJSONオブジェクト |

### 出力先

HTTPレスポンスボディ（JSON形式）

## 処理フロー

### 処理シーケンス

```
1. EHorseException.New または EHorseException.Create で例外インスタンス生成
   └─ FError := EmptyStr
   └─ FStatus := THTTPStatus.InternalServerError (500)
   └─ FCode := 0
2. フルーエントインターフェースでエラー情報設定
   └─ .Error('エラーメッセージ')
   └─ .Status(THTTPStatus.BadRequest)
   └─ .Type(TMessageType.Error)
   └─ .Title('エラータイトル')
   └─ .Code(1001)
   └─ .Hint('解決のヒント')
   └─ .Unit('ユニット名')
   └─ .Detail('詳細情報')
3. 例外発生（raise）
4. フレームワークの例外ハンドラがキャッチ
   └─ ToJSONObject でJSON生成
   └─ HTTPレスポンスとして返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[EHorseException.New]
    B --> C[初期値設定]
    C --> D[FStatus := InternalServerError]
    D --> E{エラー情報設定}
    E -->|Error| F[.Error設定]
    E -->|Status| G[.Status設定]
    E -->|Type| H[.Type設定]
    E -->|Title| I[.Title設定]
    E -->|Code| J[.Code設定]
    E -->|Hint| K[.Hint設定]
    E -->|Unit| L[.Unit設定]
    E -->|Detail| M[.Detail設定]
    F --> N[raise 例外]
    G --> N
    H --> N
    I --> N
    J --> N
    K --> N
    L --> N
    M --> N
    N --> O[例外ハンドラでキャッチ]
    O --> P[ToJSONObject呼び出し]
    P --> Q[TJSONObject生成]
    Q --> R{各フィールドチェック}
    R -->|Type != Default| S[type追加]
    R -->|Title非空| T[title追加]
    R -->|Code != 0| U[code追加]
    R -->|常時| V[error追加]
    R -->|Hint非空| W[hint追加]
    R -->|Unit非空| X[unit追加]
    R -->|Detail非空| Y[detail追加]
    S --> Z[HTTPレスポンス返却]
    T --> Z
    U --> Z
    V --> Z
    W --> Z
    X --> Z
    Y --> Z
    Z --> AA[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-40-01 | デフォルトステータス | FStatus のデフォルト値は THTTPStatus.InternalServerError (500) | 初期化時 |
| BR-40-02 | デフォルトコード | FCode のデフォルト値は 0 | 初期化時 |
| BR-40-03 | Error設定時Message同期 | Error(AValue) 呼び出し時に Self.Message も同時に設定 | Error呼び出し時 |
| BR-40-04 | JSON出力条件（type） | FType が TMessageType.Default 以外の場合のみ出力 | ToJSONObject時 |
| BR-40-05 | JSON出力条件（title,hint,unit,detail） | 各フィールドが空文字でない場合のみ出力 | ToJSONObject時 |
| BR-40-06 | JSON出力条件（code） | FCode が 0 以外の場合のみ出力 | ToJSONObject時 |
| BR-40-07 | JSON必須出力（error） | error フィールドは常に出力 | ToJSONObject時 |

### 計算ロジック

**ToJSONObject メソッド**:
```pascal
function EHorseException.ToJSONObject: TJsonObject;
begin
  Result := TJSONObject.Create;

  if FType <> TMessageType.Default then
    Result.AddPair('type', GetEnumName(TypeInfo(TMessageType), Integer(FType)));

  if not FTitle.Trim.IsEmpty then
    Result.AddPair('title', FTitle);

  if FCode <> 0 then
    Result.AddPair('code', TJSONNumber.Create(FCode));

  Result.AddPair('error', FError);

  if not FHint.Trim.IsEmpty then
    Result.AddPair('hint', FHint);

  if not FUnit.Trim.IsEmpty then
    Result.AddPair('unit', FUnit);

  if not FDetail.Trim.IsEmpty then
    Result.AddPair('detail', FDetail);
end;
```

**Error メソッドでの Message 同期**:
```pascal
function EHorseException.Error(const AValue: string): EHorseException;
begin
  FError := AValue;
  Self.Message := AValue;  // Exception.Message と同期
  Result := Self;
end;
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

本機能ではデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | バリデーションエラー | 入力値を修正する |
| 401 | Unauthorized | 認証エラー | 認証情報を確認する |
| 403 | Forbidden | 権限エラー | 権限を確認する |
| 404 | NotFound | リソース未発見 | URLを確認する |
| 500 | InternalServerError | サーバー内部エラー | システム管理者に連絡する |

### リトライ仕様

例外の種類による（クライアントエラーはリトライ不要、サーバーエラーは時間を置いてリトライ）。

## トランザクション仕様

トランザクション処理は行わない。

## パフォーマンス要件

- 例外生成は軽量な処理
- JSON変換はオンデマンドで実行

## セキュリティ考慮事項

- エラーメッセージに機密情報を含めないこと
- Detail フィールドにスタックトレースを含める場合は本番環境で無効化すること
- Unit フィールドは開発/デバッグ目的でのみ使用すること

## 備考

- EHorseException は Exception クラスを継承
- THTTPStatus は Horse.Commons で定義された列挙型
- TMessageType は Default, Error, Warning, Information の4種類
- フルーエントインターフェースによりメソッドチェーンが可能
- Delphi と Free Pascal の両方に対応（条件コンパイル使用）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 共通型定義を理解する

THTTPStatus と TMessageType の定義から始める。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Commons.pas | `src/Horse.Commons.pas` | THTTPStatus と TMessageType の定義を確認 |

**読解のコツ**:
- THTTPStatus は HTTP標準ステータスコードを列挙型として定義
- TMessageType は Default, Error, Warning, Information の4種類

**主要処理フロー**:
- **24-87行目**: THTTPStatus 列挙型定義
  ```pascal
  THTTPStatus = (
    Continue = 100,
    SwitchingProtocols = 101,
    ...
    BadRequest = 400,
    ...
    InternalServerError = 500,
    ...
  );
  ```
- **108行目**: TMessageType 列挙型定義
  ```pascal
  TMessageType = (Default, Error, Warning, Information);
  ```
- **115-117行目**: THTTPStatusHelper（ToInteger メソッド）
  ```pascal
  THTTPStatusHelper = record helper for THTTPStatus
    function ToInteger: Integer;
  end;
  ```

#### Step 2: EHorseException クラス構造を理解する

例外クラスのフィールドとメソッド宣言を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Exception.pas | `src/Horse.Exception.pas` | EHorseException クラス宣言を確認 |

**主要処理フロー**:
- **21-52行目**: EHorseException クラス宣言
  ```pascal
  EHorseException = class(Exception)
  strict private
    FError: string;          // エラーメッセージ
    FStatus: THTTPStatus;    // HTTPステータス
    FType: TMessageType;     // メッセージ種別
    FTitle: string;          // タイトル
    FCode: Integer;          // エラーコード
    FHint: string;           // ヒント
    FUnit: string;           // ユニット名
    FDetail: string;         // 詳細
  public
    constructor Create; reintroduce;
    function Error(const AValue: string): EHorseException; overload;
    function Error: string; overload;
    // ... 他のセッター/ゲッター
    function ToJSON: string; virtual;
    function ToJSONObject: TJSONObject; virtual;
    class function New: EHorseException;
  end;
  ```

#### Step 3: コンストラクタと初期化を理解する

インスタンス生成時の初期値設定を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Exception.pas | `src/Horse.Exception.pas` | Create と New メソッドを確認 |

**主要処理フロー**:
- **63-68行目**: Create コンストラクタ
  ```pascal
  constructor EHorseException.Create;
  begin
    FError := EmptyStr;
    FStatus := THTTPStatus.InternalServerError;  // デフォルト: 500
    FCode := 0;
  end;
  ```
- **70-73行目**: New クラスメソッド
  ```pascal
  class function EHorseException.New: EHorseException;
  begin
    Result := EHorseException.Create;
  end;
  ```

#### Step 4: フルーエントインターフェースを理解する

各プロパティのセッター/ゲッターを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Exception.pas | `src/Horse.Exception.pas` | 各メソッドの実装を確認 |

**主要処理フロー**:
- **124-129行目**: Error セッター（Message 同期あり）
  ```pascal
  function EHorseException.Error(const AValue: string): EHorseException;
  begin
    FError := AValue;
    Self.Message := AValue;  // 重要: Exception.Message と同期
    Result := Self;
  end;
  ```
- **136-140行目**: Status セッター
  ```pascal
  function EHorseException.Status(const AValue: THTTPStatus): EHorseException;
  begin
    FStatus := AValue;
    Result := Self;
  end;
  ```
- **75-79行目**: Code セッター
  ```pascal
  function EHorseException.Code(const AValue: Integer): EHorseException;
  begin
    FCode := AValue;
    Result := Self;
  end;
  ```

#### Step 5: JSON変換を理解する

ToJSON と ToJSONObject の実装を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | Horse.Exception.pas | `src/Horse.Exception.pas` | ToJSONObject の条件分岐を確認 |

**主要処理フロー**:
- **164-174行目**: ToJSON メソッド
  ```pascal
  function EHorseException.ToJSON: string;
  var
    LJSON: TJSONObject;
  begin
    LJSON := ToJSONObject;
    try
      Result := LJSON.ToJSON;  // または AsJSON (FPC)
    finally
      LJSON.Free;
    end;
  end;
  ```
- **176-199行目**: ToJSONObject メソッド
  ```pascal
  function EHorseException.ToJSONObject: TJsonObject;
  begin
    Result := TJSONObject.Create;

    // 条件付き出力: type (Default以外の場合のみ)
    if FType <> TMessageType.Default then
      Result.AddPair('type', GetEnumName(TypeInfo(TMessageType), Integer(FType)));

    // 条件付き出力: title (空でない場合のみ)
    if not FTitle.Trim.IsEmpty then
      Result.AddPair('title', FTitle);

    // 条件付き出力: code (0以外の場合のみ)
    if FCode <> 0 then
      Result.AddPair('code', TJSONNumber.Create(FCode));

    // 必須出力: error
    Result.AddPair('error', FError);

    // 条件付き出力: hint, unit, detail (空でない場合のみ)
    if not FHint.Trim.IsEmpty then
      Result.AddPair('hint', FHint);

    if not FUnit.Trim.IsEmpty then
      Result.AddPair('unit', FUnit);

    if not FDetail.Trim.IsEmpty then
      Result.AddPair('detail', FDetail);
  end;
  ```

### プログラム呼び出し階層図

```
[アプリケーションコード]
    │
    └─ EHorseException.New
           │
           └─ EHorseException.Create
                  │
                  ├─ FError := EmptyStr
                  ├─ FStatus := InternalServerError
                  └─ FCode := 0
                         │
                         └─ .Error('エラーメッセージ')
                                │
                                ├─ FError := 'エラーメッセージ'
                                └─ Self.Message := 'エラーメッセージ'
                                       │
                                       └─ .Status(THTTPStatus.BadRequest)
                                              │
                                              └─ FStatus := BadRequest
                                                     │
                                                     └─ .Code(1001)
                                                            │
                                                            └─ FCode := 1001
                                                                   │
                                                                   └─ raise
                                                                          │
                                                                          └─ [例外ハンドラ]
                                                                                 │
                                                                                 └─ ToJSONObject
                                                                                        │
                                                                                        ├─ TJSONObject.Create
                                                                                        ├─ AddPair('error', FError)
                                                                                        ├─ AddPair('code', FCode)
                                                                                        └─ HTTPレスポンス生成
```

### データフロー図

```
[入力]                    [処理]                         [出力]

EHorseException.New ────▶ Create コンストラクタ
                              │
                              ▼
                        初期値設定
                        FStatus := 500
                        FCode := 0
                              │
                              ▼
.Error('msg') ─────────▶ FError 設定
                        Self.Message 同期
                              │
                              ▼
.Status(BadRequest) ───▶ FStatus 設定
                              │
                              ▼
.Code(1001) ───────────▶ FCode 設定
                              │
                              ▼
raise ─────────────────▶ 例外発生
                              │
                              ▼
[例外ハンドラ] ────────▶ ToJSONObject
                              │
                              ▼
                        TJSONObject 生成
                              │
                              ├─ type: "Error" (条件付き)
                              ├─ title: "..." (条件付き)
                              ├─ code: 1001 (条件付き)
                              ├─ error: "msg" (必須)
                              ├─ hint: "..." (条件付き)
                              ├─ unit: "..." (条件付き)
                              └─ detail: "..." (条件付き)
                                      │
                                      ▼
                              HTTP レスポンス
                              Status: 400
                              Body: {"error":"msg","code":1001}
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Exception.pas | `src/Horse.Exception.pas` | ソース | EHorseException クラス実装 |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | THTTPStatus, TMessageType 定義 |
