# 機能設計書 45-MIMEタイプ判定

## 概要

本ドキュメントは、Horseフレームワークにおけるファイル拡張子からMIMEタイプを判定するTHorseMimeTypesクラスについて記述する。

### 本機能の処理概要

本機能は、ファイル名またはファイル拡張子から適切なMIMEタイプ文字列を取得し、HTTPレスポンスのContent-Type設定を自動化する。

**業務上の目的・背景**：静的ファイル配信やファイルダウンロード機能では、ファイルの種類に応じた適切なContent-Typeヘッダーを設定する必要がある。手動でMIMEタイプを指定すると、ファイル種類の増加に伴いコードが煩雑になる。本機能により、ファイル拡張子に基づいてMIMEタイプを自動判定し、開発効率と保守性を向上させる。

**機能の利用シーン**：本機能は以下のような場面で使用される。
- THorseResponse.SendFileメソッドでファイルを送信する際のContent-Type自動設定
- THorseResponse.DownloadメソッドでダウンロードファイルのContent-Type設定
- THorseCoreFileクラスでファイルのMIMEタイプを取得する際

**主要な処理内容**：
1. ファイルパスまたは拡張子を入力として受け取る
2. 拡張子を正規化（小文字変換、ドット除去）
3. 内部辞書またはSystem.Net.Mimeから対応するMIMEタイプを検索
4. MIMEタイプ文字列を返却（不明な場合はapplication/octet-stream）

**関連システム・外部連携**：ファイル送信機能、レスポンス生成機能と連携。

**権限による制御**：本機能に権限による制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に依存しないユーティリティ機能 |

## 機能種別

ユーティリティ / データ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AFileName | string | Yes | ファイルパスまたはファイル名 | GetFileType用 |
| AFileExt | string | Yes | ファイル拡張子（.を含む/含まない両対応） | GetExtType用 |

### 入力データソース

アプリケーションコードからの直接呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | string | MIMEタイプ文字列（例: 'application/json'） |

### 出力先

メソッド戻り値として返却

## 処理フロー

### 処理シーケンス

```
1. GetFileType または GetExtType 呼び出し
2. コンパイラディレクティブによる分岐
   └─ Delphi 32.0以降: System.Net.Mime.TMimeTypes使用
   └─ それ以外: THorseMimeTypesExt使用
3. THorseMimeTypesExt の場合:
   └─ NormalizeExt で拡張子を正規化
   └─ FFileType辞書から検索
   └─ 見つからない場合は 'application/octet-stream'
4. MIMEタイプ文字列を返却
```

### フローチャート

```mermaid
flowchart TD
    A[GetFileType/GetExtType呼び出し] --> B{コンパイラバージョン}
    B -->|Delphi >= 32.0| C[System.Net.Mime.TMimeTypes使用]
    B -->|それ以外| D[THorseMimeTypesExt使用]
    D --> E[NormalizeExt で正規化]
    E --> F[FFileType辞書検索]
    F --> G{見つかった?}
    G -->|Yes| H[MIMEタイプ返却]
    G -->|No| I["'application/octet-stream'"]
    C --> H
    I --> H
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | デフォルトMIME | 不明な拡張子はapplication/octet-streamを返却 | 辞書に存在しない場合 |
| BR-02 | 大文字小文字 | 拡張子は小文字に正規化して判定 | 常時 |
| BR-03 | ドット除去 | 拡張子の先頭ドットは除去して判定 | 拡張子がドットで始まる場合 |
| BR-04 | シングルトン | THorseMimeTypesExtはシングルトンパターンで実装 | 常時 |

### 計算ロジック

```pascal
function THorseMimeTypesExt.NormalizeExt(const AFileExt: String): string;
begin
  Result := ExtractFileExt(AFileExt.Trim.ToLower);
  if (Result <> EmptyStr) and (Result[Low(Result)] = '.') then
    Result := Result.Substring(1);
end;
```

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作なし

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 拡張子なしまたは不明 | 'application/octet-stream'を返却 |

### リトライ仕様

該当なし

## トランザクション仕様

トランザクション処理なし

## パフォーマンス要件

- 辞書検索によるO(1)のアクセス時間
- シングルトンパターンにより初期化は一度のみ

## セキュリティ考慮事項

- ファイル拡張子の偽装によるセキュリティリスクに注意
- コンテンツの実際の内容とMIMEタイプの不一致の可能性

## 備考

- 600種類以上のMIMEタイプが内部辞書に登録されている
- スレッドセーフな実装（TCriticalSectionによる排他制御）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

MIMEタイプ判定クラスの構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Mime.pas | `src/Horse.Mime.pas` | THorseMimeTypesクラスとTHorseMimeTypesExtクラスの定義 |

**読解のコツ**: 条件コンパイル(`{$IF DEFINED(FPC) OR (CompilerVersion <= 32.0)}`)によりTHorseMimeTypesExtが使用されるかどうかが決まる。

#### Step 2: パブリックAPIを理解する

外部から呼び出されるメソッドを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Mime.pas | `src/Horse.Mime.pas` | THorseMimeTypes.GetFileType/GetExtTypeメソッド |

**主要処理フロー**:
1. **22-26行目**: THorseMimeTypesクラス宣言 - GetFileType、GetExtTypeの静的メソッド
2. **66-77行目**: GetExtType実装 - コンパイラ分岐によりSystem.Net.MimeまたはTHorseMimeTypesExtを使用
3. **79-90行目**: GetFileType実装 - GetExtTypeを内部呼び出し

#### Step 3: 内部実装を理解する

拡張子-MIMEタイプマッピングの内部実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Mime.pas | `src/Horse.Mime.pas` | THorseMimeTypesExtクラスの実装 |

**主要処理フロー**:
- **44-62行目**: THorseMimeTypesExt宣言 - シングルトン、辞書、初期化
- **94-103行目**: コンストラクタ - TDictionary生成
- **105-114行目**: クラスデストラクタ - インスタンス解放
- **116-132行目**: GetDefaultメソッド - スレッドセーフなシングルトン取得
- **134-139行目**: NormalizeExtメソッド - 拡張子正規化
- **141-148行目**: GetExtType実装 - 辞書検索
- **150-153行目**: GetFileType実装 - GetExtType呼び出し
- **155行目以降**: InitializeFileType - 600種類以上のMIME登録

### プログラム呼び出し階層図

```
THorseMimeTypes.GetFileType(AFileName)
    │
    ├─ [Delphi >= 32.0]
    │      └─ System.Net.Mime.TMimeTypes.Default.GetFileInfo
    │
    └─ [FPC / Delphi < 32.0]
           └─ THorseMimeTypesExt.Default.GetFileType
                  ├─ GetDefault（シングルトン取得）
                  │      └─ InitializeFileType（初回のみ）
                  ├─ NormalizeExt（拡張子正規化）
                  └─ FFileType.TryGetValue（辞書検索）
```

### データフロー図

```
[入力]                    [処理]                              [出力]

'example.json'      ───▶  ExtractFileExt              ───▶  '.json'
                              │
                              ▼
                         NormalizeExt                 ───▶  'json'
                              │
                              ▼
                         FFileType検索               ───▶  'application/json'
                              │
                              ▼
[未知の拡張子]         ───▶  デフォルト返却           ───▶  'application/octet-stream'
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Mime.pas | `src/Horse.Mime.pas` | ソース | THorseMimeTypesクラス定義と実装 |
| Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | ソース | ファイル処理でMIMEタイプ取得に使用 |
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | ファイル送信時のContent-Type設定 |

### 主要なMIMEタイプマッピング例

| 拡張子 | MIMEタイプ |
|--------|-----------|
| json | application/json |
| xml | application/xml |
| html | text/html |
| css | text/css |
| js | application/javascript |
| png | image/png |
| jpg/jpeg | image/jpeg |
| gif | image/gif |
| pdf | application/pdf |
| zip | application/zip |
