# 機能設計書 46-ファイル処理

## 概要

本ドキュメントは、HorseフレームワークにおけるファイルのラッピングクラスTHorseCoreFileについて記述する。

### 本機能の処理概要

本機能は、ファイルシステム上のファイルをHTTPレスポンスとして送信するための抽象化レイヤーを提供し、ファイルの読み込み、MIMEタイプ判定、ストリーム管理を一元化する。

**業務上の目的・背景**：HTTPサーバーにおいて、静的ファイルの配信やファイルダウンロード機能は基本的な要件である。ファイルのオープン、Content-Type判定、ストリーム管理、リソース解放を個別に実装すると、コードの重複とエラーの原因になる。本機能により、ファイル処理をカプセル化し、一貫したファイル配信インターフェースを提供する。

**機能の利用シーン**：本機能は以下のような場面で使用される。
- THorseResponse.SendFileメソッドでファイルをインライン表示用に送信する際
- THorseResponse.Downloadメソッドでファイルをダウンロード用に送信する際
- カスタムファイル配信ロジックの実装時

**主要な処理内容**：
1. ファイルパスを受け取り、ファイルの存在を検証
2. THorseMimeTypesを使用してMIMEタイプを自動判定
3. TFileStreamを遅延生成してファイル内容を読み取り
4. ストリームのライフサイクル管理（自動解放オプション）

**関連システム・外部連携**：THorseMimeTypes（MIMEタイプ判定）、THorseResponse（レスポンス送信）と連携。

**権限による制御**：本機能に権限による制御はない。ファイルシステムのアクセス権限に依存。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に依存しないユーティリティ機能 |

## 機能種別

ファイル処理 / ユーティリティ

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AFileName | string | Yes | ファイルパス | 空文字でないこと、ファイルが存在すること |

### 入力データソース

ファイルシステム上のファイル

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ContentType | string | ファイルのMIMEタイプ |
| ContentStream | TStream | ファイル内容のストリーム |
| Size | Int64 | ファイルサイズ（バイト） |
| Name | string | ファイル名（パスなし） |

### 出力先

プロパティ/メソッドを通じてアクセス

## 処理フロー

### 処理シーケンス

```
1. THorseCoreFile.Create(AFileName) 呼び出し
   └─ ファイルパスが空でないことを検証
   └─ ファイルが存在することを検証
   └─ THorseMimeTypes.GetFileTypeでMIMEタイプ取得
   └─ ExtractFileNameでファイル名抽出
2. ContentStream プロパティアクセス（遅延生成）
   └─ TFileStreamを生成（fmOpenRead or fmShareDenyWrite）
3. Size プロパティアクセス
   └─ ContentStream.Size を返却
4. Destroy（解放時）
   └─ FreeContentStreamがTrueの場合、ストリーム解放
```

### フローチャート

```mermaid
flowchart TD
    A[Create AFileName] --> B{AFileName 空?}
    B -->|Yes| C[例外: Invalid FileName]
    B -->|No| D{ファイル存在?}
    D -->|No| E[例外: File not exist]
    D -->|Yes| F[MIMEタイプ取得]
    F --> G[ファイル名抽出]
    G --> H[インスタンス生成完了]
    H --> I{ContentStream アクセス?}
    I -->|Yes| J{ストリーム生成済み?}
    J -->|No| K[TFileStream生成]
    K --> L[ストリーム返却]
    J -->|Yes| L
    I -->|No| M{Destroy?}
    M -->|Yes| N{FreeContentStream?}
    N -->|Yes| O[ストリーム解放]
    N -->|No| P[解放なし]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 遅延生成 | ストリームはContentStreamアクセス時に初めて生成される | 常時 |
| BR-02 | 自動解放 | FreeContentStreamがTrue（デフォルト）の場合、Destroy時にストリーム解放 | オブジェクト解放時 |
| BR-03 | 共有モード | ファイルは読み取り専用・書き込み拒否モードでオープン | 常時 |
| BR-04 | 必須検証 | ファイルパスが空または存在しない場合は例外発生 | Create時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作なし

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | AFileNameが空文字 | 'Invalid FileName'メッセージで例外発生 |
| - | Exception | ファイルが存在しない | 'File not exist'メッセージで例外発生 |
| - | EFOpenError | ファイルアクセス権限なし | 呼び出し元で対処 |

### リトライ仕様

リトライ処理なし

## トランザクション仕様

トランザクション処理なし

## パフォーマンス要件

- ストリームの遅延生成により、不要なファイルオープンを回避
- 大容量ファイルはストリームで処理され、メモリ効率が良い

## セキュリティ考慮事項

- ディレクトリトラバーサル攻撃への対策（上位ディレクトリアクセス）
- 機密ファイルへのアクセス制限

## 備考

- FreeContentStreamプロパティをFalseに設定すると、呼び出し元でストリームのライフサイクルを管理できる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ファイルクラスの構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | THorseCoreFileクラスの定義 |

**読解のコツ**: プライベートフィールド（FFileName、FName、FFileStream、FFreeContentStream、FContentType）がクラスの状態を保持する。

#### Step 2: コンストラクタを理解する

オブジェクト生成時の処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | Createコンストラクタの実装 |

**主要処理フロー**:
1. **17-32行目**: クラス宣言 - プライベートフィールドとパブリックメソッド
2. **44-56行目**: コンストラクタ実装
   - **46-47行目**: 空文字チェック
   - **49-50行目**: ファイル存在チェック
   - **52行目**: ファイルパス保存
   - **53行目**: ファイル名抽出
   - **54行目**: FreeContentStreamデフォルト値（True）
   - **55行目**: MIMEタイプ取得

#### Step 3: ストリーム管理を理解する

遅延生成とリソース解放を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | ContentStream、Destroy実装 |

**主要処理フロー**:
- **58-63行目**: デストラクタ - FFreeContentStreamに基づくストリーム解放
- **70-75行目**: ContentStreamメソッド - 遅延生成パターン
- **77-80行目**: Sizeメソッド - ストリームサイズ返却

### プログラム呼び出し階層図

```
THorseCoreFile.Create(AFileName)
    │
    ├─ FileExists(AFileName)
    ├─ ExtractFileName(FFileName)
    └─ THorseMimeTypes.GetFileType(FFileName)

THorseCoreFile.ContentStream
    │
    └─ TFileStream.Create(FFileName, fmOpenRead or fmShareDenyWrite)

THorseCoreFile.Size
    │
    └─ ContentStream.Size

THorseCoreFile.Destroy
    │
    └─ [FFreeContentStream] FFileStream.Free
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ファイルパス         ───▶  THorseCoreFile.Create      ───▶  オブジェクト
'path/to/file.pdf'           │
                             ├─ ファイル存在確認
                             ├─ MIMEタイプ取得
                             └─ ファイル名抽出
                                      │
                                      ▼
                             ContentStream アクセス    ───▶  TStream
                                      │
                                      └─ TFileStream生成
                                      │
                                      ▼
                             Size アクセス            ───▶  Int64
                                      │
                                      ▼
                             Destroy                  ───▶  リソース解放
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.Files.pas | `src/Horse.Core.Files.pas` | ソース | THorseCoreFileクラス定義と実装 |
| Horse.Mime.pas | `src/Horse.Mime.pas` | ソース | MIMEタイプ判定（依存先） |
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | ファイル送信機能（利用元） |
