# 機能設計書 48-セッション登録

## 概要

本ドキュメントは、Horseフレームワークにおけるセッションインスタンスを登録するTHorseSessions.SetSessionメソッドについて記述する。

### 本機能の処理概要

本機能は、カスタムセッションオブジェクトをセッションマネージャーに登録し、リクエスト処理中にセッションデータへのアクセスを可能にする。

**業務上の目的・背景**：Webアプリケーションでは、リクエスト間でユーザー固有のデータを保持する必要がある（認証情報、ユーザー設定など）。本機能により、カスタムセッションクラスを定義し、型安全な方法でセッションデータを管理できる。依存性注入パターンと組み合わせることで、柔軟なセッション管理が実現できる。

**機能の利用シーン**：本機能は以下のような場面で使用される。
- 認証ミドルウェアでユーザーセッションを登録する際
- カスタムセッションストア（Redis、データベースなど）との連携
- リクエストスコープのオブジェクト管理

**主要な処理内容**：
1. セッションクラスタイプとインスタンスの整合性を検証
2. TObjectDictionaryに登録（既存の場合は上書き）
3. Fluentインターフェースによるメソッドチェーン対応

**関連システム・外部連携**：THorseRequest（セッションアクセス元）、カスタムセッションクラスと連携。

**権限による制御**：本機能に権限による制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に依存しないユーティリティ機能 |

## 機能種別

セッション管理 / データ管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ASessionClass | TSessionClass | Yes | 登録するセッションのクラス型 | AInstanceの型と一致 |
| AInstance | TSession | Yes | 登録するセッションインスタンス | ASessionClassの型と一致 |

### 入力データソース

アプリケーションコードからの直接呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | THorseSessions | 自身のインスタンス（Fluentインターフェース用） |

### 出力先

内部辞書（FSessions）に登録

## 処理フロー

### 処理シーケンス

```
1. SetSession(ASessionClass, AInstance) 呼び出し
2. 型整合性チェック
   └─ ASessionClass.InheritsFrom(AInstance.ClassType) を検証
   └─ 不一致の場合は例外発生
3. FSessions.AddOrSetValue で辞書に登録/更新
4. Self を返却（Fluentインターフェース）
```

### フローチャート

```mermaid
flowchart TD
    A[SetSession 呼び出し] --> B{型整合性チェック}
    B -->|不一致| C[例外発生]
    B -->|一致| D[AddOrSetValue で登録]
    D --> E[Self 返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 型整合性 | AInstanceはASessionClassを継承している必要がある | SetSession呼び出し時 |
| BR-02 | 上書き登録 | 同一クラスで再登録すると既存インスタンスは上書きされる | 既存セッション登録時 |
| BR-03 | 所有権管理 | 登録されたインスタンスはTHorseSessionsが所有・解放する | doOwnsValuesオプション |

### 計算ロジック

```pascal
function THorseSessions.SetSession(const ASessionClass: TSessionClass; const AInstance: TSession): THorseSessions;
begin
  Result := Self;
  if not ASessionClass.InheritsFrom(AInstance.ClassType) then
    raise Exception.CreateFmt('SessionClass differs from of instance[%s].', [AInstance.ClassType.ClassName]);
  FSessions.AddOrSetValue(ASessionClass, AInstance);
end;
```

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作なし

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | ASessionClassとAInstanceの型が不一致 | 'SessionClass differs from of instance[クラス名].'メッセージで例外発生 |

### リトライ仕様

リトライ処理なし

## トランザクション仕様

トランザクション処理なし

## パフォーマンス要件

- 辞書操作によるO(1)のアクセス時間

## セキュリティ考慮事項

- セッションに機密情報を保存する場合は、適切な暗号化・保護を実施
- セッション固定攻撃への対策

## 備考

- TSessionは基底クラスとして定義されており、アプリケーションでカスタムセッションクラスを継承して使用する
- doOwnsValuesオプションにより、THorseSessions解放時にセッションインスタンスも自動解放される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

セッション管理クラスの構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Session.pas | `src/Horse.Session.pas` | TSession、TSessionClass、THorseSessionsの定義 |

**読解のコツ**: TSessionClassは`class of TSession`として定義されており、クラス参照型（メタクラス）としてセッションクラスの型情報を保持する。

#### Step 2: コンストラクタを理解する

内部辞書の初期化を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Session.pas | `src/Horse.Session.pas` | THorseSessions.Createコンストラクタ |

**主要処理フロー**:
1. **17-18行目**: TSessionクラス定義（空の基底クラス）
2. **20行目**: TSessionClass型定義（クラス参照）
3. **22-35行目**: THorseSessionsクラス宣言
4. **46-49行目**: コンストラクタ - TObjectDictionary生成（doOwnsValues）

#### Step 3: SetSessionメソッドを理解する

セッション登録ロジックを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Session.pas | `src/Horse.Session.pas` | SetSessionメソッド実装 |

**主要処理フロー**:
- **67-73行目**: SetSession実装
  - **69行目**: Selfを返却（Fluentインターフェース）
  - **70-71行目**: 型整合性チェック
  - **72行目**: 辞書に登録

### プログラム呼び出し階層図

```
THorseSessions.SetSession(ASessionClass, AInstance)
    │
    ├─ 型整合性チェック
    │      └─ ASessionClass.InheritsFrom(AInstance.ClassType)
    │             └─ [不一致] Exception.CreateFmt
    │
    └─ FSessions.AddOrSetValue(ASessionClass, AInstance)
```

### データフロー図

```
[入力]                       [処理]                         [出力]

ASessionClass           ───▶  型整合性チェック           ───▶  [成功/例外]
AInstance                         │
                                  ▼
                         FSessions.AddOrSetValue        ───▶  辞書登録
                                  │
                                  ▼
                         Self 返却                     ───▶  THorseSessions
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Session.pas | `src/Horse.Session.pas` | ソース | THorseSessionsクラス定義と実装 |
| Horse.Request.pas | `src/Horse.Request.pas` | ソース | セッションアクセス（利用元） |

### クラス構造

```pascal
TSession = class
  // 基底セッションクラス（空）
end;

TSessionClass = class of TSession;
  // クラス参照型

THorseSessions = class
private
  FSessions: TObjectDictionary<TSessionClass, TSession>;
public
  function SetSession(const ASessionClass: TSessionClass; const AInstance: TSession): THorseSessions;
  function GetSession(const ASessionClass: TSessionClass): TSession;
  function TryGetSession<T: class>(out ASession: T): Boolean;
  function Contains(const ASessionClass: TSessionClass): Boolean;
  property Session[const ASessionClass: TSessionClass]: TSession read GetSession;
end;
```
