# 機能設計書 49-セッション取得

## 概要

本ドキュメントは、Horseフレームワークにおける登録済みセッションインスタンスを取得するTHorseSessions.Sessionプロパティについて記述する。

### 本機能の処理概要

本機能は、セッションマネージャーに登録されたカスタムセッションオブジェクトを取得し、リクエスト処理中にセッションデータへアクセスする手段を提供する。

**業務上の目的・背景**：ミドルウェアで登録されたセッションデータは、後続のルートハンドラーで参照される必要がある。本機能により、型安全な方法でセッションオブジェクトを取得し、認証情報やユーザー設定などのリクエストスコープデータにアクセスできる。

**機能の利用シーン**：本機能は以下のような場面で使用される。
- ルートハンドラーで認証済みユーザー情報を取得する際
- ミドルウェア間でデータを共有する際
- カスタムセッションクラスのメソッドを呼び出す際

**主要な処理内容**：
1. セッションクラスタイプをキーとして辞書を検索
2. 登録済みのセッションインスタンスを返却
3. 存在確認用のContainsメソッド、型安全なTryGetSessionジェネリックメソッドも提供

**関連システム・外部連携**：THorseRequest（セッションアクセスI/F）、カスタムセッションクラスと連携。

**権限による制御**：本機能に権限による制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に依存しないユーティリティ機能 |

## 機能種別

セッション管理 / データ取得

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ASessionClass | TSessionClass | Yes | 取得するセッションのクラス型 | 登録済みであること |

### 入力データソース

内部辞書（FSessions）からの取得

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | TSession | 登録されたセッションインスタンス |
| Result (TryGetSession) | Boolean | セッション取得成功/失敗 |

### 出力先

メソッド戻り値として返却

## 処理フロー

### 処理シーケンス

```
1. Session[ASessionClass] プロパティアクセス
   └─ GetSession(ASessionClass) 呼び出し
2. FSessions.Items[ASessionClass] で辞書検索
3. 登録済みのセッションインスタンスを返却
   └─ 未登録の場合は例外発生（辞書の動作）
```

### フローチャート

```mermaid
flowchart TD
    A[Session プロパティアクセス] --> B[GetSession 呼び出し]
    B --> C[FSessions.Items アクセス]
    C --> D{キー存在?}
    D -->|Yes| E[セッションインスタンス返却]
    D -->|No| F[例外発生]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 直接アクセス | 未登録のセッションクラスにアクセスすると例外が発生する | Sessionプロパティ使用時 |
| BR-02 | 安全なアクセス | TryGetSessionを使用すると例外なしで存在確認できる | 安全な取得が必要な場合 |
| BR-03 | 存在確認 | Containsメソッドで事前に登録状態を確認できる | 条件分岐が必要な場合 |

### 計算ロジック

```pascal
function THorseSessions.GetSession(const ASessionClass: TSessionClass): TSession;
begin
  Result := FSessions.Items[ASessionClass];
end;

function THorseSessions.Contains(const ASessionClass: TSessionClass): Boolean;
begin
  Result := FSessions.ContainsKey(ASessionClass);
end;

function THorseSessions.TryGetSession<T>(out ASession: T): Boolean;
begin
  Result := FSessions.TryGetValue(TSessionClass(T), TSession(ASession));
end;
```

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作なし

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | EListError | 未登録のセッションクラスにアクセス | ContainsまたはTryGetSessionで事前確認 |

### リトライ仕様

リトライ処理なし

## トランザクション仕様

トランザクション処理なし

## パフォーマンス要件

- 辞書検索によるO(1)のアクセス時間

## セキュリティ考慮事項

- セッションデータの不正アクセス防止
- セッションハイジャック対策

## 備考

- TryGetSessionはジェネリックメソッドとして実装されており、型キャストなしで特定のセッションクラスを取得できる
- Objectプロパティも提供されており、TObject型としてセッションを取得可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

セッション管理クラスの構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Session.pas | `src/Horse.Session.pas` | THorseSessionsの取得関連メソッド |

**読解のコツ**: プロパティ（Session）とメソッド（GetSession、Contains、TryGetSession）の使い分けを理解する。

#### Step 2: 取得メソッドを理解する

各取得メソッドの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Session.pas | `src/Horse.Session.pas` | GetSession、GetObject、Contains、TryGetSessionメソッド |

**主要処理フロー**:
1. **25-26行目**: GetSessionとGetObjectメソッド宣言
2. **28行目**: TryGetSessionジェネリックメソッド宣言
3. **29行目**: Containsメソッド宣言
4. **31-32行目**: SessionとObjectプロパティ宣言
5. **57-60行目**: GetObjectメソッド実装
6. **62-65行目**: GetSessionメソッド実装
7. **75-78行目**: Containsメソッド実装
8. **80-83行目**: TryGetSessionメソッド実装

### プログラム呼び出し階層図

```
THorseSessions.Session[ASessionClass]
    │
    └─ GetSession(ASessionClass)
           │
           └─ FSessions.Items[ASessionClass]
                  │
                  └─ TSession インスタンス

THorseSessions.Contains(ASessionClass)
    │
    └─ FSessions.ContainsKey(ASessionClass)
           │
           └─ Boolean

THorseSessions.TryGetSession<T>(out ASession)
    │
    └─ FSessions.TryGetValue(TSessionClass(T), TSession(ASession))
           │
           └─ Boolean
```

### データフロー図

```
[入力]                       [処理]                         [出力]

ASessionClass           ───▶  FSessions.Items            ───▶  TSession
(TSessionClass)                   │
                                  ▼
[Contains]              ───▶  FSessions.ContainsKey      ───▶  Boolean
                                  │
                                  ▼
[TryGetSession<T>]      ───▶  FSessions.TryGetValue      ───▶  Boolean + T
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Session.pas | `src/Horse.Session.pas` | ソース | THorseSessionsクラス定義と実装 |
| Horse.Request.pas | `src/Horse.Request.pas` | ソース | セッションアクセス（利用元） |

### メソッド一覧

| メソッド/プロパティ | 戻り値 | 説明 | 例外 |
|-------------------|--------|------|------|
| Session[ASessionClass] | TSession | インデックスプロパティでセッション取得 | 未登録時EListError |
| Object[ASessionClass] | TObject | TObject型でセッション取得 | 未登録時EListError |
| GetSession(ASessionClass) | TSession | メソッドでセッション取得 | 未登録時EListError |
| GetObject(ASessionClass) | TObject | TObject型でセッション取得 | 未登録時EListError |
| Contains(ASessionClass) | Boolean | セッション登録確認 | なし |
| TryGetSession<T>(out ASession) | Boolean | 型安全なセッション取得 | なし |
