# 機能設計書 5-PATCH リクエスト処理

## 概要

本ドキュメントは、Horseフレームワークにおける HTTP PATCH リクエストを処理するルート登録機能の設計を記述する。

### 本機能の処理概要

HTTP PATCHリクエストを受け付けるエンドポイントをルーティングテーブルに登録し、リクエスト受信時に対応するコールバック関数を実行する機能を提供する。

**業務上の目的・背景**：WebアプリケーションやRESTful APIにおいて、PATCHリクエストはリソースの部分的な更新に使用されるHTTPメソッドである。PUTがリソース全体の置換を行うのに対し、PATCHは指定されたフィールドのみを更新する。PATCH リクエスト処理機能は、クライアントからの部分更新要求を受け付け、サーバー側で既存データの一部のみを変更する操作を行うためのルーティング基盤を提供する。これにより、帯域幅の節約と効率的なデータ更新が可能となる。

**機能の利用シーン**：
- ユーザープロファイルの一部更新API（例: `/users/:id` でメールアドレスのみ変更）
- リソースのステータス変更（例: `/orders/:id/status`）
- 設定の一部変更（例: `/settings`）
- ドキュメントの特定フィールド更新

**主要な処理内容**：
1. `THorseCore.Patch` メソッドを呼び出し、パスとコールバック関数を指定
2. パスの正規化（先頭・末尾のスラッシュ処理）
3. 事前登録されたコールバック（AddCallback）があれば先に登録
4. `THorseRouterTree.RegisterRoute` でルーティングテーブルに PATCH ルートを追加
5. リクエスト受信時、パスマッチングを行いコールバックを実行

**関連システム・外部連携**：本機能は単独で動作するルーティング機能であり、外部システムとの直接的な連携はない。ただし、登録されたコールバック内でデータベースの部分更新や外部APIへの変更通知を行うことが一般的である。

**権限による制御**：本機能自体には権限制御は含まれない。権限制御はミドルウェア機能（機能No.8）と組み合わせて実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングにPATCHリクエスト処理に直接関連する画面は定義されていない |

## 機能種別

HTTPルーティング / リクエスト処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| APath | string | Yes | ルーティングパス（例: '/users/:id', '/settings'） | 空文字不可、先頭・末尾のスラッシュは自動正規化 |
| ACallback | THorseCallback | Yes | リクエスト処理を行うコールバック関数 | nil不可 |

### コールバック型のバリエーション

| 型名 | シグネチャ | 説明 |
|-----|-----------|------|
| THorseCallback | `procedure(AReq: THorseRequest; ARes: THorseResponse; ANext: TNextProc)` | 完全な形式（Next呼び出し可能） |
| THorseCallbackRequestResponse | `procedure(AReq: THorseRequest; ARes: THorseResponse)` | リクエストとレスポンスを使用 |
| THorseCallbackRequest | `procedure(AReq: THorseRequest)` | リクエストのみ使用 |
| THorseCallbackResponse | `procedure(ARes: THorseResponse)` | レスポンスのみ使用（Delphi専用） |

### 入力データソース

アプリケーション起動時のルート登録コード（ソースコード内で静的に定義）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | THorseCore | メソッドチェーン用のTHorseCoreインスタンス |

### 出力先

- 内部データ構造: `THorseRouterTree.FCallBack` ディクショナリ（TMethodType.mtPatch キーで格納）
- 内部データ構造: `THorseRouterTree.FRoute` ディクショナリ（パスセグメントをキーとしたツリー構造）

## 処理フロー

### 処理シーケンス

```
1. THorseCore.Patch(APath, ACallback) 呼び出し
   └─ 複数のオーバーロードメソッドが存在し、適切な形式に変換

2. GetCallback によるコールバック正規化
   └─ THorseCallbackRequestResponse → THorseCallback への変換
   └─ THorseCallbackRequest → THorseCallback への変換（Status=NoContent設定）

3. RegisterCallbacksRoute(mtPatch, APath) 実行
   └─ AddCallback で事前登録されたコールバックを同じパスに登録

4. RegisterRoute(mtPatch, APath, ACallback) 実行
   └─ TrimPath でパス正規化（'/' + APath.Trim(['/'])）
   └─ THorseRouterTree.RegisterRoute 呼び出し

5. THorseRouterTree.RegisterRoute 内部処理
   └─ GetQueuePath でパスをキュー化
   └─ RegisterInternal でツリー構造に登録
```

### フローチャート

```mermaid
flowchart TD
    A[THorseCore.Patch 呼び出し] --> B{コールバック型チェック}
    B -->|THorseCallbackRequestResponse| C[GetCallback で変換]
    B -->|THorseCallbackRequest| D[GetCallback で変換<br/>Status=NoContent設定]
    B -->|THorseCallback| E[そのまま使用]
    C --> F[RegisterCallbacksRoute]
    D --> F
    E --> F
    F --> G{事前登録コールバックあり?}
    G -->|Yes| H[事前コールバックを先に登録]
    G -->|No| I[RegisterRoute]
    H --> I
    I --> J[TrimPath でパス正規化]
    J --> K[THorseRouterTree.RegisterRoute]
    K --> L[GetQueuePath でパスをキュー化]
    L --> M[RegisterInternal でツリーに登録]
    M --> N[終了: THorseCore を返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | パス正規化 | パスは先頭にスラッシュを付与し、末尾のスラッシュは除去 | 全てのルート登録時 |
| BR-02 | パスパラメータ | `:param` 形式でパスパラメータを定義可能 | パス内に`:` で始まるセグメントがある場合 |
| BR-03 | 正規表現ルート | `(regex)` 形式で正規表現パターンを定義可能 | パス内に`()`で囲まれたセグメントがある場合 |
| BR-04 | コールバック蓄積 | 同一パス・メソッドに複数のコールバック登録可能 | AddCallback使用時 |
| BR-05 | 部分更新 | PATCHは部分更新を意味し、指定されたフィールドのみ変更 | PATCH API設計時 |
| BR-06 | コンパイラ制約 | Delphi XE6以前では条件コンパイルにより利用不可 | CompilerVersion <= 27.0 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | THorseCoreインスタンスが既に作成済みの状態でCreate呼び出し | シングルトンパターンのため、GetInstanceを使用 |
| - | コンパイルエラー | Delphi XE6以前でPatchメソッドを使用 | コンパイラバージョンを確認、または別の方法で実装 |

### リトライ仕様

リトライは不要（同期的なルート登録処理）

## トランザクション仕様

トランザクション管理は不要（メモリ上のデータ構造操作のみ）

## パフォーマンス要件

- ルート登録はアプリケーション起動時に1回のみ実行されるため、厳密な性能要件はない
- ルートマッチングはO(n)（nはパスセグメント数）で完了

## セキュリティ考慮事項

- PATCH リクエストはリクエストボディでデータを送信するため、URLにデータが露出しない
- 認証・認可はミドルウェアで実装すること
- 更新対象フィールドの検証を行い、許可されていないフィールドの更新を防ぐこと
- 楽観的ロック（ETag、If-Match ヘッダー）の使用を検討すること

## 備考

- PATCHメソッドはDelphi XE7（CompilerVersion > 27.0）以降およびFree Pascal環境で利用可能
- `{$IF (DEFINED(FPC) or (CompilerVersion > 27.0))}` 条件コンパイルにより制御されている
- DELETEメソッドと同じ条件コンパイルブロック内で定義されている
- PUTとの違い：PUTは完全な置換、PATCHは部分的な更新

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

コールバック型の定義を理解することで、ルート登録の引数パターンを把握できる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Callback.pas | `src/Horse.Callback.pas` | コールバック型（THorseCallback等）の定義を確認 |
| 1-2 | Horse.Commons.pas | `src/Horse.Commons.pas` | TMethodType列挙型でHTTPメソッド種別を確認（mtPatch） |

**読解のコツ**: TMethodType列挙型には mtPatch が含まれており、これがPATCHメソッドを識別するキーとなる。

#### Step 2: エントリーポイントを理解する

`THorseCore.Patch` メソッドがエントリーポイントとなる。条件コンパイルに注意。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.pas | `src/Horse.Core.pas` | THorseCoreクラスのPatchメソッド群を確認 |

**主要処理フロー**:
1. **96-103行目**: Patch メソッドのオーバーロード宣言（4種類）※条件コンパイル内
2. **312-334行目**: Patch メソッド実装群 ※`{$IF (DEFINED(FPC) or (CompilerVersion > 27.0))}`内
3. **329-333行目**: 基本形式の Patch メソッド実装（THorseCallback型）

#### Step 3: 条件コンパイルを理解する

Patch/Deleteメソッドは古いDelphiバージョンでは利用不可。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.pas | `src/Horse.Core.pas` | 条件コンパイル `{$IF (DEFINED(FPC) or (CompilerVersion > 27.0))}` |

**読解のコツ**:
- `CompilerVersion > 27.0` はDelphi XE7以降を意味する
- FPC（Free Pascal Compiler）環境では常に利用可能
- DELETEメソッドと同じ条件コンパイルブロック

#### Step 4: ルート登録処理を理解する

内部的なルート登録の流れを確認する。他のHTTPメソッドと同様。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.pas | `src/Horse.Core.pas` | RegisterRoute/RegisterCallbacksRoute メソッド |
| 4-2 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | THorseRouterTree.RegisterRoute/RegisterInternal |

**主要処理フロー**:
- **186-193行目（Horse.Core.pas）**: RegisterCallbacksRoute で事前コールバック登録
- **205-212行目（Horse.Core.pas）**: RegisterRoute でルーティングテーブルに登録
- **71-81行目（Horse.Core.RouterTree.pas）**: THorseRouterTree.RegisterRoute 実装

### プログラム呼び出し階層図

```
THorseCore.Patch(APath, ACallback)
    │
    ├─ GetCallback(ACallback)  ※型変換が必要な場合
    │      └─ 無名メソッドでラップしてTHorseCallback形式に統一
    │
    ├─ RegisterCallbacksRoute(mtPatch, APath)
    │      └─ GetCallbacks() で事前登録コールバック取得
    │      └─ RegisterRoute(mtPatch, APath, LCallback) を各コールバックに対して実行
    │
    └─ RegisterRoute(mtPatch, APath, ACallback)
           └─ GetInstance() でシングルトンインスタンス取得
           └─ TrimPath(APath) でパス正規化
           └─ THorseRouterTree.RegisterRoute(AHTTPType, APath, ACallback)
                  └─ GetQueuePath(APath) でパスをセグメント化
                  └─ RegisterInternal(AHTTPType, APath, ACallback)
                         └─ ForcePath() で中間ノード作成
                         └─ FCallBack.Add() でコールバック登録
```

### データフロー図

```
[入力]                      [処理]                           [出力]

パス文字列 ─────────────────┐
("/users/:id")              │
                            ▼
コールバック関数 ──────► THorseCore.Patch ──► TrimPath ──► "/' + path.Trim(['/'])"
                            │                               │
                            ▼                               ▼
                    GetCallback         THorseRouterTree.RegisterRoute
                    (型変換)                    │
                            │                   ▼
                            │           GetQueuePath
                            │           (["", "users", ":id"])
                            │                   │
                            ▼                   ▼
               THorseCallback形式 ──────► RegisterInternal ──► FCallBack[mtPatch]
                                                               FRoute["users"]
                                                                 └─FRoute[":id"]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.pas | `src/Horse.Core.pas` | ソース | THorseCoreクラス - Patchメソッド等HTTPルーティングのエントリーポイント |
| Horse.Callback.pas | `src/Horse.Callback.pas` | ソース | コールバック型定義（THorseCallback等） |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | TMethodType列挙型（mtPatch）、THTTPStatus等の共通定義 |
| Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ソース | THorseRouterTreeクラス - ルーティングテーブルのツリー構造管理 |
| Horse.Request.pas | `src/Horse.Request.pas` | ソース | THorseRequestクラス - リクエスト情報のラッパー（Bodyメソッドでリクエストボディ取得） |
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponseクラス - レスポンス処理のラッパー |
