# 機能設計書 6-HEAD リクエスト処理

## 概要

本ドキュメントは、Horseフレームワークにおける HTTP HEAD リクエストを処理するルート登録機能の設計を記述する。

### 本機能の処理概要

HTTP HEADリクエストを受け付けるエンドポイントをルーティングテーブルに登録し、リクエスト受信時に対応するコールバック関数を実行する機能を提供する。

**業務上の目的・背景**：WebアプリケーションやRESTful APIにおいて、HEADリクエストはGETリクエストと同様のレスポンスヘッダーを返すが、レスポンスボディは含まない。HEAD リクエスト処理機能は、リソースの存在確認やメタデータの取得、ファイルサイズの事前確認など、ボディを必要としない軽量な確認操作を行うためのルーティング基盤を提供する。これにより、帯域幅の節約やキャッシュの検証が効率的に行える。

**機能の利用シーン**：
- リソースの存在確認（例: `/users/:id`）
- ファイルサイズの事前確認（ダウンロード前）
- キャッシュの有効性検証（ETag、Last-Modifiedの確認）
- サーバーの疎通確認
- Content-Typeの事前確認

**主要な処理内容**：
1. `THorseCore.Head` メソッドを呼び出し、パスとコールバック関数を指定
2. パスの正規化（先頭・末尾のスラッシュ処理）
3. 事前登録されたコールバック（AddCallback）があれば先に登録
4. `THorseRouterTree.RegisterRoute` でルーティングテーブルに HEAD ルートを追加
5. リクエスト受信時、パスマッチングを行いコールバックを実行

**関連システム・外部連携**：本機能は単独で動作するルーティング機能であり、外部システムとの直接的な連携はない。ただし、登録されたコールバック内でデータベースやファイルシステムへの問い合わせを行い、ヘッダー情報を返すことが一般的である。

**権限による制御**：本機能自体には権限制御は含まれない。権限制御はミドルウェア機能（機能No.8）と組み合わせて実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングにHEADリクエスト処理に直接関連する画面は定義されていない |

## 機能種別

HTTPルーティング / リクエスト処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| APath | string | Yes | ルーティングパス（例: '/files/:id', '/resources/:name'） | 空文字不可、先頭・末尾のスラッシュは自動正規化 |
| ACallback | THorseCallback | Yes | リクエスト処理を行うコールバック関数 | nil不可 |

### コールバック型のバリエーション

| 型名 | シグネチャ | 説明 |
|-----|-----------|------|
| THorseCallback | `procedure(AReq: THorseRequest; ARes: THorseResponse; ANext: TNextProc)` | 完全な形式（Next呼び出し可能） |
| THorseCallbackRequestResponse | `procedure(AReq: THorseRequest; ARes: THorseResponse)` | リクエストとレスポンスを使用 |
| THorseCallbackRequest | `procedure(AReq: THorseRequest)` | リクエストのみ使用 |
| THorseCallbackResponse | `procedure(ARes: THorseResponse)` | レスポンスのみ使用（Delphi専用） |

### 入力データソース

アプリケーション起動時のルート登録コード（ソースコード内で静的に定義）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | THorseCore | メソッドチェーン用のTHorseCoreインスタンス |

### 出力先

- 内部データ構造: `THorseRouterTree.FCallBack` ディクショナリ（TMethodType.mtHead キーで格納）
- 内部データ構造: `THorseRouterTree.FRoute` ディクショナリ（パスセグメントをキーとしたツリー構造）

## 処理フロー

### 処理シーケンス

```
1. THorseCore.Head(APath, ACallback) 呼び出し
   └─ 複数のオーバーロードメソッドが存在し、適切な形式に変換

2. GetCallback によるコールバック正規化
   └─ THorseCallbackRequestResponse → THorseCallback への変換
   └─ THorseCallbackRequest → THorseCallback への変換（Status=NoContent設定）

3. RegisterCallbacksRoute(mtHead, APath) 実行
   └─ AddCallback で事前登録されたコールバックを同じパスに登録

4. RegisterRoute(mtHead, APath, ACallback) 実行
   └─ TrimPath でパス正規化（'/' + APath.Trim(['/'])）
   └─ THorseRouterTree.RegisterRoute 呼び出し

5. THorseRouterTree.RegisterRoute 内部処理
   └─ GetQueuePath でパスをキュー化
   └─ RegisterInternal でツリー構造に登録
```

### フローチャート

```mermaid
flowchart TD
    A[THorseCore.Head 呼び出し] --> B{コールバック型チェック}
    B -->|THorseCallbackRequestResponse| C[GetCallback で変換]
    B -->|THorseCallbackRequest| D[GetCallback で変換<br/>Status=NoContent設定]
    B -->|THorseCallback| E[そのまま使用]
    C --> F[RegisterCallbacksRoute]
    D --> F
    E --> F
    F --> G{事前登録コールバックあり?}
    G -->|Yes| H[事前コールバックを先に登録]
    G -->|No| I[RegisterRoute]
    H --> I
    I --> J[TrimPath でパス正規化]
    J --> K[THorseRouterTree.RegisterRoute]
    K --> L[GetQueuePath でパスをキュー化]
    L --> M[RegisterInternal でツリーに登録]
    M --> N[終了: THorseCore を返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | パス正規化 | パスは先頭にスラッシュを付与し、末尾のスラッシュは除去 | 全てのルート登録時 |
| BR-02 | パスパラメータ | `:param` 形式でパスパラメータを定義可能 | パス内に`:` で始まるセグメントがある場合 |
| BR-03 | 正規表現ルート | `(regex)` 形式で正規表現パターンを定義可能 | パス内に`()`で囲まれたセグメントがある場合 |
| BR-04 | コールバック蓄積 | 同一パス・メソッドに複数のコールバック登録可能 | AddCallback使用時 |
| BR-05 | ボディなし | HEADリクエストのレスポンスにはボディを含めない | HEAD API実装時 |
| BR-06 | GETと同じヘッダー | HEADはGETと同じレスポンスヘッダーを返すべき | HEAD API設計時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | THorseCoreインスタンスが既に作成済みの状態でCreate呼び出し | シングルトンパターンのため、GetInstanceを使用 |

### リトライ仕様

リトライは不要（同期的なルート登録処理）

## トランザクション仕様

トランザクション管理は不要（メモリ上のデータ構造操作のみ）

## パフォーマンス要件

- ルート登録はアプリケーション起動時に1回のみ実行されるため、厳密な性能要件はない
- ルートマッチングはO(n)（nはパスセグメント数）で完了
- HEADリクエストはボディを返さないため、GETより軽量

## セキュリティ考慮事項

- HEAD リクエストはリソースのメタデータのみを返すため、機密データの漏洩リスクは低い
- ただし、ヘッダー情報にも機密情報が含まれる可能性があるため注意が必要
- 認証・認可はミドルウェアで実装すること

## 備考

- HEADメソッドはすべてのDelphiバージョンおよびFree Pascal環境で利用可能
- GET/POST/PUTと同様の実装パターンであり、HTTPメソッドタイプ（mtHead）のみが異なる
- 通常はGETと同じルートに対して実装し、GETと同じレスポンスヘッダーを返すことが推奨される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

コールバック型の定義を理解することで、ルート登録の引数パターンを把握できる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Callback.pas | `src/Horse.Callback.pas` | コールバック型（THorseCallback等）の定義を確認 |
| 1-2 | Horse.Commons.pas | `src/Horse.Commons.pas` | TMethodType列挙型でHTTPメソッド種別を確認（mtHead） |

**読解のコツ**: TMethodType列挙型には mtHead が含まれており、これがHEADメソッドを識別するキーとなる。

#### Step 2: エントリーポイントを理解する

`THorseCore.Head` メソッドがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.pas | `src/Horse.Core.pas` | THorseCoreクラスのHeadメソッド群を確認 |

**主要処理フロー**:
1. **104-109行目**: Head メソッドのオーバーロード宣言（4種類）
2. **294-298行目**: 基本形式の Head メソッド実装（THorseCallback型）
3. **405-408行目**: THorseCallbackRequestResponse型のHead実装
4. **410-413行目**: THorseCallbackRequest型のHead実装
5. **415-420行目**: THorseCallbackResponse型のHead実装（Delphi専用）

#### Step 3: コールバック変換処理を理解する

異なるコールバック型を統一形式に変換する処理を確認する。他のHTTPメソッドと同じGetCallbackメソッドを使用する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.pas | `src/Horse.Core.pas` | GetCallback メソッドの変換ロジック |

**主要処理フロー**:
- **427-438行目**: THorseCallbackRequestResponse → THorseCallback 変換
- **440-449行目**: THorseCallbackResponse → THorseCallback 変換（Delphi専用）
- **451-463行目**: THorseCallbackRequest → THorseCallback 変換

#### Step 4: ルート登録処理を理解する

内部的なルート登録の流れを確認する。他のHTTPメソッドと同様。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.pas | `src/Horse.Core.pas` | RegisterRoute/RegisterCallbacksRoute メソッド |
| 4-2 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | THorseRouterTree.RegisterRoute/RegisterInternal |

**主要処理フロー**:
- **186-193行目（Horse.Core.pas）**: RegisterCallbacksRoute で事前コールバック登録
- **205-212行目（Horse.Core.pas）**: RegisterRoute でルーティングテーブルに登録
- **71-81行目（Horse.Core.RouterTree.pas）**: THorseRouterTree.RegisterRoute 実装

### プログラム呼び出し階層図

```
THorseCore.Head(APath, ACallback)
    │
    ├─ GetCallback(ACallback)  ※型変換が必要な場合
    │      └─ 無名メソッドでラップしてTHorseCallback形式に統一
    │
    ├─ RegisterCallbacksRoute(mtHead, APath)
    │      └─ GetCallbacks() で事前登録コールバック取得
    │      └─ RegisterRoute(mtHead, APath, LCallback) を各コールバックに対して実行
    │
    └─ RegisterRoute(mtHead, APath, ACallback)
           └─ GetInstance() でシングルトンインスタンス取得
           └─ TrimPath(APath) でパス正規化
           └─ THorseRouterTree.RegisterRoute(AHTTPType, APath, ACallback)
                  └─ GetQueuePath(APath) でパスをセグメント化
                  └─ RegisterInternal(AHTTPType, APath, ACallback)
                         └─ ForcePath() で中間ノード作成
                         └─ FCallBack.Add() でコールバック登録
```

### データフロー図

```
[入力]                      [処理]                           [出力]

パス文字列 ─────────────────┐
("/files/:id")              │
                            ▼
コールバック関数 ──────► THorseCore.Head ──► TrimPath ──► "/' + path.Trim(['/'])"
                            │                               │
                            ▼                               ▼
                    GetCallback         THorseRouterTree.RegisterRoute
                    (型変換)                    │
                            │                   ▼
                            │           GetQueuePath
                            │           (["", "files", ":id"])
                            │                   │
                            ▼                   ▼
               THorseCallback形式 ──────► RegisterInternal ──► FCallBack[mtHead]
                                                               FRoute["files"]
                                                                 └─FRoute[":id"]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.pas | `src/Horse.Core.pas` | ソース | THorseCoreクラス - Headメソッド等HTTPルーティングのエントリーポイント |
| Horse.Callback.pas | `src/Horse.Callback.pas` | ソース | コールバック型定義（THorseCallback等） |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | TMethodType列挙型（mtHead）、THTTPStatus等の共通定義 |
| Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ソース | THorseRouterTreeクラス - ルーティングテーブルのツリー構造管理 |
| Horse.Request.pas | `src/Horse.Request.pas` | ソース | THorseRequestクラス - リクエスト情報のラッパー |
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponseクラス - レスポンス処理のラッパー |
