# 機能設計書 7-ALL リクエスト処理

## 概要

本ドキュメントは、Horseフレームワークにおける全HTTPメソッドを処理するルート登録機能の設計を記述する。

### 本機能の処理概要

指定したパスに対して、全てのHTTPメソッド（GET、POST、PUT、DELETE、PATCH、HEAD）を一括で処理するエンドポイントをルーティングテーブルに登録し、リクエスト受信時に対応するコールバック関数を実行する機能を提供する。

**業務上の目的・背景**：Webアプリケーション開発において、同一エンドポイントで複数のHTTPメソッドに対して同じ処理を行いたい場合がある。ALL リクエスト処理機能は、1回の登録で全てのHTTPメソッドに対応するルートを作成でき、開発効率を向上させる。また、メソッドに依存しないCORS（Cross-Origin Resource Sharing）のプリフライトリクエスト処理や、全メソッド共通のロギング処理などに活用できる。

**機能の利用シーン**：
- CORSプリフライトリクエスト（OPTIONS含む）への対応
- 全HTTPメソッドで共通のエラーハンドリング
- APIのキャッチオール（404処理）
- デバッグ用の全リクエストログ取得
- プロキシエンドポイントの実装

**主要な処理内容**：
1. `THorseCore.All` メソッドを呼び出し、パスとコールバック関数を指定
2. TMethodType列挙型の全値（mtAny, mtGet, mtPut, mtPost, mtHead, mtDelete, mtPatch）に対してループ
3. 各HTTPメソッドに対して`RegisterCallbacksRoute`と`RegisterRoute`を実行
4. ルーティングテーブルに全HTTPメソッドのルートを追加
5. リクエスト受信時、パスマッチングを行いコールバックを実行

**関連システム・外部連携**：本機能は単独で動作するルーティング機能であり、外部システムとの直接的な連携はない。

**権限による制御**：本機能自体には権限制御は含まれない。権限制御はミドルウェア機能（機能No.8）と組み合わせて実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングにALLリクエスト処理に直接関連する画面は定義されていない |

## 機能種別

HTTPルーティング / リクエスト処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| APath | string | Yes | ルーティングパス（例: '/api/*', '/proxy/:path'） | 空文字不可、先頭・末尾のスラッシュは自動正規化 |
| ACallback | THorseCallback | Yes | リクエスト処理を行うコールバック関数 | nil不可 |

### コールバック型のバリエーション

| 型名 | シグネチャ | 説明 |
|-----|-----------|------|
| THorseCallback | `procedure(AReq: THorseRequest; ARes: THorseResponse; ANext: TNextProc)` | 完全な形式（Next呼び出し可能） |
| THorseCallbackRequestResponse | `procedure(AReq: THorseRequest; ARes: THorseResponse)` | リクエストとレスポンスを使用 |
| THorseCallbackRequest | `procedure(AReq: THorseRequest)` | リクエストのみ使用 |
| THorseCallbackResponse | `procedure(ARes: THorseResponse)` | レスポンスのみ使用（Delphi専用） |

### 入力データソース

アプリケーション起動時のルート登録コード（ソースコード内で静的に定義）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | THorseCore | メソッドチェーン用のTHorseCoreインスタンス |

### 出力先

- 内部データ構造: `THorseRouterTree.FCallBack` ディクショナリ（全TMethodTypeキーで格納）
- 内部データ構造: `THorseRouterTree.FRoute` ディクショナリ（パスセグメントをキーとしたツリー構造）

## 処理フロー

### 処理シーケンス

```
1. THorseCore.All(APath, ACallback) 呼び出し
   └─ 複数のオーバーロードメソッドが存在し、適切な形式に変換

2. TMethodType列挙型をLow(TMethodType)からHigh(TMethodType)までループ
   └─ mtAny, mtGet, mtPut, mtPost, mtHead, mtDelete, mtPatch

3. 各メソッドタイプに対して以下を実行:
   3-1. RegisterCallbacksRoute(LMethodType, APath)
        └─ AddCallback で事前登録されたコールバックを同じパスに登録
   3-2. RegisterRoute(LMethodType, APath, ACallback)
        └─ TrimPath でパス正規化
        └─ THorseRouterTree.RegisterRoute 呼び出し

4. THorseRouterTree.RegisterRoute 内部処理（各メソッドタイプごと）
   └─ GetQueuePath でパスをキュー化
   └─ RegisterInternal でツリー構造に登録
```

### フローチャート

```mermaid
flowchart TD
    A[THorseCore.All 呼び出し] --> B{コールバック型チェック}
    B -->|THorseCallbackRequestResponse| C[GetCallback で変換]
    B -->|THorseCallbackRequest| D[GetCallback で変換]
    B -->|THorseCallback| E[そのまま使用]
    C --> F[TMethodTypeループ開始]
    D --> F
    E --> F
    F --> G[RegisterCallbacksRoute]
    G --> H[RegisterRoute]
    H --> I{次のMethodType?}
    I -->|Yes| F
    I -->|No| J[終了: THorseCore を返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | パス正規化 | パスは先頭にスラッシュを付与し、末尾のスラッシュは除去 | 全てのルート登録時 |
| BR-02 | 全メソッド登録 | Low(TMethodType)からHigh(TMethodType)まで全てのメソッドを登録 | All メソッド呼び出し時 |
| BR-03 | パスパラメータ | `:param` 形式でパスパラメータを定義可能 | パス内に`:` で始まるセグメントがある場合 |
| BR-04 | 正規表現ルート | `(regex)` 形式で正規表現パターンを定義可能 | パス内に`()`で囲まれたセグメントがある場合 |
| BR-05 | コールバック蓄積 | 同一パス・メソッドに複数のコールバック登録可能 | AddCallback使用時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | THorseCoreインスタンスが既に作成済みの状態でCreate呼び出し | シングルトンパターンのため、GetInstanceを使用 |

### リトライ仕様

リトライは不要（同期的なルート登録処理）

## トランザクション仕様

トランザクション管理は不要（メモリ上のデータ構造操作のみ）

## パフォーマンス要件

- ルート登録はアプリケーション起動時に1回のみ実行されるため、厳密な性能要件はない
- Allメソッドは内部で複数回（全HTTPメソッド数分）RegisterRouteを呼び出すため、個別メソッド登録より若干時間がかかる
- ルートマッチングはO(n)（nはパスセグメント数）で完了

## セキュリティ考慮事項

- ALL でルートを登録すると、意図しないHTTPメソッドでもアクセス可能になる点に注意
- 認証・認可はミドルウェアで実装すること
- 必要なHTTPメソッドのみを個別に登録することを推奨

## 備考

- Allメソッドはすべてのバージョンで利用可能
- TMethodType列挙型には mtAny（任意のメソッド）も含まれる
- 個別のHTTPメソッド（Get, Post等）を後から登録すると、Allで登録したコールバックに追加される形となる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

TMethodType列挙型を理解することで、Allメソッドの動作を把握できる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Commons.pas | `src/Horse.Commons.pas` | TMethodType列挙型の全値を確認（mtAny, mtGet, mtPut, mtPost, mtHead, mtDelete, mtPatch） |
| 1-2 | Horse.Callback.pas | `src/Horse.Callback.pas` | コールバック型（THorseCallback等）の定義を確認 |

**読解のコツ**: TMethodType列挙型はFPC用に定義されているが、Delphi版ではWeb.HTTPAppのTMethodTypeを使用する。Allメソッドは両方に対応。

#### Step 2: エントリーポイントを理解する

`THorseCore.All` メソッドがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.pas | `src/Horse.Core.pas` | THorseCoreクラスのAllメソッド群を確認 |

**主要処理フロー**:
1. **78-83行目**: All メソッドのオーバーロード宣言（4種類）
2. **351-360行目**: 基本形式の All メソッド実装（THorseCallback型）- ループ処理が特徴
3. **362-365行目**: THorseCallbackRequestResponse型のAll実装
4. **367-370行目**: THorseCallbackRequest型のAll実装
5. **372-377行目**: THorseCallbackResponse型のAll実装（Delphi専用）

#### Step 3: ループ処理を理解する

Allメソッドの核心となるループ処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.pas | `src/Horse.Core.pas` | All メソッド内のfor文 |

**主要処理フロー**:
- **351-360行目**:
```pascal
class function THorseCore.All(const APath: string; const ACallback: THorseCallback): THorseCore;
var
  LMethodType: TMethodType;
begin
  for LMethodType := Low(TMethodType) to High(TMethodType) do
  begin
    Result := RegisterCallbacksRoute(LMethodType, APath);
    RegisterRoute(LMethodType, APath, ACallback);
  end;
end;
```

**読解のコツ**:
- `Low(TMethodType)` と `High(TMethodType)` で列挙型の全範囲をループ
- 各メソッドタイプに対してRegisterCallbacksRouteとRegisterRouteを呼び出し
- 結果として同じパスに全HTTPメソッドのルートが登録される

#### Step 4: ルート登録処理を理解する

内部的なルート登録の流れを確認する。他のHTTPメソッドと同様だが、複数回呼び出される。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.pas | `src/Horse.Core.pas` | RegisterRoute/RegisterCallbacksRoute メソッド |
| 4-2 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | THorseRouterTree.RegisterRoute/RegisterInternal |

**主要処理フロー**:
- **186-193行目（Horse.Core.pas）**: RegisterCallbacksRoute で事前コールバック登録
- **205-212行目（Horse.Core.pas）**: RegisterRoute でルーティングテーブルに登録
- **71-81行目（Horse.Core.RouterTree.pas）**: THorseRouterTree.RegisterRoute 実装

### プログラム呼び出し階層図

```
THorseCore.All(APath, ACallback)
    │
    ├─ GetCallback(ACallback)  ※型変換が必要な場合
    │      └─ 無名メソッドでラップしてTHorseCallback形式に統一
    │
    └─ for LMethodType := Low(TMethodType) to High(TMethodType) do
           │
           ├─ RegisterCallbacksRoute(LMethodType, APath)
           │      └─ GetCallbacks() で事前登録コールバック取得
           │      └─ RegisterRoute(LMethodType, APath, LCallback)
           │
           └─ RegisterRoute(LMethodType, APath, ACallback)
                  └─ GetInstance() でシングルトンインスタンス取得
                  └─ TrimPath(APath) でパス正規化
                  └─ THorseRouterTree.RegisterRoute(AHTTPType, APath, ACallback)
                         └─ GetQueuePath(APath) でパスをセグメント化
                         └─ RegisterInternal(AHTTPType, APath, ACallback)
                                └─ ForcePath() で中間ノード作成
                                └─ FCallBack.Add() でコールバック登録
```

### データフロー図

```
[入力]                      [処理]                           [出力]

パス文字列 ─────────────────┐
("/api/*")                  │
                            ▼
コールバック関数 ──────► THorseCore.All ──► TrimPath ──► "/' + path.Trim(['/'])"
                            │                               │
                            ▼                               ▼
                    GetCallback         for LMethodType ─────┐
                    (型変換)                                  │
                            │                                ▼
                            │           ┌─────────────────────────────────────┐
                            │           │ mtAny → RegisterRoute → FCallBack[mtAny]│
                            │           │ mtGet → RegisterRoute → FCallBack[mtGet]│
                            │           │ mtPut → RegisterRoute → FCallBack[mtPut]│
                            │           │ mtPost → RegisterRoute → FCallBack[mtPost]│
                            │           │ mtHead → RegisterRoute → FCallBack[mtHead]│
                            │           │ mtDelete → RegisterRoute → FCallBack[mtDelete]│
                            │           │ mtPatch → RegisterRoute → FCallBack[mtPatch]│
                            │           └─────────────────────────────────────┘
                            ▼                               │
               THorseCallback形式 ──────────────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.pas | `src/Horse.Core.pas` | ソース | THorseCoreクラス - Allメソッド等HTTPルーティングのエントリーポイント |
| Horse.Callback.pas | `src/Horse.Callback.pas` | ソース | コールバック型定義（THorseCallback等） |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | TMethodType列挙型（全HTTPメソッド）、THTTPStatus等の共通定義 |
| Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ソース | THorseRouterTreeクラス - ルーティングテーブルのツリー構造管理 |
| Horse.Request.pas | `src/Horse.Request.pas` | ソース | THorseRequestクラス - リクエスト情報のラッパー |
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponseクラス - レスポンス処理のラッパー |
