# 機能設計書 8-ミドルウェア登録

## 概要

本ドキュメントは、Horseフレームワークにおけるミドルウェアを登録する機能の設計を記述する。

### 本機能の処理概要

リクエスト処理の前後に実行されるミドルウェア（共通処理）をルーティングテーブルに登録し、全てのルートまたは特定パス配下のルートに対して横断的な処理を適用する機能を提供する。

**業務上の目的・背景**：Webアプリケーション開発において、認証・認可、ロギング、CORS処理、リクエストの前処理・後処理など、複数のエンドポイントで共通して実行したい処理がある。ミドルウェア登録機能は、これらの共通処理を一箇所に集約して登録することで、コードの重複を排除し、保守性を向上させる。Express.js等のNode.jsフレームワークと同様のパターンを採用している。

**機能の利用シーン**：
- JWT/OAuth認証の検証
- リクエストログの記録
- CORSヘッダーの付与
- レスポンスの圧縮処理
- 共通のエラーハンドリング
- リクエストボディのパース
- レート制限の実装

**主要な処理内容**：
1. `THorseCore.Use` メソッドを呼び出し、パス（省略可）とミドルウェアコールバックを指定
2. パスの正規化（先頭・末尾のスラッシュ処理）
3. `THorseRouterTree.RegisterMiddleware` でミドルウェアをルーティングテーブルに登録
4. リクエスト受信時、該当パスのミドルウェアがルートコールバックより先に実行される
5. Next()呼び出しで次のミドルウェアまたはルートコールバックに処理を移譲

**関連システム・外部連携**：本機能は単独で動作するルーティング機能であり、外部システムとの直接的な連携はない。ただし、登録されたミドルウェア内で認証サーバーや外部APIと連携することは一般的である。

**権限による制御**：ミドルウェア自体が権限制御の実装場所となる。認証・認可ロジックをミドルウェアとして実装することで、全ルートに対して一貫した権限チェックを適用できる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングにミドルウェア登録に直接関連する画面は定義されていない |

## 機能種別

HTTPルーティング / ミドルウェア処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| APath | string | No | ミドルウェアを適用するパス（省略時はルート'/'） | 先頭・末尾のスラッシュは自動正規化 |
| ACallback | THorseCallback | Yes | ミドルウェア処理を行うコールバック関数 | nil不可 |
| ACallbacks | array of THorseCallback | No | 複数のミドルウェアを一度に登録する場合 | - |

### Useメソッドのオーバーロード

| シグネチャ | 説明 |
|-----------|------|
| `Use(ACallback: THorseCallback)` | ルート全体にミドルウェアを登録 |
| `Use(APath: string; ACallback: THorseCallback)` | 特定パス配下にミドルウェアを登録 |
| `Use(ACallbacks: array of THorseCallback)` | 複数ミドルウェアをルート全体に登録 |
| `Use(APath: string; ACallbacks: array of THorseCallback)` | 複数ミドルウェアを特定パス配下に登録 |

### 入力データソース

アプリケーション起動時のミドルウェア登録コード（ソースコード内で静的に定義）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | THorseCore | メソッドチェーン用のTHorseCoreインスタンス |

### 出力先

- 内部データ構造: `THorseRouterTree.FMiddleware` リスト（ミドルウェアコールバックのリスト）

## 処理フロー

### 処理シーケンス

```
1. THorseCore.Use(APath, ACallback) 呼び出し
   └─ パス指定あり: TrimPath でパス正規化後、RegisterMiddleware 呼び出し
   └─ パス指定なし: ルート'/'に対して RegisterMiddleware 呼び出し

2. THorseRouterTree.RegisterMiddleware 内部処理
   └─ GetQueuePath でパスをキュー化
   └─ RegisterMiddlewareInternal で再帰的にツリーを辿る

3. RegisterMiddlewareInternal 処理
   └─ パスキューが空: FMiddleware リストにコールバック追加
   └─ パスキューに要素あり: ForcePath で中間ノード作成し再帰

4. リクエスト受信時の実行順序
   └─ ルートノードのミドルウェア実行
   └─ 中間ノードのミドルウェア実行
   └─ 最終ノードのミドルウェア実行
   └─ ルートコールバック実行
```

### フローチャート

```mermaid
flowchart TD
    A[THorseCore.Use 呼び出し] --> B{パス指定あり?}
    B -->|Yes| C[TrimPath でパス正規化]
    B -->|No| D[パス = '/']
    C --> E[THorseRouterTree.RegisterMiddleware]
    D --> E
    E --> F[GetQueuePath でパスをキュー化]
    F --> G[RegisterMiddlewareInternal]
    G --> H{パスキュー空?}
    H -->|Yes| I[FMiddleware.Add]
    H -->|No| J[ForcePath で中間ノード作成]
    J --> K[再帰呼び出し]
    K --> G
    I --> L[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | パス正規化 | パスは先頭にスラッシュを付与し、末尾のスラッシュは除去 | 全てのミドルウェア登録時 |
| BR-02 | 実行順序 | ミドルウェアは登録順に実行される | リクエスト処理時 |
| BR-03 | Next呼び出し | Next()を呼び出さないと後続処理が実行されない | ミドルウェア内 |
| BR-04 | パス継承 | 親パスのミドルウェアは子パスにも適用される | 階層的なパス構造 |
| BR-05 | 複数登録 | 同一パスに複数のミドルウェアを登録可能 | Use呼び出し毎に追加 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | THorseCoreインスタンスが既に作成済みの状態でCreate呼び出し | シングルトンパターンのため、GetInstanceを使用 |
| - | 処理中断 | ミドルウェア内でNext()を呼び出さない | Next()呼び出しを追加 |

### リトライ仕様

リトライは不要（同期的なミドルウェア登録処理）

## トランザクション仕様

トランザクション管理は不要（メモリ上のデータ構造操作のみ）

## パフォーマンス要件

- ミドルウェア登録はアプリケーション起動時に1回のみ実行されるため、厳密な性能要件はない
- ミドルウェアの実行はリクエスト毎に発生するため、重い処理は避けること

## セキュリティ考慮事項

- ミドルウェアは認証・認可の実装場所として最適
- グローバルミドルウェア（パス指定なし）は全リクエストに影響するため注意
- ミドルウェア内での例外は適切にハンドリングすること

## 備考

- Useメソッドはすべてのバージョンで利用可能
- Express.jsのミドルウェアパターンを参考に設計されている
- Next()を呼び出すことで処理チェーンを継続できる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ミドルウェアが格納されるデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | FMiddleware: TList<THorseCallback> の定義を確認 |
| 1-2 | Horse.Callback.pas | `src/Horse.Callback.pas` | THorseCallback型の定義を確認 |

**読解のコツ**: FMiddlewareはTList<THorseCallback>型で、複数のミドルウェアを順番に格納できる。

#### Step 2: エントリーポイントを理解する

`THorseCore.Use` メソッドがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.pas | `src/Horse.Core.pas` | THorseCoreクラスのUseメソッド群を確認 |

**主要処理フロー**:
1. **73-76行目**: Use メソッドのオーバーロード宣言（4種類）
2. **393-397行目**: パス指定なしの Use メソッド実装
3. **399-403行目**: パス指定ありの Use メソッド実装
4. **342-349行目**: 配列版の Use メソッド実装（ループで各コールバックを登録）
5. **384-391行目**: パス指定あり配列版の Use メソッド実装

#### Step 3: ミドルウェア登録処理を理解する

`THorseRouterTree.RegisterMiddleware` の処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | RegisterMiddleware/RegisterMiddlewareInternal メソッド |

**主要処理フロー**:
- **317-320行目**: RegisterMiddleware(AMiddleware) - パス指定なし版
- **322-332行目**: RegisterMiddleware(APath, AMiddleware) - パス指定あり版
- **334-341行目**: RegisterMiddlewareInternal - 再帰的なミドルウェア登録

#### Step 4: ミドルウェア実行フローを理解する

リクエスト処理時のミドルウェア実行順序を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ExecuteInternal メソッド |
| 4-2 | Horse.Core.RouterTree.NextCaller.pas | `src/Horse.Core.RouterTree.NextCaller.pas` | TNextCaller クラスの処理フロー |

**主要処理フロー**:
- **168-194行目（Horse.Core.RouterTree.pas）**: ExecuteInternal でTNextCallerを使用
- **183行目**: SetMiddleware(FMiddleware) でミドルウェアリストを設定
- TNextCaller内でミドルウェアを順番に実行し、Next()呼び出しで次の処理へ

### プログラム呼び出し階層図

```
THorseCore.Use(APath, ACallback)
    │
    ├─ GetInstance() でシングルトンインスタンス取得
    │
    └─ Routes.RegisterMiddleware(TrimPath(APath), ACallback)
           │
           ├─ GetQueuePath(APath) でパスをセグメント化
           │
           └─ RegisterMiddlewareInternal(APath, AMiddleware)
                  │
                  ├─ APath.Dequeue で先頭要素を取り出し
                  │
                  ├─ APath.Count = 0 の場合:
                  │      └─ FMiddleware.Add(AMiddleware)
                  │
                  └─ APath.Count > 0 の場合:
                         └─ ForcePath(APath.Peek).RegisterMiddlewareInternal(APath, AMiddleware)
```

### データフロー図

```
[登録時]
パス文字列 ─────────────────┐
("/api")                    │
                            ▼
コールバック関数 ──────► THorseCore.Use ──► TrimPath ──► "/api"
                            │                               │
                            ▼                               ▼
                    GetInstance      RegisterMiddleware ────────────┐
                            │                                       │
                            │                                       ▼
                            └───────────────────────► FMiddleware.Add(ACallback)
                                                      (THorseRouterTreeの該当ノード)

[実行時]
HTTPリクエスト ──────────────────┐
("/api/users")                   │
                                 ▼
THorseRouterTree.Execute ──► ExecuteInternal ──► TNextCaller
                                                     │
                                    ┌────────────────┴────────────────┐
                                    ▼                                 ▼
                              FMiddleware[0]                   FMiddleware[1]
                              (1番目MW)                        (2番目MW)
                                    │                                 │
                                    └─ Next() ────────────────────────┘
                                                                      │
                                                                      ▼
                                                              FCallBack[mtGet]
                                                              (ルートコールバック)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.pas | `src/Horse.Core.pas` | ソース | THorseCoreクラス - Useメソッドのエントリーポイント |
| Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ソース | THorseRouterTreeクラス - FMiddlewareリストの管理、RegisterMiddleware |
| Horse.Core.RouterTree.NextCaller.pas | `src/Horse.Core.RouterTree.NextCaller.pas` | ソース | TNextCallerクラス - ミドルウェアとルートコールバックの実行順序制御 |
| Horse.Callback.pas | `src/Horse.Callback.pas` | ソース | コールバック型定義（THorseCallback等） |
| Horse.Request.pas | `src/Horse.Request.pas` | ソース | THorseRequestクラス - リクエスト情報のラッパー |
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponseクラス - レスポンス処理のラッパー |
| Horse.Proc.pas | `src/Horse.Proc.pas` | ソース | TNextProc型の定義（Next関数の型） |
