# 機能設計書 9-コールバック追加

## 概要

本ドキュメントは、Horseフレームワークにおけるルート実行前に追加のコールバックを登録する機能の設計を記述する。

### 本機能の処理概要

ルートコールバックが実行される前に、追加のコールバック関数を一時的に蓄積し、次に登録されるルートに対して自動的に適用する機能を提供する。これにより、特定のルート群に対して共通の前処理を簡潔に定義できる。

**業務上の目的・背景**：Webアプリケーション開発において、複数の連続したルート定義に対して同じ前処理（認証チェック、入力バリデーション等）を適用したい場合がある。コールバック追加機能は、AddCallback/AddCallbacksメソッドでコールバックを蓄積し、次のルート登録時（Get, Post等）に自動的にそれらのコールバックを先に登録する。Use（ミドルウェア）とは異なり、特定のルートにのみ適用される一時的な仕組みである。

**機能の利用シーン**：
- 特定のルートグループに対する認証チェックの追加
- 入力バリデーションの前処理
- 特定エンドポイントへのアクセスログ記録
- 条件付きの前処理（特定ルートのみに適用）

**主要な処理内容**：
1. `THorseCore.AddCallback` または `AddCallbacks` メソッドを呼び出し、コールバック関数を指定
2. `FCallbacks` リストにコールバックを蓄積
3. 次のルート登録時（Get, Post等）に `RegisterCallbacksRoute` が呼び出される
4. `GetCallbacks` で蓄積されたコールバックを取得し、各々を `RegisterRoute` で登録
5. `FCallbacks` リストがクリアされる

**関連システム・外部連携**：本機能は単独で動作するルーティング機能であり、外部システムとの直接的な連携はない。

**権限による制御**：本機能自体には権限制御は含まれない。登録されるコールバック内で権限チェックを実装することは可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングにコールバック追加に直接関連する画面は定義されていない |

## 機能種別

HTTPルーティング / コールバック管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ACallback | THorseCallback | Yes | 追加するコールバック関数 | nil不可 |
| ACallbacks | TArray<THorseCallback> | No | 複数のコールバックを一度に追加する場合 | - |

### AddCallbackメソッドのオーバーロード

| シグネチャ | 説明 |
|-----------|------|
| `AddCallback(ACallback: THorseCallback)` | 単一のコールバックを追加 |
| `AddCallbacks(ACallbacks: TArray<THorseCallback>)` | 複数のコールバックを一度に追加 |

### 入力データソース

アプリケーション起動時のルート登録コード（ソースコード内で静的に定義）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Result | THorseCore | メソッドチェーン用のTHorseCoreインスタンス |

### 出力先

- 内部データ構造: `THorseCore.FCallbacks` リスト（TList<THorseCallback>型）
- 次のルート登録時: `THorseRouterTree.FCallBack` ディクショナリ

## 処理フロー

### 処理シーケンス

```
1. THorseCore.AddCallback(ACallback) 呼び出し
   └─ FCallbacksがnilの場合、TList<THorseCallback>.Createで初期化
   └─ FCallbacks.Add(ACallback) でコールバックを蓄積

2. THorseCore.AddCallbacks(ACallbacks) 呼び出し（複数追加の場合）
   └─ 各コールバックに対してAddCallbackを呼び出し

3. 次のルート登録（例: Get(APath, ACallback)）呼び出し
   └─ RegisterCallbacksRoute(mtGet, APath) が先に実行される
   └─ GetCallbacks でFCallbacksの内容を取得
   └─ 各蓄積コールバックに対してRegisterRouteを呼び出し
   └─ FCallbacks.Clear でリストをクリア

4. RegisterRoute(mtGet, APath, ACallback) でメインコールバックを登録
```

### フローチャート

```mermaid
flowchart TD
    A[AddCallback 呼び出し] --> B{FCallbacks初期化済み?}
    B -->|No| C[TList<THorseCallback>.Create]
    B -->|Yes| D[FCallbacks.Add]
    C --> D
    D --> E[終了: THorseCore を返却]

    F[Get/Post等 呼び出し] --> G[RegisterCallbacksRoute]
    G --> H[GetCallbacks]
    H --> I{蓄積コールバックあり?}
    I -->|Yes| J[各コールバックをRegisterRoute]
    I -->|No| K[RegisterRoute メインコールバック]
    J --> L[FCallbacks.Clear]
    L --> K
    K --> M[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 遅延初期化 | FCallbacksリストは初回使用時に初期化される | AddCallback初回呼び出し時 |
| BR-02 | 自動クリア | GetCallbacks呼び出し後にリストはクリアされる | ルート登録時 |
| BR-03 | 一時的蓄積 | 蓄積されたコールバックは次のルート登録にのみ適用 | ルート登録時 |
| BR-04 | 実行順序 | 蓄積コールバックはメインコールバックより先に登録・実行 | リクエスト処理時 |
| BR-05 | 複数適用 | AddCallbackを複数回呼ぶと、次のルートに全て適用される | 連続したAddCallback呼び出し |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception | THorseCoreインスタンスが既に作成済みの状態でCreate呼び出し | シングルトンパターンのため、GetInstanceを使用 |

### リトライ仕様

リトライは不要（同期的なコールバック蓄積処理）

## トランザクション仕様

トランザクション管理は不要（メモリ上のデータ構造操作のみ）

## パフォーマンス要件

- コールバック追加はアプリケーション起動時に実行されるため、厳密な性能要件はない
- リスト操作のみのため、高速に動作

## セキュリティ考慮事項

- AddCallbackは一時的な蓄積のため、意図しないルートに適用されないよう注意
- 認証・認可をAddCallbackで実装する場合、クリア後の挙動に注意

## 備考

- AddCallback/AddCallbacksは全てのバージョンで利用可能
- Use（ミドルウェア）との違い：
  - Use: 全ルートまたは特定パス配下に永続的に適用
  - AddCallback: 次に登録されるルートにのみ一時的に適用
- メソッドチェーンが可能: `THorse.AddCallback(cb1).Get('/path', cb2)`

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

コールバックが蓄積されるデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Core.pas | `src/Horse.Core.pas` | FCallbacks: TList<THorseCallback> クラス変数の定義を確認 |
| 1-2 | Horse.Callback.pas | `src/Horse.Callback.pas` | THorseCallback型の定義を確認 |

**読解のコツ**: FCallbacksはクラス変数（class var）として定義されており、THorseCoreの全インスタンスで共有される。

#### Step 2: エントリーポイントを理解する

`THorseCore.AddCallback` メソッドがエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.pas | `src/Horse.Core.pas` | THorseCoreクラスのAddCallback/AddCallbacksメソッドを確認 |

**主要処理フロー**:
1. **67-68行目**: AddCallback/AddCallbacks メソッドの宣言
2. **143-149行目**: AddCallback メソッド実装
3. **151-158行目**: AddCallbacks メソッド実装

#### Step 3: GetCallbacks処理を理解する

蓄積されたコールバックを取得してクリアする処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.pas | `src/Horse.Core.pas` | GetCallbacks メソッド |

**主要処理フロー**:
- **176-184行目**: GetCallbacks メソッド実装
```pascal
class function THorseCore.GetCallbacks: TArray<THorseCallback>;
begin
  Result := [];
  if Assigned(FCallbacks) then
  begin
    Result := FCallbacks.ToArray;
    FCallbacks.Clear;
  end;
end;
```

**読解のコツ**:
- `FCallbacks.ToArray` で配列に変換
- `FCallbacks.Clear` でリストをクリア（次のルート登録には影響しない）

#### Step 4: RegisterCallbacksRoute処理を理解する

ルート登録時に蓄積コールバックを適用する処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Horse.Core.pas | `src/Horse.Core.pas` | RegisterCallbacksRoute メソッド |

**主要処理フロー**:
- **186-193行目**: RegisterCallbacksRoute メソッド実装
```pascal
class function THorseCore.RegisterCallbacksRoute(const AMethod: TMethodType; const APath: string): THorseCore;
var
  LCallback: THorseCallback;
begin
  Result := GetInstance;
  for LCallback in GetCallbacks do
    RegisterRoute(AMethod, APath, LCallback);
end;
```

### プログラム呼び出し階層図

```
THorseCore.AddCallback(ACallback)
    │
    ├─ GetInstance() でシングルトンインスタンス取得
    │
    ├─ FCallbacks = nil の場合:
    │      └─ FCallbacks := TList<THorseCallback>.Create
    │
    └─ FCallbacks.Add(ACallback)

--- 次のルート登録時 ---

THorseCore.Get(APath, ACallback)
    │
    ├─ RegisterCallbacksRoute(mtGet, APath)
    │      │
    │      ├─ GetCallbacks() で蓄積コールバック取得
    │      │      ├─ FCallbacks.ToArray で配列化
    │      │      └─ FCallbacks.Clear でクリア
    │      │
    │      └─ for LCallback in GetCallbacks do
    │             └─ RegisterRoute(mtGet, APath, LCallback)
    │
    └─ RegisterRoute(mtGet, APath, ACallback)
```

### データフロー図

```
[AddCallback時]
コールバック関数 ──────► THorseCore.AddCallback ──► FCallbacks.Add
                                                      │
                                                      ▼
                                              FCallbacks リスト
                                              [cb1, cb2, cb3, ...]

[ルート登録時（Get/Post等）]
THorseCore.Get('/path', mainCallback)
    │
    ├─ RegisterCallbacksRoute(mtGet, '/path')
    │      │
    │      ▼
    │   GetCallbacks() ─────────────────────────────┐
    │      │                                         │
    │      ▼                                         ▼
    │   FCallbacks.ToArray ─────► [cb1, cb2, cb3]   FCallbacks.Clear
    │      │
    │      ▼
    │   for each cb in [cb1, cb2, cb3]:
    │      RegisterRoute(mtGet, '/path', cb)
    │
    └─ RegisterRoute(mtGet, '/path', mainCallback)
                │
                ▼
        THorseRouterTree.FCallBack[mtGet]
        └─ [cb1, cb2, cb3, mainCallback]  ← 実行順序
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.pas | `src/Horse.Core.pas` | ソース | THorseCoreクラス - AddCallback/AddCallbacks/GetCallbacks/RegisterCallbacksRoute |
| Horse.Callback.pas | `src/Horse.Callback.pas` | ソース | コールバック型定義（THorseCallback等） |
| Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ソース | THorseRouterTreeクラス - RegisterRouteでコールバックを登録 |
| Horse.Request.pas | `src/Horse.Request.pas` | ソース | THorseRequestクラス - リクエスト情報のラッパー |
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponseクラス - レスポンス処理のラッパー |
