# 通知設計書 9-パラメータ必須エラー通知

## 概要

本ドキュメントは、Horse Webフレームワークにおけるパラメータ必須エラー通知機能の設計を記述する。必須パラメータが不足している場合にBadRequest例外を発生させる通知機能である。

### 本通知の処理概要

本通知は、HTTPリクエストのパラメータ（クエリパラメータ、パスパラメータ、フォームデータ等）が必須として設定されているにも関わらず、リクエストに含まれていない場合に、EHorseException（HTTP 400 Bad Request）を発生させる機能を提供する。

**業務上の目的・背景**：Web APIにおいて、必須パラメータの検証はリクエストの妥当性を確保するための基本要件である。この通知により、クライアントアプリケーションは不足しているパラメータを特定でき、正しいリクエストを再構成できる。バリデーションエラーを明確に伝達することで、API利用者の開発効率向上とデバッグの容易化に寄与する。

**通知の送信タイミング**：THorseCoreParamFieldクラスのAs*メソッド（AsString、AsInteger、AsList等）が呼ばれた際に、FRequired = True かつ パラメータが存在しない（FContains = False）場合に発生する。

**通知の受信者**：HTTPリクエストを送信したクライアント（Webブラウザ、モバイルアプリ、他のサービス等）。例外がキャッチされてHTTPレスポンスに変換される。

**通知内容の概要**：カスタマイズ可能なエラーメッセージが含まれ、HTTPステータスコード400 Bad Requestと共に返却される。デフォルトではパラメータ名を含むメッセージが生成される。

**期待されるアクション**：クライアントは必須パラメータの不足を検知し、エラーメッセージからどのパラメータが不足しているかを特定して、正しいリクエストを再送する。

## 通知種別

例外メッセージ（EHorseException） - HTTP 400 Bad Request

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（例外発生） |
| 優先度 | 即時 |
| リトライ | 無し（クライアント側で判断） |

### 送信先決定ロジック

EHorseExceptionが発生し、上位のエラーハンドラでキャッチされてHTTPレスポンスに変換される。送信先はHTTPリクエスト元のクライアントに自動的に決定される。

## 通知テンプレート

### HTTPレスポンスの場合

| 項目 | 内容 |
|-----|------|
| HTTPステータスコード | 400 |
| ステータスメッセージ | Bad Request |
| Content-Type | application/json (EHorseException.ToJSON使用時) |
| 形式 | テキストまたはJSON |

### 本文テンプレート

```
{RequiredMessage}
```

デフォルトメッセージ例（FRequiredMessageが設定されている場合）：
```
The parameter '{FieldName}' is required
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| なし | - | - | HTTPレスポンスボディのみ |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| FFieldName | パラメータ名 | THorseCoreParamField.FFieldName | Yes |
| FRequiredMessage | カスタムエラーメッセージ | THorseCoreParamField.FRequiredMessage | No |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| パラメータ取得 | AsString/AsInteger/AsList等のメソッド呼び出し | FRequired = True かつ FContains = False | 必須パラメータが存在しない |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| FRequired = False | パラメータが必須でない場合は例外は発生しない |
| FContains = True | パラメータが存在する場合は例外は発生しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[パラメータ取得メソッド呼び出し] --> B{FContains?}
    B -->|True| C[値を返却]
    B -->|False| D{FRequired?}
    D -->|True| E[RaiseHorseException呼び出し]
    E --> F[EHorseException生成]
    F --> G[Status: BadRequest設定]
    G --> H[Error: FRequiredMessage設定]
    H --> I[例外をthrow]
    D -->|False| J[デフォルト値/空値を返却]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| なし | - | 本通知はデータベースを使用しない |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| なし | - | 本通知はデータベースを更新しない |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 必須パラメータ不足 | Required()設定済みのパラメータが未送信 | 400レスポンスを返却 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | サーバー側では行わない（クライアント側で判断） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

パラメータ取得時に即座に発生する。時間帯の制限はない。

## セキュリティ考慮事項

- エラーメッセージにパラメータ名が含まれるが、これはAPI仕様として公開情報であることが多い
- カスタムメッセージ（FRequiredMessage）に機密情報を含めないよう注意が必要
- パラメータ名の列挙攻撃を防ぐため、エラーメッセージの詳細度を適切に設定すること

## 備考

- `Required()`メソッドまたは`Required(True)`メソッドを呼び出すことでFRequired = Trueに設定される
- `RequiredMessage()`メソッドでカスタムエラーメッセージを設定可能
- Format関数で`[FFieldName]`がプレースホルダーとして使用される
- EHorseExceptionはHTTPステータスコードとエラーメッセージを保持する独自例外クラス

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

パラメータフィールドクラスの構造を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | THorseCoreParamFieldクラスのフィールド定義（行23-36） |
| 1-2 | Horse.Exception.pas | `src/Horse.Exception.pas` | EHorseExceptionクラス定義 |

**読解のコツ**: FRequired、FContains、FRequiredMessage、FFieldNameの4つのフィールドが必須エラー判定に関与する。

#### Step 2: エントリーポイントを理解する

パラメータ取得メソッドの処理フローを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | AsStringメソッド（行277-284）が代表的なエントリーポイント |

**主要処理フロー**:
1. **行277**: `function THorseCoreParamField.AsString: string;`
2. **行279-280**: FContains = True の場合は値を返却
3. **行281-283**: FContains = False かつ FRequired = True の場合
   - `RaiseHorseException(FRequiredMessage, [FFieldName])` - **ここが必須エラー通知の送信箇所**

#### Step 3: 例外発生メカニズムを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | RaiseHorseExceptionメソッド（行386-398） |

**RaiseHorseExceptionの詳細**:
```pascal
procedure THorseCoreParamField.RaiseHorseException(const AMessage: string; const Args: array of const);
begin
  RaiseHorseException(Format(AMessage, Args));
end;

procedure THorseCoreParamField.RaiseHorseException(const AMessage: string);
var
  LException: EHorseException;
begin
  LException := EHorseException.New.Status(THTTPStatus.BadRequest).Error(AMessage);
  LException.Message := AMessage;
  raise LException;
end;
```

### プログラム呼び出し階層図

```
[ユーザーコード]
    │
    └─ Request.Query['param'].Required.AsString
           │
           ├─ THorseCoreParamField.Required (行406-410)
           │      │
           │      └─ FRequired := True
           │
           └─ THorseCoreParamField.AsString (行277)
                  │
                  ├─ if FContains then (行280)
                  │      └─ Result := FValue
                  │
                  └─ else if FRequired then (行281-283)
                         │
                         └─ RaiseHorseException(FRequiredMessage, [FFieldName])
                                │
                                └─ RaiseHorseException(Format(...)) (行386-388)
                                       │
                                       └─ RaiseHorseException(AMessage) (行391-398)
                                              │
                                              ├─ EHorseException.New
                                              │      .Status(THTTPStatus.BadRequest)
                                              │      .Error(AMessage)
                                              │
                                              └─ raise LException
```

### データフロー図

```
[入力]                      [処理]                           [出力]

Request.Query       ───▶ THorseCoreParamField         ───▶ EHorseException
['user_id']                │                                 Status: 400
.Required                  │                                 Error: "{message}"
.AsString                  ▼
                    FContains確認
                    [False - パラメータなし]
                           │
                           ▼
                    FRequired確認
                    [True - 必須]
                           │
                           ▼
                    Format(FRequiredMessage, [FFieldName])
                           │
                           ▼
                    EHorseException.New
                           │
                           ▼
                    .Status(THTTPStatus.BadRequest)
                           │
                           ▼
                    .Error(formattedMessage)
                           │
                           ▼
                    raise LException
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.Param.Field.pas | `src/Horse.Core.Param.Field.pas` | ソース | パラメータフィールドクラス、必須エラー通知の送信元 |
| Horse.Exception.pas | `src/Horse.Exception.pas` | ソース | EHorseException例外クラス定義 |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | THTTPStatus列挙型、BadRequest = 400 |
| Horse.Core.Param.pas | `src/Horse.Core.Param.pas` | ソース | パラメータ管理クラス |
