# Horse Webフレームワーク 業務運用マニュアル

## 目次

**【基本操作】**
1. [システム概要](#1-システム概要)
2. [環境準備・初期設定](#2-環境準備初期設定)
3. [基本的なサーバー起動](#3-基本的なサーバー起動)

**【業務フローに沿った機能】**
4. [【STEP1】ルーティングの設定（エンドポイント定義）](#4-step1ルーティングの設定エンドポイント定義)
5. [【STEP2-3】リクエスト処理（データ受信・解析）](#5-step2-3リクエスト処理データ受信解析)
6. [【STEP3-4】レスポンス生成（データ送信）](#6-step3-4レスポンス生成データ送信)
7. [【STEP4-5】サーバー管理・運用完了](#7-step4-5サーバー管理運用完了)

**【サポート機能】**
8. [ミドルウェア機能](#8-ミドルウェア機能)
9. [SSL/TLS設定](#9-ssltls設定)
10. [プロバイダー選択](#10-プロバイダー選択)

**【運用ガイド】**
11. [日常業務の流れ](#11-日常業務の流れ)
12. [よくある質問と対処法](#12-よくある質問と対処法)
13. [トラブルシューティング](#13-トラブルシューティング)

---

## 1. システム概要

### 1.1 Horse Webフレームワークとは

HorseはNode.jsのExpressにインスパイアされた、Delphi/Lazarus向けの軽量かつ高性能なWebフレームワークです。ミニマリストな設計思想に基づき、シンプルなAPIで迅速なWeb開発を実現します。

### 1.2 本システムで行う主要業務

本フレームワークは、主に以下の業務を支援します：

| 業務カテゴリ | 具体的な業務内容 |
|------------|----------------|
| **Web API開発** | RESTful APIの構築、HTTPメソッド別ルーティング、JSONレスポンス生成 |
| **Webサーバー構築** | HTTPサーバーの起動・停止、ポート設定、接続管理 |
| **セキュリティ対応** | SSL/TLS暗号化通信、認証ミドルウェアの実装 |
| **マルチプラットフォーム** | VCL/LCL/CGI/ISAPI/Apache/Daemon等の多様な実行環境対応 |

### 1.3 業務フロー概要

本フレームワークを使用したWeb開発は、以下の流れで進行します：

```
┌─────────────────────────────────────────────────────────────────┐
│                        開発フロー                                │
├─────────────────────────────────────────────────────────────────┤
│                                                                 │
│  【STEP 1】ルーティングの設定（エンドポイントの定義）             │
│      ↓                                                          │
│  【STEP 2】リクエストの受信（クライアントからのデータ取得）        │
│      ↓                                                          │
│  【STEP 3】リクエストの解析（パラメータ・ボディの処理）           │
│      ↓                                                          │
│  【STEP 4】レスポンスの生成（処理結果の返却）                     │
│      ↓                                                          │
│  【STEP 5】サーバーの運用管理（起動・停止・監視）                 │
│                                                                 │
└─────────────────────────────────────────────────────────────────┘
```

**役割別の業務概要：**

| 役割 | 主な業務 |
|------|---------|
| **開発者** | ルーティング定義、リクエストハンドラ実装、レスポンス生成ロジック作成 |
| **運用担当者** | サーバー起動・停止、ポート設定、SSL証明書管理、監視 |

### 1.4 主な機能

| 機能 | 説明 | 業務フローでの位置 |
|------|------|------------------|
| HTTPルーティング | GET/POST/PUT/DELETE/PATCH/HEAD等のHTTPメソッドに対応したルーティング | STEP 1 |
| リクエスト処理 | ヘッダー、クエリパラメータ、パスパラメータ、Cookie、フォームデータの取得 | STEP 2-3 |
| レスポンス生成 | テキスト送信、ファイル送信、ダウンロード、リダイレクト対応 | STEP 4 |
| サーバー管理 | Listen/StopListenによるサーバー制御 | STEP 5 |
| ミドルウェア | Use()メソッドによるミドルウェアチェーンの構築 | 随時利用 |
| SSL/TLS | OpenSSLを使用したHTTPS通信のサポート | 随時利用 |

### 1.5 ユーザー種別

システムには以下のユーザー種別があります：

| ユーザー種別 | 説明 |
|-------------|------|
| 開発者 | Horseを使用してWebアプリケーションを開発するDelph/Lazarusプログラマー |
| 運用担当者 | Horseベースのアプリケーションを運用・管理する担当者 |
| APIクライアント | Horseで構築されたWeb APIを利用するエンドユーザー・システム |

### 1.6 対応環境

**対応IDE/コンパイラ：**
- Delphi XE7以降（Delphi 13 Florence、12 Athens、11 Alexandria、10.4 Sydney、10.3 Rio、10.2 Tokyo、10.1 Berlin、10 Seattle、XE8対応）
- Lazarus/Free Pascal

**推奨ブラウザ（API動作確認用）：**
- Google Chrome（最新版）
- Microsoft Edge（最新版）
- Safari（最新版）
- Firefox（最新版）

---

## 2. 環境準備・初期設定

### 2.1 インストール方法

**Bossパッケージマネージャーを使用する場合：**

1. コマンドプロンプトまたはターミナルを開きます
2. プロジェクトディレクトリに移動します
3. 以下のコマンドを実行します：
   ```sh
   boss install horse
   ```

**手動でインストールする場合：**

1. GitHubリポジトリからHorseのソースコードをダウンロードします
2. `src/` フォルダをプロジェクトのライブラリパスに追加します
3. DelphiまたはLazarus IDEでプロジェクトを開きます

### 2.2 プロジェクトへの組み込み

**入力項目：**

| 項目 | 入力内容 |
|------|---------|
| ライブラリパス | Horseの`src/`フォルダへのパス |
| コンパイルディレクティブ | 使用するプロバイダーに応じたディレクティブ（後述） |

### 2.3 コンパイルディレクティブの設定

プロバイダーを切り替えるには、以下のコンパイルディレクティブを設定します：

| ディレクティブ | 用途 |
|--------------|------|
| `HORSE_VCL` | VCLアプリケーション用（Windows GUI） |
| `HORSE_CGI` | CGIアプリケーション用 |
| `HORSE_ISAPI` | ISAPIモジュール用（IIS） |
| `HORSE_APACHE` | Apacheモジュール用 |
| `HORSE_DAEMON` | デーモン/サービス用 |
| `HORSE_LCL` | LCL（Lazarus）アプリケーション用 |
| `HORSE_FCGI` | FastCGI用（Lazarusのみ） |

---

## 3. 基本的なサーバー起動

### 3.1 VCLアプリケーションでの起動手順

1. VCLアプリケーションを起動します
2. 「VCLメインフォーム」画面が表示されます

**表示される項目：**

| 項目名 | コンポーネント | 説明 |
|--------|--------------|------|
| Port | edtPort | サーバーがリッスンするポート番号（デフォルト: 9000） |
| Start | btnStart | サーバーを起動するボタン |
| Stop | btnStop | サーバーを停止するボタン |

3. 「Port」欄にポート番号を入力します（デフォルト: 9000）
4. 「Start」ボタンをクリックします
5. サーバーが起動し、指定ポートでリクエストを待機します
6. 停止する場合は「Stop」ボタンをクリックします

### 3.2 コードでの起動手順（Delphi）

```delphi
uses Horse;

begin
  THorse.Get('/ping',
    procedure(Req: THorseRequest; Res: THorseResponse)
    begin
      Res.Send('pong');
    end);

  THorse.Listen(9000);
end.
```

### 3.3 コードでの起動手順（Lazarus）

```delphi
{$MODE DELPHI}{$H+}

uses Horse;

procedure GetPing(Req: THorseRequest; Res: THorseResponse);
begin
  Res.Send('Pong');
end;

begin
  THorse.Get('/ping', GetPing);
  THorse.Listen(9000);
end.
```

---

## 4. 【STEP1】ルーティングの設定（エンドポイント定義）

> **業務フローでの位置**: 開発者がWeb APIのエンドポイントを定義し、HTTPメソッドとパスに対するハンドラを登録するステップです。

### 4.1 基本的なルーティング登録

**やるべきこと：**
- 対応するHTTPメソッドを決定する
- URLパスを設計する
- コールバック関数を実装する

**システムでの操作手順：**

1. `THorse.Get` / `THorse.Post` / `THorse.Put` / `THorse.Delete` などのメソッドを呼び出します
2. 第1引数にパス（例：'/users'）を指定します
3. 第2引数にコールバックプロシージャを指定します

**対応HTTPメソッド一覧：**

| メソッド | Horseメソッド | 用途 |
|---------|--------------|------|
| GET | THorse.Get | データの取得 |
| POST | THorse.Post | データの作成 |
| PUT | THorse.Put | データの更新（全体） |
| DELETE | THorse.Delete | データの削除 |
| PATCH | THorse.Patch | データの更新（部分） |
| HEAD | THorse.Head | ヘッダーのみ取得 |
| ALL | THorse.All | 全メソッド対応 |

### 4.2 パスパラメータの使用

動的なURLパラメータを使用する場合：

```delphi
THorse.Get('/users/:id',
  procedure(Req: THorseRequest; Res: THorseResponse)
  var
    UserId: string;
  begin
    UserId := Req.Params.Items['id'];
    Res.Send('User ID: ' + UserId);
  end);
```

### 4.3 ルートグループの作成

共通のプレフィックスを持つルートをグループ化する場合：

```delphi
THorse.Route('/api')
  .Get('/users', GetUsersHandler)
  .Post('/users', CreateUserHandler);
```

**注意点・Tips：**
- パスは先頭にスラッシュ（/）を含めて指定します
- パスパラメータは`:paramName`形式で定義します
- 正規表現によるパスマッチングも可能です

---

## 5. 【STEP2-3】リクエスト処理（データ受信・解析）

> **業務フローでの位置**: クライアントからのHTTPリクエストを受信し、ヘッダー、パラメータ、ボディ等を解析するステップです。

### 5.1 クエリパラメータの取得

URLのクエリ文字列（?key=value）からパラメータを取得します。

**操作手順：**

1. コールバック内で`Req.Query`オブジェクトを使用します
2. `Items['key']`でパラメータ値を取得します

```delphi
THorse.Get('/search',
  procedure(Req: THorseRequest; Res: THorseResponse)
  var
    Keyword: string;
  begin
    Keyword := Req.Query.Items['q'];
    Res.Send('Search: ' + Keyword);
  end);
```

### 5.2 パスパラメータの取得

URLパスに埋め込まれたパラメータを取得します。

```delphi
THorse.Get('/users/:id/posts/:postId',
  procedure(Req: THorseRequest; Res: THorseResponse)
  var
    UserId, PostId: string;
  begin
    UserId := Req.Params.Items['id'];
    PostId := Req.Params.Items['postId'];
    Res.Send('User: ' + UserId + ', Post: ' + PostId);
  end);
```

### 5.3 リクエストヘッダーの取得

HTTPヘッダーの値を取得します。

```delphi
THorse.Get('/protected',
  procedure(Req: THorseRequest; Res: THorseResponse)
  var
    AuthHeader: string;
  begin
    AuthHeader := Req.Headers.Items['Authorization'];
    // 認証処理...
  end);
```

### 5.4 リクエストボディの取得

POSTやPUTリクエストのボディデータを取得します。

```delphi
THorse.Post('/users',
  procedure(Req: THorseRequest; Res: THorseResponse)
  var
    Body: string;
  begin
    Body := Req.Body;
    // JSONパース等の処理...
  end);
```

### 5.5 Cookieの取得

HTTPリクエストのCookie値を取得します。

```delphi
THorse.Get('/profile',
  procedure(Req: THorseRequest; Res: THorseResponse)
  var
    SessionId: string;
  begin
    SessionId := Req.Cookie.Items['session_id'];
    // セッション処理...
  end);
```

### 5.6 フォームデータの取得

multipart/form-dataやapplication/x-www-form-urlencoded形式のデータを取得します。

```delphi
THorse.Post('/upload',
  procedure(Req: THorseRequest; Res: THorseResponse)
  var
    FileName: string;
  begin
    FileName := Req.ContentFields.Items['file'];
    // ファイル処理...
  end);
```

**パラメータ型変換一覧：**

| 変換先型 | メソッド例 |
|---------|----------|
| String | `.AsString` |
| Integer | `.AsInteger` |
| Int64 | `.AsInt64` |
| Float | `.AsFloat` |
| Boolean | `.AsBoolean` |
| Date | `.AsDate` |
| Time | `.AsTime` |
| DateTime | `.AsDateTime` |

---

## 6. 【STEP3-4】レスポンス生成（データ送信）

> **業務フローでの位置**: リクエストを処理した結果をHTTPレスポンスとしてクライアントに返却するステップです。

### 6.1 テキストレスポンスの送信

```delphi
THorse.Get('/hello',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res.Send('Hello, World!');
  end);
```

### 6.2 HTTPステータスコードの設定

```delphi
THorse.Get('/notfound',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res.Status(THTTPStatus.NotFound).Send('Resource not found');
  end);
```

**主要なHTTPステータスコード：**

| ステータス | 定数名 | 用途 |
|----------|--------|------|
| 200 | THTTPStatus.OK | 成功 |
| 201 | THTTPStatus.Created | 作成成功 |
| 204 | THTTPStatus.NoContent | コンテンツなし |
| 400 | THTTPStatus.BadRequest | 不正なリクエスト |
| 401 | THTTPStatus.Unauthorized | 認証エラー |
| 403 | THTTPStatus.Forbidden | アクセス禁止 |
| 404 | THTTPStatus.NotFound | リソース未発見 |
| 500 | THTTPStatus.InternalServerError | サーバーエラー |

### 6.3 ヘッダーの追加

```delphi
THorse.Get('/api/data',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res
      .AddHeader('X-Custom-Header', 'CustomValue')
      .AddHeader('Cache-Control', 'no-cache')
      .Send('Data');
  end);
```

### 6.4 Content-Typeの設定

```delphi
THorse.Get('/api/json',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res
      .ContentType('application/json')
      .Send('{"status":"ok"}');
  end);
```

### 6.5 リダイレクト

```delphi
THorse.Get('/old-url',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res.RedirectTo('/new-url');
  end);
```

### 6.6 ファイル送信（インライン表示）

```delphi
THorse.Get('/view/document',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res.SendFile('C:\Documents\report.pdf');
  end);
```

### 6.7 ファイルダウンロード

```delphi
THorse.Get('/download/document',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res.Download('C:\Documents\report.pdf', 'report.pdf');
  end);
```

### 6.8 HTMLレンダリング

```delphi
THorse.Get('/page',
  procedure(Req: THorseRequest; Res: THorseResponse)
  begin
    Res.Render('C:\Templates\index.html');
  end);
```

---

## 7. 【STEP4-5】サーバー管理・運用完了

> **業務フローでの位置**: 開発したアプリケーションをデプロイし、サーバーの起動・停止・監視を行うステップです。

### 7.1 サーバーの起動

**VCLアプリケーションの場合：**

1. アプリケーションを起動します
2. 「Port」欄にポート番号を入力します
3. 「Start」ボタンをクリックします

**コードでの起動：**

```delphi
// デフォルトポート（9000）で起動
THorse.Listen;

// 指定ポートで起動
THorse.Listen(8080);

// ホストとポートを指定して起動
THorse.Listen(8080, '0.0.0.0');

// コールバック付きで起動
THorse.Listen(9000,
  procedure
  begin
    Writeln('Server is running on port 9000');
  end);
```

### 7.2 サーバーの停止

**VCLアプリケーションの場合：**

1. 「Stop」ボタンをクリックします
2. またはフォームを閉じます（自動停止）

**コードでの停止：**

```delphi
THorse.StopListen;
```

### 7.3 起動状態の確認

```delphi
if THorse.IsRunning then
  Writeln('Server is running')
else
  Writeln('Server is stopped');
```

### 7.4 接続設定

```delphi
// 最大接続数の設定
THorse.MaxConnections := 100;

// 接続キューの設定
THorse.ListenQueue := 50;

// Keep-Aliveの設定
THorse.KeepConnectionAlive := True;
```

---

## 8. ミドルウェア機能

### 8.1 グローバルミドルウェアの登録

全てのリクエストに適用されるミドルウェアを登録します。

```delphi
// ロギングミドルウェア
THorse.Use(
  procedure(Req: THorseRequest; Res: THorseResponse; Next: TNextProc)
  begin
    Writeln('Request: ' + Req.RawWebRequest.PathInfo);
    Next;
  end);
```

### 8.2 パス別ミドルウェアの登録

特定のパスにのみ適用されるミドルウェアを登録します。

```delphi
// /api/*へのリクエストのみに適用
THorse.Use('/api',
  procedure(Req: THorseRequest; Res: THorseResponse; Next: TNextProc)
  begin
    // 認証チェック
    if Req.Headers.Items['Authorization'] = '' then
    begin
      Res.Status(THTTPStatus.Unauthorized).Send('Unauthorized');
      Exit;
    end;
    Next;
  end);
```

### 8.3 ミドルウェアチェーンの実行

`Next`プロシージャを呼び出すことで、次のミドルウェアまたはハンドラに処理を渡します。

**注意点・Tips：**
- `Next`を呼び出さない場合、以降の処理は実行されません
- 認証失敗時など、処理を中断したい場合は`Next`を呼び出しません

---

## 9. SSL/TLS設定

### 9.1 SSL対応サーバーの起動（VCLアプリケーション）

**前提条件：**
- libeay32.dllとssleay32.dllが実行ファイルフォルダに必要
- SSL証明書（秘密鍵・公開鍵）が必要

**操作手順：**

1. VCL-SSLアプリケーションを起動します
2. 以下の項目を設定します：

| 項目名 | コンポーネント | 入力内容 |
|--------|--------------|----------|
| Port | edtPort | ポート番号（デフォルト: 9000） |
| Private Key | leKey | 秘密鍵ファイル（.key）のパス |
| Public Key | leCrt | 公開鍵ファイル（.crt）のパス |
| Password | lePassword | 秘密鍵のパスワード |

3. 「...」ボタンで各ファイルを選択します
4. 「Start」ボタンをクリックします
5. StatusBarに「Securite Server is running on https://...」と表示されます

### 9.2 SSL証明書の生成（テスト用）

自己署名証明書を生成する場合（Git Bash推奨）：

```bash
openssl req -x509 -nodes -days 365 -newkey rsa:2048 -keyout cert.key -out cert.crt
```

### 9.3 コードでのSSL設定

```delphi
THorse.IOHandleSSL
  .KeyFile('cert.key')
  .CertFile('cert.crt')
  .Password('your-password')
  .SSLVersions([sslvTLSv1_2])
  .Active(True);

THorse.Listen(443);
```

**注意点・Tips：**
- 本番環境では正規証明書（Let's Encrypt等）の使用を推奨
- TLS 1.2以上の使用を推奨

---

## 10. プロバイダー選択

### 10.1 VCLプロバイダー

Windows GUIアプリケーション用のプロバイダーです。

**特徴：**
- Windows専用
- GUIスレッドでの動作をサポート
- 開発時のデバッグに便利

**コンパイルディレクティブ：** `HORSE_VCL`

### 10.2 Consoleプロバイダー

コマンドラインアプリケーション用のプロバイダーです。

**特徴：**
- Windows/Linux対応
- GUIなしで動作
- サーバー用途に最適

### 10.3 CGIプロバイダー

CGIモードで動作するプロバイダーです。

**特徴：**
- Webサーバー（Apache/IIS等）との連携
- リクエストごとにプロセス起動

**コンパイルディレクティブ：** `HORSE_CGI`

### 10.4 ISAPIプロバイダー

IISのISAPIモジュールとして動作するプロバイダーです。

**特徴：**
- IIS専用
- 高パフォーマンス

**コンパイルディレクティブ：** `HORSE_ISAPI`

### 10.5 Apacheプロバイダー

Apacheモジュールとして動作するプロバイダーです。

**特徴：**
- Apache専用
- mod_delphiとして動作

**コンパイルディレクティブ：** `HORSE_APACHE`

### 10.6 Daemonプロバイダー

Linuxデーモンとして動作するプロバイダーです。

**特徴：**
- Linux専用
- バックグラウンドサービスとして動作

**コンパイルディレクティブ：** `HORSE_DAEMON`

### 10.7 LCLプロバイダー

Lazarus/LCLアプリケーション用のプロバイダーです。

**特徴：**
- クロスプラットフォーム対応
- Free Pascal/Lazarus環境

**コンパイルディレクティブ：** `HORSE_LCL`

---

## 11. 日常業務の流れ

本セクションでは、業務フローに沿った日常業務の流れを説明します。

```
業務フロー概要（再掲）
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
【STEP 1】ルーティングの設定（エンドポイントの定義） → 4章参照
    ↓
【STEP 2】リクエストの受信（クライアントからのデータ取得） → 5章参照
    ↓
【STEP 3】リクエストの解析（パラメータ・ボディの処理） → 5章参照
    ↓
【STEP 4】レスポンスの生成（処理結果の返却） → 6章参照
    ↓
【STEP 5】サーバーの運用管理（起動・停止・監視） → 7章参照
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
```

### 11.1 【開発者】STEP1: ルーティングの設定

新しいAPIエンドポイントを追加する場合の手順：

1. HTTPメソッドを決定します（GET/POST/PUT/DELETE等）
2. URLパスを設計します（例：/api/v1/users）
3. `THorse.Get`等のメソッドでルートを登録します
4. コールバック関数を実装します

### 11.2 【開発者】STEP2-3: リクエスト処理の実装

リクエストデータを処理する場合の手順：

1. 必要なパラメータの種類を特定します
   - クエリパラメータ: `Req.Query`
   - パスパラメータ: `Req.Params`
   - ヘッダー: `Req.Headers`
   - ボディ: `Req.Body`
2. 必須パラメータのバリデーションを行います
3. 型変換を行います

### 11.3 【開発者】STEP4: レスポンス生成

レスポンスを返却する場合の手順：

1. 処理結果に応じたHTTPステータスコードを設定します
2. 必要なヘッダーを追加します
3. Content-Typeを設定します
4. レスポンスボディを送信します

### 11.4 【運用担当者】STEP5: サーバー運用

日常的なサーバー運用の手順：

1. アプリケーションを起動します
2. ポート番号を確認・設定します
3. 「Start」ボタンでサーバーを起動します
4. 動作確認のため`/ping`エンドポイントにアクセスします
5. 「pong」が返却されれば正常動作
6. 終了時は「Stop」ボタンでサーバーを停止します

---

## 12. よくある質問と対処法

### 12.1 サーバー起動に関するQ&A

**Q: サーバーが起動しません**

A: 以下を確認してください：
- 指定したポートが他のアプリケーションで使用されていないか
- ファイアウォールがポートをブロックしていないか
- 管理者権限で実行しているか（1024未満のポートを使用する場合）

**Q: ポートが既に使用中というエラーが出ます**

A: 以下を確認してください：
- `netstat -an | findstr :9000` で使用状況を確認
- 別のポート番号を指定して起動
- 前回のプロセスが残っている場合はタスクマネージャーで終了

### 12.2 SSL/TLSに関するQ&A

**Q: SSL証明書が読み込めません**

A: 以下を確認してください：
- libeay32.dllとssleay32.dllが実行ファイルフォルダにあるか
- 証明書ファイルのパスが正しいか
- パスワードが正しいか

**Q: 「securite pong」ではなく「pong」が返ります**

A: SSL設定が正しく適用されていない可能性があります：
- `THorse.IOHandleSSL.Active(True)`が呼び出されているか確認
- HTTPS（https://）でアクセスしているか確認

### 12.3 ルーティングに関するQ&A

**Q: ルートが見つかりません（404エラー）**

A: 以下を確認してください：
- ルート登録時のパスが正しいか（先頭の/を含む）
- HTTPメソッドが一致しているか
- ルート登録がサーバー起動前に行われているか

**Q: パスパラメータが取得できません**

A: 以下を確認してください：
- ルート定義が`:paramName`形式になっているか
- `Req.Params.Items['paramName']`の名前が一致しているか

---

## 13. トラブルシューティング

### 13.1 エラーメッセージと対処法

| エラーメッセージ | 原因 | 対処法 |
|----------------|------|--------|
| Address already in use | ポートが既に使用中 | 別のポートを指定するか、使用中のプロセスを終了 |
| Unable to load OpenSSL library | OpenSSL DLLが見つからない | libeay32.dll、ssleay32.dllを実行ファイルフォルダに配置 |
| Certificate file not found | 証明書ファイルが存在しない | ファイルパスを確認し、正しいパスを指定 |
| Invalid password | 秘密鍵のパスワードが不正 | 正しいパスワードを入力 |
| Route not found | 登録されていないパスへのアクセス | ルート定義を確認し、正しいパスを使用 |

### 13.2 よくあるトラブルと解決方法

**トラブル1: サーバーが応答しない**

解決方法：
1. `THorse.IsRunning`でサーバーが起動しているか確認
2. ファイアウォール設定を確認
3. ネットワーク設定（ホスト、ポート）を確認

**トラブル2: レスポンスが空になる**

解決方法：
1. `Res.Send()`が呼び出されているか確認
2. 例外が発生していないか確認
3. ミドルウェアで処理が中断されていないか確認

**トラブル3: 文字化けする**

解決方法：
1. `Res.ContentType('application/json; charset=utf-8')`でエンコーディングを指定
2. ソースファイルのエンコーディングをUTF-8に設定

### 13.3 サポートへの連絡方法

**GitHub Issues：**
- リポジトリ: https://github.com/hashload/horse
- バグ報告や機能リクエストはIssuesで受け付けています

**コミュニティ：**
- Delphiコミュニティやフォーラムでの質問も可能です

---

## 改訂履歴

| 版数 | 改訂日 | 改訂内容 |
|-----|--------|---------|
| 1.0 | 2026-01-22 | 初版作成 |

---

*本マニュアルは、Horse Webフレームワークのエンドユーザー向け業務運用マニュアルです。*
*フレームワークの更新により、画面や操作方法が変更される場合があります。*
