# 機能設計書 27-リポジトリブラウザ

## 概要

本ドキュメントは、Jenkinsのリポジトリブラウザ（RepositoryBrowser）機能の設計仕様を定義する。RepositoryBrowserは、ViewCVSやFishEye、GitWebなどの外部リポジトリブラウザとJenkinsを連携するための抽象化レイヤーを提供する。

### 本機能の処理概要

リポジトリブラウザ機能は、変更ログの各エントリからリポジトリブラウザへのリンクを生成し、開発者がコードの詳細を簡単に確認できるようにする機能である。

**業務上の目的・背景**：開発者がビルド結果から直接コードの変更内容を確認したい場合、リポジトリブラウザへのリンクがあると便利である。RepositoryBrowserは、JenkinsとGitHub、GitLab、Bitbucket、FishEye、ViewCVSなどの外部ブラウザを連携する仕組みを提供する。

**機能の利用シーン**：
- ビルド詳細画面からコミットの詳細ページへ遷移
- 変更されたファイルの差分表示ページへ遷移
- コードレビューツールとの連携
- 課題管理システムとの連携（コミットIDからの紐付け）

**主要な処理内容**：
1. 変更セット（ChangeLogSet.Entry）からリポジトリブラウザURLの生成
2. ファイル変更（AffectedFile）から差分表示URLの生成
3. SCM設定との関連付け
4. プラグインによる拡張（各種リポジトリブラウザ対応）

**関連システム・外部連携**：GitHub、GitLab、Bitbucket、FishEye、ViewCVS、Gitiles等の外部リポジトリブラウザ。

**権限による制御**：リポジトリブラウザ設定はジョブの設定権限を持つユーザーが変更可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 12 | ジョブ設定 | 主画面 | リポジトリブラウザの設定 |
| 16 | ビルド詳細 | 表示 | リポジトリブラウザへのリンク表示 |

## 機能種別

拡張ポイント / 外部連携

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| changeSet | E extends ChangeLogSet.Entry | Yes | 変更セットエントリ | null不可 |

### 入力データソース

ジョブ設定画面からの設定値。リポジトリブラウザのURL等は各実装固有。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| changeSetLink | URL | 変更セットの詳細ページURL |
| diffLink | URL | ファイル差分ページURL（実装依存） |
| fileLink | URL | ファイル表示ページURL（実装依存） |

### 出力先

ビルド詳細画面のHTMLリンクとして出力。

## 処理フロー

### 処理シーケンス

```
1. ビルド詳細表示リクエスト
   └─ ChangeLogSet取得
2. 各Entry表示時
   └─ RepositoryBrowser.getChangeSetLink(entry) 呼び出し
   └─ URLを生成してリンク表示
3. ファイル変更表示時
   └─ RepositoryBrowser固有のdiffLink/fileLinkメソッド呼び出し
   └─ URLを生成してリンク表示
```

### フローチャート

```mermaid
flowchart TD
    A[ビルド詳細表示] --> B[ChangeLogSet取得]
    B --> C[各Entry表示]
    C --> D{RepositoryBrowser設定?}
    D -->|Yes| E[getChangeSetLink呼び出し]
    D -->|No| F[リンクなし表示]
    E --> G{URL生成成功?}
    G -->|Yes| H[リンク表示]
    G -->|No/null| F
    H --> I[ファイル変更表示]
    I --> J[diffLink生成]
    J --> K[差分リンク表示]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | null許容 | getChangeSetLinkはnullを返すことが許可されている | 全ての変更セット |
| BR-002 | URLスラッシュ正規化 | normalizeToEndWithSlashでURLを正規化 | URL生成時 |
| BR-003 | 先頭スラッシュ除去 | trimHeadSlashでパスの先頭スラッシュを除去 | パス処理時 |

### 計算ロジック

```java
// URL正規化の例
protected static URL normalizeToEndWithSlash(URL url) {
    if (url.getPath().endsWith("/"))
        return url;
    String q = url.getQuery();
    q = q != null ? '?' + q : "";
    return new URL(url, url.getPath() + '/' + q);
}
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 設定保存 | config.xml | INSERT/UPDATE | リポジトリブラウザ設定の永続化 |

### テーブル別操作詳細

#### config.xml（ジョブ設定ファイル）

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT/UPDATE | scm/browser | リポジトリブラウザ設定 | SCM設定の一部として |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | IOException | URL生成時のエラー | nullを返す |
| - | MalformedURLException | 不正なURL形式 | Error例外として再スロー |

### リトライ仕様

リトライは行わない。URL生成失敗時はリンクなしで表示。

## トランザクション仕様

トランザクション管理はない。URL生成は純粋な計算処理。

## パフォーマンス要件

- URL生成は軽量な処理であり、特別なパフォーマンス要件はない
- 大量の変更エントリがある場合でもメモリ効率的に処理

## セキュリティ考慮事項

- 外部リポジトリブラウザへのリンクはtarget="_blank"等で新規ウィンドウで開く
- URLはエスケープ処理済み
- 認証が必要なリポジトリブラウザへのリンクはユーザー認証に委ねる

## 備考

- RepositoryBrowserはDescribableとExtensionPointを実装する抽象クラス
- 具体的な実装はプラグインで提供（Git Plugin、Subversion Plugin等に含まれる）
- SCMと1:1で関連付けられ、SCMと共に永続化される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

RepositoryBrowserの基本構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | RepositoryBrowser.java | `core/src/main/java/hudson/scm/RepositoryBrowser.java` | 抽象クラス定義、型パラメータ |

**読解のコツ**: RepositoryBrowserはジェネリクスを使用し、`E extends ChangeLogSet.Entry`を型パラメータとして持つ。これにより、SCM固有のEntry型を扱える。

#### Step 2: 主要メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RepositoryBrowser.java | `core/src/main/java/hudson/scm/RepositoryBrowser.java` | getChangeSetLink() |

**主要処理フロー**:
- **59行目**: RepositoryBrowser - クラス定義（Describable, ExtensionPoint, Serializable実装）
- **68行目**: getChangeSetLink() - 抽象メソッド、変更セットへのリンク生成

#### Step 3: ユーティリティメソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | RepositoryBrowser.java | `core/src/main/java/hudson/scm/RepositoryBrowser.java` | trimHeadSlash()、normalizeToEndWithSlash() |

**主要処理フロー**:
- **73-76行目**: trimHeadSlash() - パスの先頭スラッシュ除去
- **83-96行目**: normalizeToEndWithSlash() - URLの末尾スラッシュ正規化

#### Step 4: 拡張ポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | RepositoryBrowser.java | `core/src/main/java/hudson/scm/RepositoryBrowser.java` | all() |

**主要処理フロー**:
- **101-103行目**: all() - 全登録RepositoryBrowserのDescriptor取得

### プログラム呼び出し階層図

```
ビルド詳細表示（Jelly/Groovy View）
    │
    ├─ ChangeLogSet.getBrowser()
    │      └─ RepositoryBrowser取得
    │
    └─ RepositoryBrowser.getChangeSetLink(entry)
           │
           └─ [実装依存] URL生成
                  │
                  ├─ normalizeToEndWithSlash()
                  │      └─ URL正規化
                  │
                  └─ trimHeadSlash()
                         └─ パス正規化

SCM設定保存
    │
    └─ SCM.browser
           └─ RepositoryBrowser永続化
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ChangeLogSet.Entry   RepositoryBrowser
├─ commitId        ├─ getChangeSetLink() ────▶     変更セットURL
├─ affectedFiles   │
└─ ...             └─ [実装固有メソッド]
                       ├─ getDiffLink() ─────────▶     差分URL
                       └─ getFileLink() ─────────▶     ファイルURL

ベースURL           URL正規化
(設定値)          ├─ normalizeToEndWithSlash() ──▶     正規化URL
                  └─ trimHeadSlash() ───────────▶     パス正規化
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| RepositoryBrowser.java | `core/src/main/java/hudson/scm/RepositoryBrowser.java` | ソース | リポジトリブラウザ抽象基底クラス |
| RepositoryBrowsers.java | `core/src/main/java/hudson/scm/RepositoryBrowsers.java` | ソース | リポジトリブラウザ関連ユーティリティ |
| ChangeLogSet.java | `core/src/main/java/hudson/scm/ChangeLogSet.java` | ソース | 変更ログセット |
| SCM.java | `core/src/main/java/hudson/scm/SCM.java` | ソース | SCM基底クラス（browser保持） |
