# 機能設計書 8-エグゼキューター

## 概要

本ドキュメントは、Jenkinsで実際にビルドを実行するスレッドを管理するエグゼキューター（hudson.model.Executor）の機能設計を記述する。ビルドの実行制御とリソース管理の中核コンポーネントである。

### 本機能の処理概要

Executorクラスは、ビルドを実行するスレッドを表現する。各ノード（コントローラー/エージェント）は1つ以上のエグゼキュータを持ち、それぞれが独立してビルドを実行する。

**業務上の目的・背景**：並列ビルド実行を可能にするため、各ノードに複数のエグゼキュータを配置できる。エグゼキュータ数の設定により、同時実行可能なビルド数を制御し、リソースの効率的な利用と過負荷防止を実現する。

**機能の利用シーン**：ビルドの並列実行、ビルドの中断・停止、エグゼキュータ数の調整、実行中ビルドの監視、アイドルエグゼキュータの確認。

**主要な処理内容**：
1. キューからのワークユニット受け取りと実行開始
2. ビルド実行スレッドの管理
3. ビルドの中断（interrupt）処理
4. 実行時間・進捗のトラッキング
5. 非同期実行（AsynchronousExecution）のサポート
6. 実行完了後のクリーンアップ

**関連システム・外部連携**：Queue（キュー管理）、Computer（ノード管理）、Run（ビルド実行）。

**権限による制御**：ビルド中断はItem.CANCEL権限で制御。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 15 | コンソール出力 | 参照画面 | 実行中ビルドの状態表示 |
| 30 | エグゼキュータ設定 | 主画面 | エグゼキュータ数の設定 |
| 32 | 負荷統計 | 参照画面 | エグゼキュータ使用状況の表示 |

## 機能種別

実行制御 / スレッド管理 / リソース管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| owner | Computer | Yes | 所属するComputer | 非null |
| n | int | Yes | エグゼキュータ番号 | 0以上 |
| workUnit | WorkUnit | Yes | 実行するワークユニット | 非null（開始時） |

### 入力データソース

- Queue: ビルドキューからのワークユニット割り当て
- Computer: 所属ノード情報

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| executable | Executable | 実行中のビルド |
| startTime | long | 実行開始時刻 |
| progress | int | 進捗（-1は不明） |
| interruptStatus | Result | 中断時の結果 |
| causes | List<CauseOfInterruption> | 中断理由一覧 |

### 出力先

- Webインターフェース: エグゼキュータステータス表示
- REST API: /computer/{name}/executors/{number}/api/json

## 処理フロー

### 処理シーケンス

```
1. エグゼキュータ生成
   ├─ Computer.createExecutors() で生成
   └─ スレッド名設定（"Executor #{n} for {computerName}"）
2. ワークユニット受け取り
   ├─ Queue.JobOffer.set() から呼び出し
   ├─ workUnit 設定
   └─ スレッド開始（start()）
3. ビルド実行
   ├─ run() メソッドでループ
   ├─ Queue.Executable 生成
   ├─ executable.run() 実行
   └─ 結果記録
4. 中断処理
   ├─ interrupt() で中断要求
   ├─ interruptStatus 設定
   └─ causes 記録
5. 完了処理
   ├─ executable クリア
   ├─ Computer に完了通知
   └─ 次のワーク待機
```

### フローチャート

```mermaid
flowchart TD
    A[Executor生成] --> B[アイドル状態]
    B --> C{JobOffer?}
    C -->|No| B
    C -->|Yes| D[workUnit設定]
    D --> E[スレッド開始]
    E --> F[Executable生成]
    F --> G[ビルド実行]
    G --> H{中断要求?}
    H -->|Yes| I[中断処理]
    H -->|No| J{完了?}
    I --> K[結果設定]
    J -->|No| G
    J -->|Yes| L[クリーンアップ]
    K --> L
    L --> B
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 1ビルド1エグゼキュータ | 1つのエグゼキュータは同時に1つのビルドのみ実行 | 常時 |
| BR-002 | 自己中断許可 | 同一スレッドからのinterrupt()は通常の中断 | 中断時 |
| BR-003 | 外部中断記録 | 外部スレッドからの中断は原因を記録 | 中断時 |
| BR-004 | 非同期実行 | AsynchronousExecutionで非同期ビルドをサポート | 対応ビルド時 |

### 計算ロジック

- 推定残り時間: executableEstimatedDuration - (現在時刻 - startTime)
- 進捗率: (経過時間 / 推定所要時間) * 100

## データベース操作仕様

Executorはメモリ上のオブジェクトで永続化されない。

### ファイル操作一覧

該当なし（メモリ上で管理）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | InterruptedException | ビルド中断 | 中断理由を記録して終了 |
| - | Error | JVMエラー | ログ出力、Computer通知 |
| - | RuntimeException | 実行時エラー | エラー記録、次ビルドへ |

### リトライ仕様

エグゼキュータレベルでのリトライなし。次のビルドを待機。

## トランザクション仕様

ReadWriteLockで状態アクセスを排他制御。

## パフォーマンス要件

- スレッド開始: ミリ秒オーダー
- 中断応答: 実行中処理に依存
- 状態取得: 即時

## セキュリティ考慮事項

- 中断権限: 適切な権限チェック（UserInterruption記録）
- リソースアクセス: ビルド実行時の権限に従う

## 備考

- エグゼキュータはオンデマンドでスレッド開始（1.536以降）
- isActive()とisAlive()の違いに注意
- parking状態でキューからのジョブ待機

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Executorクラスの主要フィールドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Executor.java | `core/src/main/java/hudson/model/Executor.java` | ownerフィールド（94行目）、numberフィールド（109行目）、executableフィールド（114行目）を確認 |
| 1-2 | Executor.java | `core/src/main/java/hudson/model/Executor.java` | lockフィールド（96行目）でReadWriteLockによる排他制御を確認 |

**読解のコツ**: Executorは Thread を継承している（93行目）。スレッドとしてのライフサイクルとビルド実行のライフサイクルの両方を意識する。

#### Step 2: 中断処理を理解する

interrupt()メソッドの複数オーバーロードを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Executor.java | `core/src/main/java/hudson/model/Executor.java` | interrupt()（158-188行目）の自己中断と外部中断の分岐 |
| 2-2 | Executor.java | `core/src/main/java/hudson/model/Executor.java` | interrupt(Result, CauseOfInterruption...)（218-248行目）の詳細処理 |

**主要処理フロー**:
1. **158-188行目**: interrupt()の基本実装
2. **160-184行目**: 自己中断の場合はsuper.interrupt()のみ
3. **186行目**: 外部中断の場合はinterrupt(Result.ABORTED)呼び出し
4. **218-248行目**: 詳細な中断処理（ログ、ロック、状態更新）

#### Step 3: 中断結果の取得を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Executor.java | `core/src/main/java/hudson/model/Executor.java` | abortResult()（250-268行目）で中断結果を取得 |
| 3-2 | Executor.java | `core/src/main/java/hudson/model/Executor.java` | recordCauseOfInterruption()（275-298行目）で中断原因を記録 |

### プログラム呼び出し階層図

```
Executor extends Thread
    │
    ├─ owner: Computer
    │      └─ 所属ノード
    │
    ├─ queue: Queue
    │      └─ ビルドキュー参照
    │
    ├─ executable: Queue.Executable
    │      └─ 実行中ビルド
    │
    ├─ workUnit: WorkUnit
    │      └─ 割り当てワーク
    │
    └─ asynchronousExecution: AsynchronousExecution
           └─ 非同期実行管理
```

### データフロー図

```
[入力]                      [処理]                           [出力]

Queue.JobOffer ─────────▶ workUnit設定 ──────────────▶ スレッド開始
                                │
                                ▼
WorkUnit ──────────────▶ Executable生成 ─────────────▶ ビルド実行
                                │
                                ▼
interrupt()要求 ────────▶ 中断処理 ────────────────▶ Result設定
                                │
                                ▼
完了 ──────────────────▶ クリーンアップ ─────────────▶ アイドル状態
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Executor.java | `core/src/main/java/hudson/model/Executor.java` | ソース | エグゼキュータ実装 |
| Computer.java | `core/src/main/java/hudson/model/Computer.java` | ソース | コンピューター（ノードUI） |
| WorkUnit.java | `core/src/main/java/hudson/model/queue/WorkUnit.java` | ソース | ワークユニット |
| AsynchronousExecution.java | `core/src/main/java/jenkins/model/queue/AsynchronousExecution.java` | ソース | 非同期実行 |
| CauseOfInterruption.java | `core/src/main/java/jenkins/model/CauseOfInterruption.java` | ソース | 中断原因 |
| executors.jelly | `core/src/main/resources/hudson/model/Computer/executors.jelly` | テンプレート | エグゼキュータ表示 |
