# API設計書

## 概要

本ドキュメントは、Julia言語ランタイムが提供するC言語レベルの公開API（Embedding API / C API）の設計を記載する。Julia言語はWebアプリケーションフレームワークではなく、プログラミング言語のランタイムであるため、本書で扱う「API」はHTTPエンドポイントではなく、C言語の共有ライブラリ（`libjulia`）が外部プログラムに対してエクスポートする関数群（`JL_DLLEXPORT`マーカー付き関数）を指す。

これらのAPIは主に以下の用途で使用される:
- C/C++プログラムへのJuliaの埋め込み（Embedding）
- Julia拡張モジュールの開発
- ガーベジコレクション（GC）の拡張
- 外部ツールとの連携

主要なヘッダーファイル:
- `src/julia.h` - コアランタイムAPI
- `src/julia_gcext.h` - GC拡張API

## 共通仕様

### ベースインターフェース

```
C言語共有ライブラリ: libjulia
ヘッダーファイル: julia.h, julia_gcext.h
エクスポートマーカー: JL_DLLEXPORT
```

### 呼び出し規約

- 全ての公開API関数は `JL_DLLEXPORT` マーカーで修飾されている
- 関数名は `jl_` プレフィックスで統一されている
- スレッドセーフティは関数ごとに `JL_NOTSAFEPOINT` アノテーションで示される
- GCルーティング規約は `JL_PROPAGATES_ROOT`, `JL_ROOTING_ARGUMENT`, `JL_ROOTED_ARGUMENT`, `JL_MAYBE_UNROOTED` アノテーションで示される

### 共通型定義

| 型名 | 説明 |
| --- | --- |
| `jl_value_t *` | Julia値への汎用ポインタ（全てのJuliaオブジェクトの基底型） |
| `jl_datatype_t *` | Julia型オブジェクト |
| `jl_module_t *` | Juliaモジュールオブジェクト |
| `jl_sym_t *` | Juliaシンボル |
| `jl_array_t *` | Julia配列 |
| `jl_genericmemory_t *` | Julia汎用メモリ |
| `jl_genericmemoryref_t` | Julia汎用メモリ参照 |
| `jl_svec_t *` | Julia SimpleVector |
| `jl_task_t *` | Juliaタスク |
| `jl_ptls_t` | Per-Thread Local State |
| `jl_binding_t *` | モジュールバインディング |
| `jl_method_t *` | Juliaメソッド |
| `jl_method_instance_t *` | メソッドインスタンス |
| `jl_code_info_t *` | コード情報 |
| `jl_code_instance_t *` | コードインスタンス |
| `jl_weakref_t *` | 弱参照 |
| `jl_libhandle` | 動的ライブラリハンドル (`void *` 互換) |

### 共通エラーハンドリング

Julia C APIでは、エラーは例外（`jl_throw`）として発生する。安全にエラーを処理するためには `JL_TRY` / `JL_CATCH` マクロを使用する。

| 関数 | 説明 |
| --- | --- |
| `jl_exception_occurred()` | 発生中の例外を返す（なければNULL） |
| `jl_exception_clear()` | 例外状態をクリア |
| `jl_current_exception(ct)` | 現在のタスクで処理中の例外を返す |

## API一覧

| カテゴリ | 関数名 | 説明 |
| --- | --- | --- |
| 初期化・終了 | `jl_init` | Juliaランタイムの初期化 |
| 初期化・終了 | `jl_init_with_image_file` | システムイメージ指定付きの初期化 |
| 初期化・終了 | `jl_init_with_image_handle` | ハンドル指定付きの初期化 |
| 初期化・終了 | `jl_is_initialized` | 初期化済みかどうかの確認 |
| 初期化・終了 | `jl_atexit_hook` | 終了フックの実行 |
| 初期化・終了 | `jl_exit` | プロセス終了 |
| 初期化・終了 | `jl_adopt_thread` | 外部スレッドのJuliaランタイムへの登録 |
| 評価・実行 | `jl_eval_string` | 文字列のJuliaコードを評価 |
| 評価・実行 | `jl_call` | Julia関数の呼び出し（可変引数） |
| 評価・実行 | `jl_call0` - `jl_call4` | Julia関数の呼び出し（固定引数） |
| 評価・実行 | `jl_apply_generic` | 汎用関数適用 |
| 評価・実行 | `jl_invoke` | メソッドインスタンス指定の呼び出し |
| 評価・実行 | `jl_toplevel_eval` | トップレベル式の評価 |
| 評価・実行 | `jl_toplevel_eval_in` | モジュール指定のトップレベル評価 |
| 評価・実行 | `jl_load` | ファイルの読み込みと評価 |
| 型・値操作 | `jl_typeof` | オブジェクトの型を取得 |
| 型・値操作 | `jl_subtype` | サブタイプ判定 |
| 型・値操作 | `jl_isa` | 型チェック |
| 型・値操作 | `jl_types_equal` | 型の等価判定 |
| 型・値操作 | `jl_egal` | オブジェクトの等価判定 |
| 型・値操作 | `jl_object_id` | オブジェクトIDの取得 |
| 型・値操作 | `jl_new_datatype` | 新しいデータ型の作成 |
| 型・値操作 | `jl_new_primitivetype` | 新しいプリミティブ型の作成 |
| 型・値操作 | `jl_apply_type` | パラメトリック型の適用 |
| ボックス化・アンボックス化 | `jl_box_bool` - `jl_box_float64` | C値からJulia値への変換 |
| ボックス化・アンボックス化 | `jl_unbox_bool` - `jl_unbox_float64` | Julia値からC値への変換 |
| ボックス化・アンボックス化 | `jl_box_voidpointer` | voidポインタのボックス化 |
| 配列操作 | `jl_alloc_array_1d` - `jl_alloc_array_nd` | 配列の割り当て |
| 配列操作 | `jl_ptr_to_array_1d` | 既存バッファからの1D配列作成 |
| 配列操作 | `jl_ptr_to_array` | 既存バッファからの多次元配列作成 |
| 配列操作 | `jl_array_grow_end` | 配列末尾の拡張 |
| 配列操作 | `jl_array_del_end` | 配列末尾の削除 |
| 文字列操作 | `jl_cstr_to_string` | C文字列からJulia文字列への変換 |
| 文字列操作 | `jl_pchar_to_string` | 長さ指定文字列からJulia文字列への変換 |
| 文字列操作 | `jl_string_ptr` | Julia文字列からCポインタの取得 |
| 文字列操作 | `jl_alloc_string` | 指定長のJulia文字列の割り当て |
| シンボル | `jl_symbol` | 文字列からシンボルの作成 |
| シンボル | `jl_symbol_n` | 長さ指定文字列からシンボルの作成 |
| シンボル | `jl_gensym` | 一意シンボルの生成 |
| モジュール・バインディング | `jl_new_module` | 新しいモジュールの作成 |
| モジュール・バインディング | `jl_get_global` | グローバル変数の取得 |
| モジュール・バインディング | `jl_set_global` | グローバル変数の設定 |
| モジュール・バインディング | `jl_set_const` | 定数の設定 |
| モジュール・バインディング | `jl_get_binding` | バインディングの取得 |
| モジュール・バインディング | `jl_boundp` | バインディング存在確認 |
| モジュール・バインディング | `jl_is_const` | 定数判定 |
| モジュール・バインディング | `jl_module_import` | モジュールインポート |
| モジュール・バインディング | `jl_module_using` | モジュールusing |
| 構造体フィールド | `jl_get_nth_field` | N番目のフィールドを取得 |
| 構造体フィールド | `jl_set_nth_field` | N番目のフィールドを設定 |
| 構造体フィールド | `jl_field_index` | フィールド名からインデックスを取得 |
| 構造体フィールド | `jl_get_field` | フィールド名で値を取得 |
| 構造体フィールド | `jl_field_isdefined` | フィールド定義済み判定 |
| メモリ・GC | `jl_gc_collect` | GCの手動実行 |
| メモリ・GC | `jl_gc_enable` | GCの有効化・無効化 |
| メモリ・GC | `jl_gc_is_enabled` | GC有効状態の確認 |
| メモリ・GC | `jl_gc_safepoint` | GCセーフポイント |
| メモリ・GC | `jl_gc_add_finalizer` | ファイナライザの追加 |
| メモリ・GC | `jl_gc_add_ptr_finalizer` | Cポインタファイナライザの追加 |
| メモリ・GC | `jl_gc_new_weakref` | 弱参照の作成 |
| タスク | `jl_new_task` | 新しいタスクの作成 |
| タスク | `jl_switchto` | タスクの切り替え |
| タスク | `jl_get_current_task` | 現在のタスクを取得 |
| タスク | `jl_set_task_tid` | タスクのスレッドID設定 |
| タスク | `jl_task_wait_empty` | 全タスクの完了待ち |
| 例外 | `jl_error` | エラーの発生 |
| 例外 | `jl_errorf` | フォーマット付きエラーの発生 |
| 例外 | `jl_throw` | Julia例外のスロー |
| 例外 | `jl_rethrow` | 例外の再スロー |
| 例外 | `jl_exception_occurred` | 発生中の例外を取得 |
| 例外 | `jl_exception_clear` | 例外状態のクリア |
| パース・ローワリング | `jl_parse_all` | テキストのパース |
| パース・ローワリング | `jl_parse_string` | 文字列のパース |
| パース・ローワリング | `jl_lower` | 式のローワリング |
| 動的ライブラリ | `jl_load_dynamic_library` | 動的ライブラリの読み込み |
| 動的ライブラリ | `jl_dlopen` | dlopen相当 |
| 動的ライブラリ | `jl_dlclose` | dlclose相当 |
| 動的ライブラリ | `jl_dlsym` | dlsym相当 |
| システム情報 | `jl_cpu_threads` | CPU スレッド数の取得 |
| システム情報 | `jl_getpagesize` | ページサイズの取得 |
| システム情報 | `jl_ver_string` | バージョン文字列の取得 |
| システム情報 | `jl_get_UNAME` | OS名の取得 |
| システム情報 | `jl_get_ARCH` | アーキテクチャ名の取得 |
| I/O | `jl_printf` | Julia出力ストリームへのprintf |
| I/O | `jl_uv_puts` | UV ストリームへの出力 |
| I/O | `jl_flush_cstdio` | C標準I/Oのフラッシュ |
| I/O | `jl_static_show` | 値の静的表示 |
| シグナル | `jl_install_sigint_handler` | SIGINTハンドラのインストール |
| シグナル | `jl_sigatomic_begin` | シグナルアトミック区間の開始 |
| シグナル | `jl_sigatomic_end` | シグナルアトミック区間の終了 |
| GC拡張 | `jl_gc_set_cb_root_scanner` | ルートスキャナコールバックの設定 |
| GC拡張 | `jl_gc_set_cb_pre_gc` | GC前コールバックの設定 |
| GC拡張 | `jl_gc_set_cb_post_gc` | GC後コールバックの設定 |
| GC拡張 | `jl_new_foreign_type` | 外部型の作成 |
| GC拡張 | `jl_gc_alloc_typed` | 型指定割り当て |
| GC拡張 | `jl_gc_mark_queue_obj` | GCマークキューへの追加 |
| GC拡張 | `jl_gc_enable_conservative_gc_support` | 保守的GCサポートの有効化 |

## 各API関数定義

### 初期化・終了

#### 1. jl_init

Juliaランタイムを初期化する。デフォルトのシステムイメージを使用する。プログラム開始時に必ず1回呼び出す必要がある。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_init(void)` |
| ヘッダー | `julia.h` (行 2161) |
| スレッドセーフティ | 非スレッドセーフ（1回のみ呼び出し可能） |

**パラメータ**

なし

**戻り値**

なし（void）

**使用例**

```c
#include <julia.h>

int main(int argc, char *argv[]) {
    jl_init();
    // ... Julia APIの使用 ...
    jl_atexit_hook(0);
    return 0;
}
```

---

#### 2. jl_init_with_image_file

指定したシステムイメージファイルを使用してJuliaランタイムを初期化する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_init_with_image_file(const char *julia_bindir, const char *image_path)` |
| ヘッダー | `julia.h` (行 2162-2163) |
| スレッドセーフティ | 非スレッドセーフ（1回のみ呼び出し可能） |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| julia_bindir | `const char *` | ○ | Juliaバイナリのディレクトリパス |
| image_path | `const char *` | ○ | システムイメージファイルのパス |

**戻り値**

なし（void）

---

#### 3. jl_is_initialized

Juliaランタイムが初期化済みかどうかを返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `int jl_is_initialized(void)` |
| ヘッダー | `julia.h` (行 2166) |
| スレッドセーフティ | スレッドセーフ |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `int` | 初期化済みなら非ゼロ、未初期化なら0 |

---

#### 4. jl_atexit_hook

Juliaランタイムの終了処理を実行する。プロセス終了前に呼び出す必要がある。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_atexit_hook(int status)` |
| ヘッダー | `julia.h` (行 2167) |
| スレッドセーフティ | 非スレッドセーフ |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| status | `int` | ○ | 終了ステータスコード |

---

#### 5. jl_adopt_thread

外部で作成されたスレッドをJuliaランタイムに登録する。Julia APIを使用する前に呼び出す必要がある。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_gcframe_t **jl_adopt_thread(void)` |
| ヘッダー | `julia.h` (行 2173) |
| スレッドセーフティ | スレッドセーフ |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_gcframe_t **` | GCフレームスタックへのポインタ |

---

### 評価・実行

#### 6. jl_eval_string

文字列として渡されたJuliaコードを評価し、結果を返す。主要な埋め込みインターフェースの一つ。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_value_t *jl_eval_string(const char *str)` |
| ヘッダー | `julia.h` (行 2229) |
| スレッドセーフティ | タスクコンテキスト必須 |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| str | `const char *` | ○ | 評価するJuliaコードの文字列 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_value_t *` | 評価結果のJulia値。エラー時はNULL |

**使用例**

```c
jl_value_t *result = jl_eval_string("sqrt(2.0)");
if (jl_exception_occurred()) {
    // エラー処理
}
double val = jl_unbox_float64(result);
```

---

#### 7. jl_call / jl_call0 - jl_call4

Julia関数を引数付きで呼び出す。`jl_call` は可変長引数、`jl_call0`-`jl_call4` は固定引数数版。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_value_t *jl_call(jl_value_t *f, jl_value_t **args, uint32_t nargs)` |
| ヘッダー | `julia.h` (行 2288-2296) |
| GCルーティング | 引数は `JL_MAYBE_UNROOTED` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| f | `jl_value_t *` | ○ | 呼び出す関数 |
| args | `jl_value_t **` | ○ (jl_call) | 引数配列 |
| nargs | `uint32_t` | ○ (jl_call) | 引数の数 |
| a, b, c, d | `jl_value_t *` | ○ (各バリアント) | 各引数 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_value_t *` | 関数呼び出しの結果。例外発生時はNULL |

**使用例**

```c
jl_function_t *func = jl_get_function(jl_base_module, "println");
jl_value_t *arg = jl_cstr_to_string("Hello from C!");
jl_call1(func, arg);
```

---

#### 8. jl_apply_generic

汎用関数適用。型ディスパッチを含む完全な関数呼び出し。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_value_t *jl_apply_generic(jl_value_t *F, jl_value_t **args, uint32_t nargs)` |
| ヘッダー | `julia.h` (行 2278) |
| スレッドセーフティ | タスクコンテキスト必須 |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| F | `jl_value_t *` | ○ | 呼び出す関数オブジェクト |
| args | `jl_value_t **` | ○ | 引数配列 |
| nargs | `uint32_t` | ○ | 引数の数 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_value_t *` | 関数呼び出しの結果 |

---

#### 9. jl_toplevel_eval_in

指定したモジュールのコンテキストでトップレベル式を評価する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_value_t *jl_toplevel_eval_in(jl_module_t *m, jl_value_t *ex)` |
| ヘッダー | `julia.h` (行 2227) |
| スレッドセーフティ | タスクコンテキスト必須 |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| m | `jl_module_t *` | ○ | 評価対象のモジュール |
| ex | `jl_value_t *` | ○ | 評価する式 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_value_t *` | 評価結果 |

---

### 型・値操作

#### 10. jl_subtype

型aが型bのサブタイプであるかを判定する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `int jl_subtype(jl_value_t *a, jl_value_t *b)` |
| ヘッダー | `julia.h` (行 1612) |
| スレッドセーフティ | タスクコンテキスト必須 |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| a | `jl_value_t *` | ○ | 検査対象の型 |
| b | `jl_value_t *` | ○ | スーパータイプ候補 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `int` | サブタイプであれば非ゼロ、そうでなければ0 |

---

#### 11. jl_isa

値aが型tのインスタンスであるかを判定する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `int jl_isa(jl_value_t *a, jl_value_t *t)` |
| ヘッダー | `julia.h` (行 1805) |
| スレッドセーフティ | タスクコンテキスト必須 |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| a | `jl_value_t *` | ○ | 検査対象の値 |
| t | `jl_value_t *` | ○ | 型 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `int` | インスタンスであれば非ゼロ |

---

#### 12. jl_egal

二つのJulia値が `===` の意味で等価であるかを判定する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `int jl_egal(const jl_value_t *a, const jl_value_t *b)` |
| ヘッダー | `julia.h` (行 1770) |
| スレッドセーフティ | `JL_NOTSAFEPOINT` |
| GCルーティング | 引数は `JL_MAYBE_UNROOTED` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| a | `const jl_value_t *` | ○ | 比較対象1 |
| b | `const jl_value_t *` | ○ | 比較対象2 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `int` | 等価であれば非ゼロ |

---

#### 13. jl_object_id

Juliaオブジェクトのハッシュ値（objectid）を返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `uintptr_t jl_object_id(jl_value_t *v)` |
| ヘッダー | `julia.h` (行 1774) |
| スレッドセーフティ | `JL_NOTSAFEPOINT` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| v | `jl_value_t *` | ○ | 対象オブジェクト |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `uintptr_t` | オブジェクトのハッシュ値 |

---

### ボックス化・アンボックス化

#### 14. jl_box_* / jl_unbox_* シリーズ

C言語のプリミティブ値とJulia値の相互変換を行う。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| ヘッダー | `julia.h` (行 1894-1922) |
| スレッドセーフティ | `JL_NOTSAFEPOINT` (ほぼ全て) |

**ボックス化関数（C値 -> Julia値）**

| 関数 | 入力型 | Julia型 |
| --- | --- | --- |
| `jl_box_bool(int8_t x)` | `int8_t` | `Bool` |
| `jl_box_int8(int8_t x)` | `int8_t` | `Int8` |
| `jl_box_uint8(uint8_t x)` | `uint8_t` | `UInt8` |
| `jl_box_int16(int16_t x)` | `int16_t` | `Int16` |
| `jl_box_uint16(uint16_t x)` | `uint16_t` | `UInt16` |
| `jl_box_int32(int32_t x)` | `int32_t` | `Int32` |
| `jl_box_uint32(uint32_t x)` | `uint32_t` | `UInt32` |
| `jl_box_int64(int64_t x)` | `int64_t` | `Int64` |
| `jl_box_uint64(uint64_t x)` | `uint64_t` | `UInt64` |
| `jl_box_float32(float x)` | `float` | `Float32` |
| `jl_box_float64(double x)` | `double` | `Float64` |
| `jl_box_char(uint32_t x)` | `uint32_t` | `Char` |
| `jl_box_voidpointer(void *x)` | `void *` | `Ptr{Cvoid}` |

**アンボックス化関数（Julia値 -> C値）**

| 関数 | 出力型 | Julia型 |
| --- | --- | --- |
| `jl_unbox_bool(jl_value_t *v)` | `int8_t` | `Bool` |
| `jl_unbox_int8(jl_value_t *v)` | `int8_t` | `Int8` |
| `jl_unbox_uint8(jl_value_t *v)` | `uint8_t` | `UInt8` |
| `jl_unbox_int16(jl_value_t *v)` | `int16_t` | `Int16` |
| `jl_unbox_uint16(jl_value_t *v)` | `uint16_t` | `UInt16` |
| `jl_unbox_int32(jl_value_t *v)` | `int32_t` | `Int32` |
| `jl_unbox_uint32(jl_value_t *v)` | `uint32_t` | `UInt32` |
| `jl_unbox_int64(jl_value_t *v)` | `int64_t` | `Int64` |
| `jl_unbox_uint64(jl_value_t *v)` | `uint64_t` | `UInt64` |
| `jl_unbox_float32(jl_value_t *v)` | `float` | `Float32` |
| `jl_unbox_float64(jl_value_t *v)` | `double` | `Float64` |
| `jl_unbox_voidpointer(jl_value_t *v)` | `void *` | `Ptr{Cvoid}` |

**プラットフォーム依存マクロ**

64ビットプラットフォームでは:
- `jl_box_long(x)` = `jl_box_int64(x)`
- `jl_box_ulong(x)` = `jl_box_uint64(x)`
- `jl_unbox_long(x)` = `jl_unbox_int64(x)`
- `jl_unbox_ulong(x)` = `jl_unbox_uint64(x)`

32ビットプラットフォームでは:
- `jl_box_long(x)` = `jl_box_int32(x)`
- `jl_box_ulong(x)` = `jl_box_uint32(x)`
- `jl_unbox_long(x)` = `jl_unbox_int32(x)`
- `jl_unbox_ulong(x)` = `jl_unbox_uint32(x)`

---

### 配列操作

#### 15. jl_alloc_array_1d / jl_alloc_array_2d / jl_alloc_array_3d

指定した型と次元でJulia配列を割り当てる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_array_t *jl_alloc_array_1d(jl_value_t *atype, size_t nr)` |
| ヘッダー | `julia.h` (行 1966-1968) |
| スレッドセーフティ | タスクコンテキスト必須 |

**パラメータ (1D)**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| atype | `jl_value_t *` | ○ | 配列型（例: `jl_apply_array_type(jl_float64_type, 1)`） |
| nr | `size_t` | ○ | 要素数 |

**パラメータ (2D)**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| atype | `jl_value_t *` | ○ | 配列型 |
| nr | `size_t` | ○ | 行数 |
| nc | `size_t` | ○ | 列数 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_array_t *` | 割り当てられた配列 |

**使用例**

```c
jl_value_t *array_type = jl_apply_array_type((jl_value_t*)jl_float64_type, 1);
jl_array_t *arr = jl_alloc_array_1d(array_type, 10);
```

---

#### 16. jl_ptr_to_array_1d

既存のCバッファから1次元Julia配列を作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_array_t *jl_ptr_to_array_1d(jl_value_t *atype, void *data, size_t nel, int own_buffer)` |
| ヘッダー | `julia.h` (行 1961-1962) |
| スレッドセーフティ | タスクコンテキスト必須 |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| atype | `jl_value_t *` | ○ | 配列型 |
| data | `void *` | ○ | データバッファへのポインタ |
| nel | `size_t` | ○ | 要素数 |
| own_buffer | `int` | ○ | JuliaがバッファのGC管理を行うか（1=管理する） |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_array_t *` | 作成された配列 |

---

#### 17. jl_array_grow_end / jl_array_del_end

配列の末尾の拡張・縮小を行う。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_array_grow_end(jl_array_t *a, size_t inc)` / `void jl_array_del_end(jl_array_t *a, size_t dec)` |
| ヘッダー | `julia.h` (行 1976-1977) |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| a | `jl_array_t *` | ○ | 対象配列 |
| inc / dec | `size_t` | ○ | 増加 / 減少する要素数 |

---

### 文字列操作

#### 18. jl_cstr_to_string

NUL終端のC文字列からJulia文字列を作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_value_t *jl_cstr_to_string(const char *str)` |
| ヘッダー | `julia.h` (行 1972) |
| スレッドセーフティ | タスクコンテキスト必須 |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| str | `const char *` | ○ | NUL終端のC文字列 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_value_t *` | Julia String オブジェクト |

---

#### 19. jl_string_ptr

Julia文字列の内部バッファへのCポインタを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `const char *jl_string_ptr(jl_value_t *s)` |
| ヘッダー | `julia.h` (行 2009) |
| スレッドセーフティ | `JL_NOTSAFEPOINT` 相当 |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| s | `jl_value_t *` | ○ | Julia文字列 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `const char *` | 文字列データへのポインタ（NUL終端保証あり） |

---

### シンボル

#### 20. jl_symbol

C文字列からJuliaシンボルを作成する。同じ名前のシンボルは同一オブジェクトとなる（インターン化）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_sym_t *jl_symbol(const char *str)` |
| ヘッダー | `julia.h` (行 1876) |
| スレッドセーフティ | `JL_NOTSAFEPOINT` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| str | `const char *` | ○ | シンボル名（NUL終端） |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_sym_t *` | インターン化されたシンボル |

---

### モジュール・バインディング

#### 21. jl_new_module

新しいJuliaモジュールを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_module_t *jl_new_module(jl_sym_t *name, jl_module_t *parent)` |
| ヘッダー | `julia.h` (行 2012) |
| スレッドセーフティ | タスクコンテキスト必須 |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| name | `jl_sym_t *` | ○ | モジュール名 |
| parent | `jl_module_t *` | ○ | 親モジュール |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_module_t *` | 作成されたモジュール |

---

#### 22. jl_get_global / jl_set_global

モジュール内のグローバル変数の取得・設定を行う。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ (get) | `jl_value_t *jl_get_global(jl_module_t *m, jl_sym_t *var)` |
| シグネチャ (set) | `void jl_set_global(jl_module_t *m, jl_sym_t *var, jl_value_t *val)` |
| ヘッダー | `julia.h` (行 2037-2038) |
| GCルーティング | set: m は `JL_ROOTING_ARGUMENT`, val は `JL_ROOTED_ARGUMENT` |

**パラメータ (get)**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| m | `jl_module_t *` | ○ | 対象モジュール |
| var | `jl_sym_t *` | ○ | 変数名のシンボル |

**パラメータ (set)**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| m | `jl_module_t *` | ○ | 対象モジュール |
| var | `jl_sym_t *` | ○ | 変数名のシンボル |
| val | `jl_value_t *` | ○ | 設定する値 |

---

#### 23. jl_set_const

モジュール内で定数を宣言・設定する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_set_const(jl_module_t *m, jl_sym_t *var, jl_value_t *val)` |
| ヘッダー | `julia.h` (行 2039) |
| GCルーティング | m は `JL_ROOTING_ARGUMENT`, val は `JL_ROOTED_ARGUMENT` |

---

### 構造体フィールド操作

#### 24. jl_get_nth_field / jl_set_nth_field

構造体のN番目のフィールドの取得・設定を行う。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ (get) | `jl_value_t *jl_get_nth_field(jl_value_t *v, size_t i)` |
| シグネチャ (set) | `void jl_set_nth_field(jl_value_t *v, size_t i, jl_value_t *rhs)` |
| ヘッダー | `julia.h` (行 1948, 1952) |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| v | `jl_value_t *` | ○ | 構造体オブジェクト |
| i | `size_t` | ○ | フィールドインデックス（0始まり） |
| rhs | `jl_value_t *` | ○ (set) | 設定する値 |

---

#### 25. jl_get_field

フィールド名を指定して構造体フィールドの値を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_value_t *jl_get_field(jl_value_t *o, const char *fld)` |
| ヘッダー | `julia.h` (行 1955) |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| o | `jl_value_t *` | ○ | 構造体オブジェクト |
| fld | `const char *` | ○ | フィールド名 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_value_t *` | フィールドの値 |

---

### メモリ・GC

#### 26. jl_gc_add_finalizer

Juliaオブジェクトにファイナライザを追加する。オブジェクトがGC回収される際にファイナライザが呼び出される。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_gc_add_finalizer(jl_value_t *v, jl_value_t *f)` |
| ヘッダー | `julia.h` (行 1158) |
| スレッドセーフティ | `JL_NOTSAFEPOINT` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| v | `jl_value_t *` | ○ | ファイナライザを追加するオブジェクト |
| f | `jl_value_t *` | ○ | ファイナライザ関数（Julia関数） |

---

#### 27. jl_gc_add_ptr_finalizer

Cポインタベースのファイナライザを追加する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_gc_add_ptr_finalizer(jl_ptls_t ptls, jl_value_t *v, void *f)` |
| ヘッダー | `julia.h` (行 1159) |
| スレッドセーフティ | `JL_NOTSAFEPOINT` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| ptls | `jl_ptls_t` | ○ | スレッドローカルステート |
| v | `jl_value_t *` | ○ | 対象オブジェクト |
| f | `void *` | ○ | Cファイナライザ関数ポインタ |

---

#### 28. jl_gc_new_weakref

新しい弱参照を作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_weakref_t *jl_gc_new_weakref(jl_value_t *value)` |
| ヘッダー | `julia.h` (行 1168) |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| value | `jl_value_t *` | ○ | 弱参照の対象値 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_weakref_t *` | 作成された弱参照 |

---

#### 29. jl_gc_safepoint

GCセーフポイントを挿入する。長時間実行のCコードの中でGCが実行される機会を提供する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_gc_safepoint(void)` |
| ヘッダー | `julia.h` (行 1170) |

---

### タスク

#### 30. jl_new_task

新しいJuliaタスク（コルーチン）を作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_task_t *jl_new_task(jl_value_t *start, jl_value_t *completion_future, size_t ssize)` |
| ヘッダー | `julia.h` (行 2323) |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| start | `jl_value_t *` | ○ | タスクで実行する関数 |
| completion_future | `jl_value_t *` | ○ | 完了通知用のFuture |
| ssize | `size_t` | ○ | スタックサイズ（0でデフォルト） |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_task_t *` | 作成されたタスク |

---

#### 31. jl_get_current_task

現在実行中のタスクを返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_task_t *jl_get_current_task(void)` |
| ヘッダー | `julia.h` (行 2631) |
| スレッドセーフティ | `JL_NOTSAFEPOINT`, `JL_GLOBALLY_ROOTED` |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_task_t *` | 現在のタスク |

---

### 例外

#### 32. jl_error / jl_errorf

C文字列からJuliaエラーを発生させる。この関数は返らない（`JL_NORETURN`）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_error(const char *str)` |
| シグネチャ | `void jl_errorf(const char *fmt, ...)` |
| ヘッダー | `julia.h` (行 2086-2087) |
| 属性 | `JL_NORETURN` |

**パラメータ (jl_error)**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| str | `const char *` | ○ | エラーメッセージ |

**パラメータ (jl_errorf)**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| fmt | `const char *` | ○ | printf形式のフォーマット文字列 |
| ... | 可変引数 | - | フォーマット引数 |

---

#### 33. jl_throw

Julia例外オブジェクトをスローする。JL_TRY/JL_CATCH で捕捉可能。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_throw(jl_value_t *e)` |
| ヘッダー | `julia.h` (行 2327) |
| 属性 | `JL_NORETURN` |
| GCルーティング | e は `JL_MAYBE_UNROOTED` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| e | `jl_value_t *` | ○ | スローする例外オブジェクト |

---

#### 34. jl_exception_occurred / jl_exception_clear

例外の発生確認とクリアを行う。Embedding API使用時の標準的なエラーチェック方法。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ (occurred) | `jl_value_t *jl_exception_occurred(void)` |
| シグネチャ (clear) | `void jl_exception_clear(void)` |
| ヘッダー | `julia.h` (行 2349-2350) |

**戻り値 (occurred)**

| 型 | 説明 |
| --- | --- |
| `jl_value_t *` | 発生中の例外。なければNULL |

---

### パース・ローワリング

#### 35. jl_parse_all

テキストをJulia式としてパースする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_value_t *jl_parse_all(const char *text, size_t text_len, const char *filename, size_t filename_len, size_t lineno)` |
| ヘッダー | `julia.h` (行 2191-2192) |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| text | `const char *` | ○ | パースするテキスト |
| text_len | `size_t` | ○ | テキストの長さ |
| filename | `const char *` | ○ | ソースファイル名（エラーメッセージ用） |
| filename_len | `size_t` | ○ | ファイル名の長さ |
| lineno | `size_t` | ○ | 開始行番号 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_value_t *` | パースされた式 |

---

### 動的ライブラリ

#### 36. jl_load_dynamic_library / jl_dlopen / jl_dlclose / jl_dlsym

動的ライブラリの読み込み・シンボル解決・解放を行う。POSIX dlopenファミリーのラッパー。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ (load) | `jl_libhandle jl_load_dynamic_library(const char *fname, unsigned flags, int throw_err)` |
| シグネチャ (open) | `jl_libhandle jl_dlopen(const char *filename, unsigned flags)` |
| シグネチャ (close) | `int jl_dlclose(jl_libhandle handle)` |
| シグネチャ (sym) | `int jl_dlsym(jl_libhandle handle, const char *symbol, void **value, int throw_err, int search_deps)` |
| ヘッダー | `julia.h` (行 2220-2223) |

**フラグ定数（JL_RTLD_*）**

| 定数 | 値 | 説明 |
| --- | --- | --- |
| `JL_RTLD_LOCAL` | 1 | シンボルをローカルスコープに制限 |
| `JL_RTLD_GLOBAL` | 2 | シンボルをグローバルに公開 |
| `JL_RTLD_LAZY` | 4 | 遅延バインディング |
| `JL_RTLD_NOW` | 8 | 即時バインディング |
| `JL_RTLD_NODELETE` | 16 | dlclose時に解放しない |
| `JL_RTLD_NOLOAD` | 32 | ロードせずに確認のみ |
| `JL_RTLD_DEEPBIND` | 64 | シンボル解決を自身優先 |
| `JL_RTLD_FIRST` | 128 | 最初に見つかったシンボルを使用 |

デフォルト: `JL_RTLD_DEFAULT` = `JL_RTLD_LAZY | JL_RTLD_DEEPBIND`

---

### システム情報

#### 37. jl_cpu_threads / jl_effective_threads

利用可能なCPUスレッド数を返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `int jl_cpu_threads(void)` / `int jl_effective_threads(void)` |
| ヘッダー | `julia.h` (行 2065-2066) |
| スレッドセーフティ | `JL_NOTSAFEPOINT` |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `int` | CPUスレッド数 / 有効スレッド数 |

---

#### 38. jl_ver_string / jl_ver_major / jl_ver_minor / jl_ver_patch

Juliaのバージョン情報を返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `const char *jl_ver_string(void)` |
| ヘッダー | `julia.h` (行 2613-2617) |

**バリアント**

| 関数 | 戻り値型 | 説明 |
| --- | --- | --- |
| `jl_ver_string()` | `const char *` | バージョン文字列（例: "1.11.0"） |
| `jl_ver_major()` | `int` | メジャーバージョン |
| `jl_ver_minor()` | `int` | マイナーバージョン |
| `jl_ver_patch()` | `int` | パッチバージョン |
| `jl_ver_is_release()` | `int` | リリースビルドなら非ゼロ |

---

### I/O

#### 39. jl_printf / jl_uv_puts

JuliaのUV出力ストリームに文字列を出力する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `int jl_printf(struct uv_stream_s *s, const char *format, ...)` |
| シグネチャ | `void jl_uv_puts(struct uv_stream_s *stream, const char *str, size_t n)` |
| ヘッダー | `julia.h` (行 2485-2488) |

---

#### 40. jl_static_show

Julia値を文字列として出力ストリームに表示する。GCセーフポイント外で安全に使用可能。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `size_t jl_static_show(JL_STREAM *out, jl_value_t *v)` |
| ヘッダー | `julia.h` (行 2507) |
| スレッドセーフティ | `JL_NOTSAFEPOINT` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| out | `JL_STREAM *` | ○ | 出力ストリーム |
| v | `jl_value_t *` | ○ | 表示するJulia値 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `size_t` | 書き込まれたバイト数 |

---

### シグナル

#### 41. jl_sigatomic_begin / jl_sigatomic_end

シグナルアトミック区間の開始・終了を行う。この区間内ではシグナル処理が遅延される。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_sigatomic_begin(void)` / `void jl_sigatomic_end(void)` |
| ヘッダー | `julia.h` (行 2301-2302) |

---

### GC拡張 (julia_gcext.h)

#### 42. jl_gc_set_cb_root_scanner / jl_gc_set_cb_task_scanner

GCルートスキャン時に呼び出されるコールバックを設定する。外部のルートオブジェクトをGCに認識させるために使用する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_gc_set_cb_root_scanner(jl_gc_cb_root_scanner_t cb, int enable)` |
| シグネチャ | `void jl_gc_set_cb_task_scanner(jl_gc_cb_task_scanner_t cb, int enable)` |
| ヘッダー | `julia_gcext.h` (行 28-29) |

**コールバック型**

```c
typedef void (*jl_gc_cb_root_scanner_t)(int full);
typedef void (*jl_gc_cb_task_scanner_t)(jl_task_t *task, int full);
```

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| cb | コールバック関数ポインタ | ○ | スキャナコールバック |
| enable | `int` | ○ | 1=有効化, 0=無効化 |

---

#### 43. jl_gc_set_cb_pre_gc / jl_gc_set_cb_post_gc

GC実行前後に呼び出されるコールバックを設定する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void jl_gc_set_cb_pre_gc(jl_gc_cb_pre_gc_t cb, int enable)` |
| シグネチャ | `void jl_gc_set_cb_post_gc(jl_gc_cb_post_gc_t cb, int enable)` |
| ヘッダー | `julia_gcext.h` (行 30-31) |

**コールバック型**

```c
typedef void (*jl_gc_cb_pre_gc_t)(int full);
typedef void (*jl_gc_cb_post_gc_t)(int full);
```

---

#### 44. jl_new_foreign_type

カスタムのマーク関数とスイープ関数を持つ外部型を作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_datatype_t *jl_new_foreign_type(jl_sym_t *name, jl_module_t *module, jl_datatype_t *super, jl_markfunc_t markfunc, jl_sweepfunc_t sweepfunc, int haspointers, int large)` |
| ヘッダー | `julia_gcext.h` (行 47-54) |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| name | `jl_sym_t *` | ○ | 型名 |
| module | `jl_module_t *` | ○ | 所属モジュール |
| super | `jl_datatype_t *` | ○ | スーパータイプ |
| markfunc | `jl_markfunc_t` | ○ | カスタムマーク関数 |
| sweepfunc | `jl_sweepfunc_t` | ○ | カスタムスイープ関数 |
| haspointers | `int` | ○ | ポインタを含むか |
| large | `int` | ○ | ラージオブジェクトか |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `jl_datatype_t *` | 作成されたデータ型 |

---

#### 45. jl_gc_alloc_typed

外部型のオブジェクトを割り当てる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `void *jl_gc_alloc_typed(jl_ptls_t ptls, size_t sz, void *ty)` |
| ヘッダー | `julia_gcext.h` (行 77) |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| ptls | `jl_ptls_t` | ○ | スレッドローカルステート |
| sz | `size_t` | ○ | 割り当てサイズ（バイト） |
| ty | `void *` | ○ | 型オブジェクト |

---

#### 46. jl_gc_mark_queue_obj / jl_gc_mark_queue_objarray

GCマークキューにオブジェクトを追加する。カスタムマーク関数またはルートスキャナ内からのみ呼び出し可能。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ (obj) | `int jl_gc_mark_queue_obj(jl_ptls_t ptls, jl_value_t *obj)` |
| シグネチャ (array) | `void jl_gc_mark_queue_objarray(jl_ptls_t ptls, jl_value_t *parent, jl_value_t **objs, size_t nobjs)` |
| ヘッダー | `julia_gcext.h` (行 82-84) |
| スレッドセーフティ | `JL_NOTSAFEPOINT` |

---

#### 47. jl_gc_enable_conservative_gc_support

保守的GCサポートを有効化する。マシンワードからオブジェクトポインタへの逆引きが可能になる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `int jl_gc_enable_conservative_gc_support(void)` |
| ヘッダー | `julia_gcext.h` (行 116) |

**戻り値**

| 型 | 説明 |
| --- | --- |
| `int` | 以前から有効化されていた場合は非ゼロ |

**注意事項**

- 有効化により完全なGCがトリガーされる場合がある
- 32ビットマシンではメモリリークのリスクがある
- 最適化コンパイラによりスタック上の参照が隠される可能性がある

---

### GenericMemory操作

#### 48. jl_alloc_genericmemory / jl_new_genericmemory

汎用メモリオブジェクトを割り当てる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_genericmemory_t *jl_alloc_genericmemory(jl_value_t *mtype, size_t nel)` |
| シグネチャ | `jl_genericmemory_t *jl_new_genericmemory(jl_value_t *mtype, jl_value_t *dim)` |
| ヘッダー | `julia.h` (行 1987, 1990) |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| mtype | `jl_value_t *` | ○ | メモリ型 |
| nel | `size_t` | ○ (alloc) | 要素数 |
| dim | `jl_value_t *` | ○ (new) | 次元（Julia値） |

---

### SimpleVector操作

#### 49. jl_svec / jl_alloc_svec

SimpleVector（固定サイズのタプル的構造）を作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_svec_t *jl_svec(size_t n, ...)` |
| シグネチャ | `jl_svec_t *jl_alloc_svec(size_t n)` |
| ヘッダー | `julia.h` (行 1868, 1872) |
| GCルーティング | `JL_MAYBE_UNROOTED` (svec) |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| n | `size_t` | ○ | 要素数 |
| ... | 可変引数 | ○ (svec) | 初期要素 |

---

### 構造体作成

#### 50. jl_new_struct / jl_new_structv

新しい構造体オブジェクトを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `jl_value_t *jl_new_struct(jl_datatype_t *type, ...)` |
| シグネチャ | `jl_value_t *jl_new_structv(jl_datatype_t *type, jl_value_t **args, uint32_t na)` |
| ヘッダー | `julia.h` (行 1863-1864) |

**パラメータ (jl_new_structv)**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| type | `jl_datatype_t *` | ○ | 構造体の型 |
| args | `jl_value_t **` | ○ | フィールド値の配列 |
| na | `uint32_t` | ○ | フィールド数 |

---

## GCルーティング規約

Julia C APIを使用する際は、GCルーティングに細心の注意が必要である。以下のマクロを使用してJulia値をGCルートとして保護する。

### GC_PUSH / GC_POP マクロ

```c
// 1-9個の値をGCルートとして保護
JL_GC_PUSH1(&val1);
JL_GC_PUSH2(&val1, &val2);
JL_GC_PUSH3(&val1, &val2, &val3);
// ... JL_GC_PUSH9まで

// 可変個数の場合
jl_value_t **args;
JL_GC_PUSHARGS(args, n);

// スコープ終了時に解除
JL_GC_POP();
```

### 例外処理マクロ

```c
JL_TRY {
    // Julia例外が発生する可能性のあるコード
    jl_value_t *result = jl_eval_string("error(\"test\")");
}
JL_CATCH {
    // 例外処理
    jl_value_t *ex = jl_current_exception(jl_current_task);
    // ...
}
```

### アノテーション一覧

| アノテーション | 意味 |
| --- | --- |
| `JL_NOTSAFEPOINT` | この関数はGCセーフポイントを含まない（GCが発生しない） |
| `JL_PROPAGATES_ROOT` | 戻り値は引数のルートを継承する |
| `JL_ROOTING_ARGUMENT` | この引数はルーティング先として機能する |
| `JL_ROOTED_ARGUMENT` | この引数はルートとして登録される |
| `JL_MAYBE_UNROOTED` | この引数はルートされていない可能性がある |
| `JL_GLOBALLY_ROOTED` | この値はグローバルにルートされている |

## 備考

### プロジェクト特性

本プロジェクトはJuliaプログラミング言語のソースリポジトリであり、一般的なWebアプリケーションとは異なる。HTTP APIやRESTエンドポイントは存在しない。ここに記載されたAPIは、`libjulia` 共有ライブラリが外部C/C++プログラムに対してエクスポートするC関数インターフェースである。

### 使用上の注意

1. **初期化**: 全てのJulia API呼び出しの前に `jl_init()` を呼び出す必要がある
2. **GCルーティング**: Julia値を扱う際は必ず `JL_GC_PUSH*` / `JL_GC_POP` で保護する
3. **例外処理**: Julia関数呼び出し後は `jl_exception_occurred()` でエラーチェックを行う
4. **スレッドセーフティ**: 外部スレッドからJulia APIを使用する前に `jl_adopt_thread()` を呼び出す
5. **終了処理**: プロセス終了前に `jl_atexit_hook()` を呼び出す

### 参照先

- `src/julia.h` - メインヘッダーファイル（388個の `JL_DLLEXPORT` 関数を含む）
- `src/julia_gcext.h` - GC拡張API
- `src/jloptions.h` - Juliaオプション構造体の定義
- Julia公式ドキュメント: Embedding Julia (https://docs.julialang.org/en/v1/manual/embedding/)
