# アーキテクチャ設計書

## 概要

本ドキュメントは、Julia言語処理系（バージョン1.14.0-DEV）のアーキテクチャ設計を記述する。Juliaはハイパフォーマンスな動的プログラミング言語であり、C/C++で実装されたランタイムシステム、LLVM基盤のJITコンパイラ、Julia自身で記述された標準ライブラリから構成される。本書ではアプリケーションレイヤー構成、モジュール構造、ミドルウェア選定、データフロー、横断的関心事について整理する。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図](./アーキテクチャ構成図.md) を参照。

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| LLVM | C++ API / 共有ライブラリリンク | JITコンパイル、ネイティブコード生成、最適化パイプライン |
| libuv | C API (`ccall`) | 非同期I/O、イベントループ、クロスプラットフォームOS抽象化 |
| OpenBLAS / libblastrampoline | BLAS/LAPACK API | 線形代数演算（行列乗算等） |
| GMP | C API (`ccall`) | 多倍長整数演算（BigInt） |
| MPFR | C API (`ccall`) | 多倍長浮動小数点演算（BigFloat） |
| PCRE2 | C API (`ccall`) | 正規表現エンジン |
| libgit2 | C API（stdlib経由） | Gitリポジトリ操作（Pkg管理） |
| libcurl / nghttp2 | C API（stdlib経由） | HTTP通信・ダウンロード |
| libssh2 / OpenSSL | C API（stdlib経由） | SSH/TLS暗号化通信 |
| SuiteSparse | C API（stdlib経由） | スパース行列演算 |
| utf8proc | C API | Unicode文字列処理 |
| libunwind | C API | スタックトレース取得（非Windows環境） |
| dSFMT | C API | 疑似乱数生成（Mersenne Twister） |
| OpenLibm | C API | 数学関数ライブラリ |

## レイヤー構成

### アーキテクチャスタイル

Juliaは階層化アーキテクチャ（Layered Architecture）を採用している。最下層にC/C++ランタイムとLLVMコード生成層があり、その上にJulia言語で記述されたCoreモジュール・Baseモジュール・標準ライブラリが積み重なる。コンパイラ（Compiler）はBase層と同格の独立モジュールとして設計されており、差し替え可能（swappable）である。ユーザーコードはこれらの上で動作する。

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| CLI / Loader | プロセス起動、ライブラリのロード、エントリポイント | `cli/loader_exe.c`, `cli/loader_lib.c`, `cli/trampolines/` |
| Runtime (C/C++) | メモリ管理、GC、型システム、スレッド管理、シグナル処理、FFI | `src/gc-*.c`, `src/task.c`, `src/threading.c`, `src/builtins.c`, `src/jltypes.c`, `src/datatype.c` |
| Codegen (C++/LLVM) | Julia IRからネイティブコードへのJITコンパイル | `src/codegen.cpp`, `src/aotcompile.cpp`, `src/cgutils.cpp`, `src/ccall.cpp`, `src/jitlayers.cpp` |
| Core (Julia) | 原始型定義、ブートストラップ、組み込み関数 | `base/boot.jl`, `base/essentials.jl`, `base/operators.jl` |
| Base (Julia) | 標準データ構造、I/O、文字列処理、数値演算、並行処理、パッケージローディング | `base/Base.jl`, `base/array.jl`, `base/dict.jl`, `base/io.jl`, `base/task.jl`, `base/loading.jl` |
| Compiler (Julia) | 型推論、抽象解釈、SSA IR、最適化、インライン化 | `Compiler/src/Compiler.jl`, `Compiler/src/abstractinterpretation.jl`, `Compiler/src/optimize.jl`, `Compiler/src/typeinfer.jl` |
| Frontend (Julia/Scheme) | パーサー、構文解析、マクロ展開、スコープ解析、IR lowering | `JuliaSyntax/src/`, `JuliaLowering/src/`, `src/flisp/`, `src/ast.scm` |
| Standard Library | 各種ドメイン固有機能の提供 | `stdlib/` 配下の約60パッケージ（Dates, Random, Test, REPL, Pkg等） |
| Support Library | 低レベルユーティリティ（ハッシュ、UTF-8、I/O） | `src/support/` |

### レイヤー間の依存関係ルール

- **依存の方向**: 上位レイヤーは下位レイヤーに依存するが、逆方向の依存は許可されない。
- **CLI/Loader -> Runtime**: ローダーはランタイムライブラリ（`libjulia`）とコード生成ライブラリ（`libjulia-codegen`）を動的にロードする。
- **Runtime -> Support Library**: ランタイムCコードはsupportライブラリのユーティリティ関数を使用する。
- **Core -> Runtime**: Julia Coreモジュールはランタイムのbuiltinsを`ccall`経由で呼び出す。
- **Base -> Core**: BaseモジュールはCoreモジュールのプリミティブ型・関数に依存する。
- **Compiler -> Base/Core**: Compilerモジュールは、Base・Coreの型定義やユーティリティを`using`で参照する。ただしCompilerは独立モジュールとして設計されており、差し替え可能。
- **stdlib -> Base**: 標準ライブラリパッケージはBaseを`using`で参照する。stdlibパッケージ間の依存は`Project.toml`/`Manifest.toml`で管理される。
- **禁止される参照**: Runtime C層からJulia層への直接的な関数呼び出し（ただし、ランタイムからJuliaコールバックを呼ぶメカニズムは例外的に存在する）。Baseから特定のstdlibへの直接依存は原則禁止。

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| 言語ランタイム | メモリ管理、GC、型システム、タスク管理、FFI | `src/` (C/C++), `base/boot.jl`, `base/essentials.jl` |
| フロントエンド | ソースコード解析、構文木構築、マクロ展開、lowering | `JuliaSyntax/`, `JuliaLowering/`, `src/flisp/`, `src/ast.c` |
| コンパイラ | 型推論、抽象解釈、最適化、コード生成 | `Compiler/`, `src/codegen.cpp`, `src/aotcompile.cpp` |
| 基本ライブラリ | データ構造、I/O、文字列、数値、並行処理 | `base/` |
| 標準ライブラリ | 日付、乱数、テスト、REPL、パッケージ管理等 | `stdlib/` |
| ビルドシステム | 依存関係管理、コンパイル、テスト | `Makefile`, `Make.inc`, `deps/`, `contrib/` |

### パッケージ構造

```
julia-master/
├── base/                          # Baseモジュール（Julia、起動時にロード）
│   ├── Base_compiler.jl           # Baseモジュールのブートストラップ開始点
│   ├── Base.jl                    # Baseモジュールの本体（includeチェーン）
│   ├── boot.jl                    # Core型定義（Any, Int, Array等）
│   ├── essentials.jl              # 基本関数（convert, promote等）
│   ├── array.jl                   # 配列操作
│   ├── dict.jl                    # 辞書型
│   ├── io.jl                      # I/Oプリミティブ
│   ├── task.jl                    # タスク（コルーチン）
│   ├── loading.jl                 # パッケージローディング（import/using）
│   ├── threads.jl                 # スレッド管理
│   ├── channels.jl                # チャネル（通信プリミティブ）
│   └── ...                        # その他多数（約160ファイル）
├── Compiler/                      # コンパイラモジュール（差し替え可能）
│   ├── src/
│   │   ├── Compiler.jl            # エントリポイント
│   │   ├── abstractinterpretation.jl  # 抽象解釈
│   │   ├── typeinfer.jl           # 型推論
│   │   ├── optimize.jl            # 最適化パス
│   │   ├── tfuncs.jl              # 型転送関数
│   │   ├── ssair/                 # SSA IR表現
│   │   └── ...
│   └── test/
├── JuliaSyntax/                   # パーサー（Julia実装）
│   └── src/
├── JuliaLowering/                 # Lowering（Julia実装）
│   └── src/
│       ├── desugaring.jl          # 構文糖衣の除去
│       ├── scope_analysis.jl      # スコープ解析
│       ├── closure_conversion.jl  # クロージャ変換
│       ├── linear_ir.jl           # 線形IR生成
│       └── ...
├── src/                           # C/C++ランタイム
│   ├── init.c                     # 初期化
│   ├── julia.h                    # メインヘッダ
│   ├── builtins.c                 # 組み込み関数
│   ├── codegen.cpp                # LLVM JITコード生成
│   ├── aotcompile.cpp             # Ahead-of-Time コンパイル
│   ├── gc-stock.c                 # Stock GC実装
│   ├── gc-mmtk.c                  # MMTk GC実装（代替）
│   ├── task.c                     # タスクスケジューリング
│   ├── threading.c                # スレッド管理
│   ├── gf.c                       # ジェネリック関数ディスパッチ
│   ├── jltypes.c                  # 型システム
│   ├── datatype.c                 # DataType操作
│   ├── flisp/                     # FemtoLisp（レガシーフロントエンド）
│   ├── support/                   # 低レベルサポートライブラリ
│   └── llvm-*.cpp                 # LLVMカスタムパス群
├── cli/                           # CLIローダー
│   ├── loader_exe.c               # 実行ファイルエントリポイント
│   ├── loader_lib.c               # ライブラリローダー
│   └── trampolines/               # トランポリンコード
├── stdlib/                        # 標準ライブラリパッケージ群
│   ├── Dates/                     # 日付・時刻
│   ├── Random/                    # 乱数
│   ├── Test/                      # テストフレームワーク
│   ├── REPL/                      # 対話型環境
│   ├── Printf/                    # フォーマット出力
│   ├── Profile/                   # プロファイリング
│   ├── Serialization/             # シリアライズ
│   ├── Sockets/                   # ネットワークソケット
│   ├── Logging/                   # ロギング
│   ├── Markdown/                  # Markdown処理
│   ├── *_jll/                     # JLLバイナリラッパー群
│   └── ...                        # 約60パッケージ
├── deps/                          # 外部依存ライブラリ
│   ├── llvm.mk                    # LLVMビルド設定
│   ├── openblas.mk                # OpenBLASビルド設定
│   ├── gmp.mk                     # GMPビルド設定
│   ├── libuv.mk                   # libuvビルド設定
│   └── ...                        # その他約30の外部依存
├── doc/                           # ドキュメント・ユーザーマニュアル
├── test/                          # テストスイート
└── contrib/                       # 補助ツール・スクリプト
```

### コンポーネント依存関係

主要コンポーネント間の依存関係は以下の通り:

1. **CLI/Loader** は `libjulia`（ランタイム）と `libjulia-codegen`（コード生成）を動的にロードする。`libjulia-codegen` はオプショナルであり、削除してもランタイムは起動可能（コード生成は不可）。
2. **Runtime (`libjulia`)** は `support/` ライブラリ、libuv、GC（stock/MMTk選択可能）に依存する。
3. **Codegen (`libjulia-codegen`)** は LLVM 共有ライブラリに依存し、`libjulia` のシンボルを参照する。
4. **Core/Base** は Runtime の `ccall` インターフェースを通じてC関数を呼び出す。
5. **Compiler** は Base/Core の型定義・ユーティリティを使用するが、独立パッケージとして差し替え可能。
6. **JuliaSyntax** / **JuliaLowering** はトップレベルパッケージとして `base/` と同格に配置され、フロントエンドパイプラインを構成する。
7. **stdlib** パッケージはBaseに依存し、一部はJLLパッケージ経由で外部Cライブラリにアクセスする。

## ミドルウェア構成

### データベース

Julia言語処理系自体はデータベースを使用しない。パッケージ管理（Pkg）はTOML形式のマニフェストファイルでパッケージメタデータを管理する。

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| ファイルベース | TOML (Project.toml / Manifest.toml) | - | パッケージ依存関係・バージョン管理 |
| ファイルベース | プリコンパイルキャッシュ (.ji / .so) | - | コンパイル済みモジュールのキャッシュ |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| システムイメージ (sysimage) | - | Base/stdlibのプリコンパイル済みバイナリ | ビルド時に生成、再ビルドまで永続 |
| パッケージイメージ (pkgimage) | - | 各パッケージのプリコンパイル済みバイナリ | パッケージ更新まで永続 |
| ネイティブコードキャッシュ | - | JITコンパイル済みネイティブコード | プロセス寿命 |

### メッセージキュー

Julia言語処理系は外部メッセージキューを使用しない。プロセス内ではChannelによるタスク間通信を提供する。

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| Channel (Julia組み込み) | - | タスク間の同期・非同期メッセージパッシング |

### 検索エンジン

該当なし。

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| LLVM | deps/llvm.version で管理 | JITコンパイラバックエンド、最適化パイプライン |
| libuv | deps/libuv.version で管理 | 非同期I/O、イベントループ、プラットフォーム抽象化 |
| FemtoLisp (flisp) | 組み込み | レガシーフロントエンド（マクロ展開等、段階的にJulia実装に移行中） |

## データフロー

### リクエスト処理の流れ（ユーザーコード実行）

Julia REPLまたはスクリプト実行時の処理フロー:

1. **CLI/Loader**: `loader_exe.c` がプロセスを起動し、`libjulia` と `libjulia-codegen` を動的ロードする。
2. **Runtime初期化**: `src/init.c` の `jl_init()` がランタイムを初期化する（GC、スレッドプール、型システム、システムイメージのロード）。
3. **ソースコード入力**: ユーザーがREPLに入力した式、またはファイルから読み込んだソースコードを受け取る。
4. **パース（構文解析）**: `JuliaSyntax` がソースコードをパースし、構文木（SyntaxTree/GreenTree）を生成する。レガシーパスでは `src/flisp/` の FemtoLisp パーサーが使用される場合もある。
5. **マクロ展開**: マクロが展開され、AST（Abstract Syntax Tree）が変換される。
6. **Lowering**: `JuliaLowering` が高レベルASTを低レベルIR（Intermediate Representation）に変換する。スコープ解析、脱糖（desugaring）、クロージャ変換、線形IR生成が行われる。
7. **型推論（Type Inference）**: `Compiler/src/typeinfer.jl` が抽象解釈を実行し、各式の型を推論する。
8. **最適化**: `Compiler/src/optimize.jl` がSSA IRレベルでインライン化、定数畳み込み、不要コード除去等の最適化を行う。
9. **コード生成**: `src/codegen.cpp` がJulia IRからLLVM IRを生成し、LLVMの最適化パイプラインとネイティブコード生成を経てマシンコードを出力する。
10. **実行**: 生成されたネイティブコードが実行される。結果がREPLに表示されるか、プログラムの出力として処理される。

### 非同期処理の流れ

Juliaのタスク（コルーチン）ベースの非同期処理フロー:

1. **タスク生成**: `Task(f)` でユーザー関数をラップした軽量タスク（コルーチン）が生成される（`base/task.jl`）。
2. **スケジューリング**: `schedule(task)` でタスクがランナブルキューに追加される。ワークスティーリングスケジューラ（`src/scheduler.c`, `src/work-stealing-queue.h`）がタスクをOSスレッドに割り当てる。
3. **I/O待機**: I/O操作はlibuv経由で非同期に実行される。I/O完了時にコールバックが発火し、待機中のタスクが再スケジュールされる（`base/libuv.jl`, `base/asyncevent.jl`）。
4. **Channel通信**: タスク間はChannel（`base/channels.jl`）を通じてデータを送受信する。`put!` / `take!` でブロッキング送受信を行う。
5. **タスク完了**: タスクの実行が完了すると結果が格納され、待機中のタスクに通知される。

### データ永続化の流れ

Julia言語処理系におけるデータ永続化:

1. **システムイメージ生成**: ビルド時に `base/sysimg.jl` が実行され、Core/Base/stdlibのプリコンパイル済みバイナリ（システムイメージ）が `src/staticdata.c` によってシリアライズ・保存される。
2. **パッケージプリコンパイル**: `base/precompilation.jl` と `base/loading.jl` がパッケージの初回ロード時にプリコンパイルを実行し、`.ji` ファイル（シリアライズされたモジュール状態）および `.so`/`.dylib`（ネイティブコードキャッシュ）を生成する。
3. **AOTコンパイル**: `src/aotcompile.cpp` によるAhead-of-Timeコンパイルで、特定のメソッドのネイティブコードを事前に生成・保存する。

## 横断的関心事

### 認証・認可

Julia言語処理系自体には認証・認可機構は存在しない。パッケージレジストリへのアクセスはHTTP(S)を使用し、TLS証明書の検証は `MozillaCACerts_jll` および `OpenSSL` / `LibGit2` 経由で行われる。

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| TLS証明書検証 | `stdlib/MozillaCACerts_jll/`, `stdlib/OpenSSL_jll/` | パッケージレジストリ通信 |
| SSH鍵認証 | `stdlib/LibSSH2_jll/`, `stdlib/LibGit2/` | Gitリポジトリアクセス |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| アプリケーションログ | `base/logging/logging.jl` (CoreLogging) | stderr（デフォルト）、カスタムLoggerで変更可能 |
| コンパイラ診断 | `Compiler/src/timing.jl` | stderr / プロファイリングツール |
| GCデバッグログ | `src/gc-debug.c` | stderr / 環境変数で制御 |
| プロファイリング | `stdlib/Profile/` | インメモリバッファ、後処理で出力 |
| Tracy統合 | `deps/libtracyclient.*` | Tracy Profiler |

### エラーハンドリング

Julia言語処理系は例外ベースのエラーハンドリングを採用している。

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| Julia例外 | `try-catch-finally` 構文 | 例外オブジェクト（Exception型のサブタイプ）とバックトレース |
| 型エラー | `TypeError` / `MethodError` | 型不一致・メソッド未発見のエラーメッセージ |
| 境界エラー | `BoundsError` | 配列インデックス範囲外のエラー |
| C層エラー | `jl_error()` / `jl_throw()` | C関数からJulia例外をスロー |
| システムエラー | `SystemError` (errno wrap) | OS呼び出しの失敗をJulia例外に変換 |
| パース/構文エラー | パーサーからのエラーノード | 構文エラーメッセージとソース位置情報 |
| スタックオーバーフロー | シグナルハンドラ（`src/signals-*.c`） | `StackOverflowError` への変換 |

### トランザクション管理

Julia言語処理系ではデータベーストランザクション管理は該当しない。ただし、ワールドカウンタ（World Age）メカニズムにより、メソッド定義の一貫性が保証される。

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| メソッドテーブル | ワールドカウンタ（World Age） | 各タスクは特定のワールドエイジのメソッドテーブルスナップショットを参照する |
| GC | Stop-the-world GC（stock GC） | GC中は全スレッドが停止 |
| アトミック操作 | `base/atomics.jl`, `src/julia_atomics.h` | ロックフリーデータ構造での原子的操作 |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| 多重ディスパッチ | 言語全体 | 関数の引数型に基づくメソッド選択（`src/gf.c`）。Juliaの核心的設計原則 |
| 型安定性（Type Stability） | コンパイラ最適化 | 型推論により関数の戻り値型が引数型から一意に決定可能な場合、高効率なネイティブコードを生成 |
| コンポーザビリティ | Base/stdlib | 抽象型・インターフェースによるジェネリックプログラミング（`AbstractArray`, `AbstractDict`等） |
| モジュール差し替え可能性 | Compiler | Compilerモジュールは独立パッケージとして設計され、別のCompiler実装に差し替え可能 |
| GC抽象化 | Runtime GC | `gc-interface.h` を通じたGC実装の抽象化。Stock GCとMMTk GCの切り替えが可能 |
| プリコンパイル・キャッシュ | Base/Compiler | システムイメージとパッケージイメージによるコンパイル結果のキャッシュで起動時間を短縮 |
| FFIとの密統合 | Base `ccall` | `ccall` 構文による外部Cライブラリとの直接的なゼロオーバーヘッド連携（`base/c.jl`） |
| JLL（Julia Library Linking） | stdlib/*_jll | 外部バイナリ依存をJuliaパッケージとしてラップし、BinaryBuilder経由で配布 |

### コーディング規約

- **Julia側**: `ccall` でランタイムC関数を直接呼び出すことは避け、既存のJuliaラッパーを使用する。モジュール内のコードでは明示的な `Base.` プレフィックスを使用しない（同一モジュール内の場合）。
- **C/C++側**: 静的解析（Clang SA, Clang Tidy）によるコード品質担保。GC安全性アノテーション（`JL_GC_PUSH`, `JL_GC_PROMISE_ROOTED` 等）の適切な使用が必須。
- **空白文字**: コミット前に `make fix-whitespace` を実行し、空白エラーを排除する。
- **テスト**: 変更に対応するテストは既存テストファイルに追加する（新規ファイル作成は原則不可）。テストの先頭にテスト内容を説明するコメントを1行記載する。
- **コミットメッセージ**: `component: Brief summary` 形式のタイトルとし、本文は変更の目的を簡潔に記述する。

## 備考

- Julia v1.14.0-DEVはアクティブに開発中のバージョンであり、アーキテクチャの一部（特にフロントエンド: JuliaSyntaxおよびJuliaLoweringへの移行）は進行中の変更を含む。
- FemtoLisp（`src/flisp/`）はレガシーフロントエンドであり、JuliaSyntaxおよびJuliaLoweringへの段階的な移行が進められている。
- GCの実装は `gc-interface.h` による抽象化層を通じて、Stock GC（`gc-stock.c`）とMMTk GC（`gc-mmtk.c`）を切り替え可能。
- `USE_BINARYBUILDER` フラグにより、外部依存を事前ビルド済みバイナリ（JLL）で取得するか、ソースからビルドするかを選択可能。
