# セキュリティ設計書

## 概要

本ドキュメントは、Julia言語ランタイムおよびその標準ライブラリにおけるセキュリティ設計を記述する。Juliaはプログラミング言語処理系であり、一般的なWebアプリケーションとは異なるセキュリティモデルを持つ。本書では、メモリ安全性、暗号・乱数生成、認証情報管理、機密データ保護、バイナリ保護機構、および通信セキュリティの観点からセキュリティ設計を整理する。

## 認証設計

### 認証方式

JuliaランタイムはWebサービスのような汎用認証フレームワークを持たないが、以下の認証機構が標準ライブラリに組み込まれている。

#### LibGit2認証メカニズム

`stdlib/LibGit2` はリモートGitリポジトリとの通信時に、`credential_callback` 関数を通じて以下の認証方式を順序立てて試行する。

1. **SSH-Agent認証**：`ssh-agent` デーモンを通じた鍵ベース認証（`git_cred_ssh_key_from_agent`）
2. **SSH鍵ペア認証**：秘密鍵/公開鍵ファイルを用いた認証（`git_cred_ssh_key_new`）。鍵ファイルパスは環境変数 `SSH_KEY_PATH` / `SSH_PUB_KEY_PATH` またはユーザプロンプトから取得する。パスフレーズが必要な場合は `SSH_KEY_PASS` 環境変数またはプロンプトで取得する。
3. **ユーザ名/パスワード認証**：プレーンテキストでのユーザ名・パスワード認証（`git_cred_userpass_plaintext_new`）。Git credential helperとの連携をサポートする。

認証失敗時はプロンプト回数に上限（デフォルト3回）が設けられ、無限ループを防止する。

#### 認証情報キャッシュ

`CachedCredentials` 構造体により、認証に成功したクレデンシャルをURLベースのキー（`credential_identifier`）でキャッシュし、再認証時に再利用する。キャッシュ破棄時には `shred!` による安全な消去が行われる。

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | 該当なし | Julia処理系はセッション管理機構を持たない |
| セッション固定化対策 | 該当なし | 同上 |
| 認証プロンプト上限 | 3回（`remaining_prompts`） | `CredentialPayload` で管理。`allow_prompt=false` で無効化可能 |

## 認可設計

### 権限体系

| ロール | 権限 | 説明 |
| --- | --- | --- |
| 該当なし | 該当なし | Julia処理系はロールベースのアクセス制御を持たない。プロセスの権限はOSレベルで管理される |

### アクセス制御

Juliaランタイムはプロセスレベルのアクセス制御をOS委任としている。ファイルシステム操作（`chmod` 等）はlibuv経由でOSのファイルパーミッション機構を利用する。

- `mkdir` のデフォルトパーミッション: `0o777`（umaskにより制限）
- `chmod` 関数: POSIX準拠のパーミッション設定をサポート（`base/file.jl`）
- コード実行: `include`、`eval`、`Meta.parse` によるコード実行は明示的な制限なし（ユーザ責任）

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| TLS/SSL | OpenSSL 3.x を `OpenSSL_jll` として同梱（`libssl`、`libcrypto`）。TLS通信に使用 |
| SSH通信 | `LibSSH2_jll` による libssh2 を利用。known_hostsファイルによるホスト検証を実施 |
| TLSホスト検証 | LibGit2の `certificate_callback` でTLS証明書の有効性を検証。検証失敗時は接続拒否 |
| SSHホスト検証 | `ssh_knownhost_check` 関数で known_hosts ファイルに基づくホストキー検証を実施。MATCH/NOTFOUND/MISMATCHを判定 |
| 証明書バンドル | `MozillaCACerts_jll` でMozilla CA証明書バンドルを提供 |
| ホスト検証オプトアウト | `NetworkOptions.verify_host` により、ユーザが明示的にホスト検証を無効化可能 |
| ダウンロード | `Downloads.jl` ライブラリを使用。`LibCURL_jll` 経由でcurl/libcurlを利用 |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信（TLS） | OpenSSL 3.x による TLS 1.2/1.3。`libssl`/`libcrypto` を使用 |
| SSH通信 | libssh2 による暗号化。RSA、ECDSA鍵をサポート |
| ハッシュ | SHA-2ファミリー（SHA2-256、SHA2-512）を `SHA` stdlibで提供。乱数シード生成時（`SeedHasher`）に使用 |
| CRC32c | データ整合性チェック用。`jl_crc32c` で実装（`src/crc32c.c`） |
| 保存データ暗号化 | 該当なし（処理系レベルでのデータ保存暗号化機能は提供しない） |

### 機密情報管理

#### SecretBuffer（`base/secretbuffer.jl`）

`SecretBuffer` は、パスワードやトークン等の機密情報を安全に扱うための `IO` オブジェクトである。

- **安全な消去**: `shred!` 関数により `securezero!` を呼び出し、バッファの内容をゼロクリアする
- **定数時間比較**: `==` 演算子は XOR ベースの定数時間比較（`_bufcmp`）を使用し、タイミング攻撃を防止する
- **ハッシュ均一化**: すべての `SecretBuffer` は同一のハッシュ値を返し、情報漏洩を防止する
- **GCファイナライザ**: GC時に未消去の `SecretBuffer` を検出し、警告を発する（`final_shred!`）
- **リサイズ時の安全性**: バッファ拡張時に旧データを `securezero!` で消去してからコピーする
- **表示時のマスク**: `show` メソッドは `SecretBuffer("*******")` と表示し、内容を表示しない

#### securezero!（`base/util.jl`）

- `securezero!(a)`: コンパイラの最適化で省略されないよう `@noinline` で修飾されたゼロクリア関数
- `unsafe_securezero!(p, len)`: ポインタベースのメモリゼロクリア（`memset` 使用）

#### パスワード入力（`base/util.jl`）

- `getpass`: ターミナルエコーを無効化してパスワードを安全に読み取り、`SecretBuffer` として返す
- `winprompt`（Windows）: `CredUIPromptForWindowsCredentialsW` API を使用し、暗号化されたクレデンシャルバンドルを経由して安全にユーザ名/パスワードを取得する。使用後は `unsafe_securezero!` で即座に消去する

#### LibGit2クレデンシャルの安全な消去

- `UserPasswordCredential`: パスワードを `SecretBuffer` で保持。`shred!` で安全に消去
- `SSHCredential`: パスフレーズを `SecretBuffer` で保持。`shred!` で安全に消去
- `GitCredential`: パスワードフィールドを `SecretBuffer` で保持。`shred!` ですべてのフィールドを消去
- `CachedCredentials`: `shred!` でキャッシュ内の全クレデンシャルを消去
- `CredentialPayload`: 認証完了後、`approve`/`reject` メソッドで不要なクレデンシャルを `shred!`

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| バッファオーバーフロー | `@boundscheck` マクロによる配列境界チェック。`checkbounds` 関数でインデックスの妥当性を検証。`@inbounds` で明示的にチェックを無効化可能 |
| スタックオーバーフロー | ガードページ機構（`jl_guard_size = 4096 * 8 = 32768 bytes`）。`mprotect` / `VirtualProtect` でスタック末尾に `PROT_NONE` のガードページを設定（`src/gc-stacks.c`） |
| メモリ安全性 | GC（ガベージコレクション）による自動メモリ管理。`ccall` 等のFFI使用時はユーザ責任 |
| XSS | 該当なし（Julia処理系はWebフロントエンドを持たない） |
| SQLインジェクション | 該当なし（Julia処理系はSQL機能を持たない） |
| CSRF | 該当なし（Julia処理系はWebサーバ機能を持たない） |
| コードインジェクション | `eval`、`Meta.parse`、`include` によるコード実行は言語仕様として許可。サンドボックス機構は未実装 |

## 乱数生成

### 暗号学的乱数生成器（CSPRNG）

| 項目 | 詳細 |
| --- | --- |
| RandomDevice | OS提供のエントロピーソースを利用。`Libc.getrandom!` 経由で `uv_random`（libuv）を呼び出す |
| OS実装 | Linux: `getrandom(2)` syscall、macOS: `/dev/urandom`、Windows: `BCryptGenRandom` 等（libuv内部で選択） |
| 用途 | シード生成、暗号用途の乱数 |

### 疑似乱数生成器（PRNG）

| 項目 | 詳細 |
| --- | --- |
| Xoshiro256++ | デフォルトのPRNG。高速・小メモリフットプリント。タスクローカルRNG（`TaskLocalRNG`）として使用 |
| MersenneTwister | 代替PRNG。`dSFMT_jll` ライブラリ使用 |
| SeedHasher | SHA2-256/SHA2-512ベースのシードハッシュ機構。異なるシードから異なる乱数列を保証 |
| 初期シード | `__init__` 時に `RandomDevice` からOSエントロピーを取得して自動シード |

**注意**: Xoshiro256++ および MersenneTwister は暗号学的に安全な乱数生成器ではない。暗号用途には `RandomDevice` を使用すべきである。

## バイナリ保護機構

| 項目 | 対策 |
| --- | --- |
| RELRO | Linux ビルドで `-Wl,-z,relro` を使用。GOT（Global Offset Table）の書き換え攻撃を防止（`Make.inc`） |
| W^X メモリ保護 | JITコード生成時、`cgmemmgr.cpp` で `mprotect` / `VirtualProtect` を使用し、RW（書込可能）とRX（実行可能）を分離。W^X（Write XOR Execute）ポリシーを実施 |
| ガードページ | タスクスタックに `PROT_NONE` のガードページ（32KB）を設定し、スタックオーバーフローを検出（`src/gc-stacks.c`） |
| サニタイザ対応 | AddressSanitizer、MemorySanitizer、ThreadSanitizer に対応（`src/jitlayers.h`）。開発・テスト時のメモリ安全性検証に使用 |
| シグナル保護 | `jl_sigatomic_begin` / `jl_sigatomic_end` によるアトミックなシグナル保護領域（`src/jlapi.c`） |
| 静的解析 | clang static analyzer および clang-tidy によるビルド時静的解析（`make -C src analyze-<filename>`） |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| SecretBuffer警告ログ | GC時に未消去の `SecretBuffer` を検出した場合に `@warn` で記録 | ランタイムセッション中 |
| SSH検証ログ | TLS/SSHホスト検証の失敗・不一致を `stderr` に出力 | ランタイムセッション中 |
| 認証エラーログ | LibGit2認証試行の失敗を `git_error_set_str` 経由で記録 | libgit2内部管理 |

Julia処理系は汎用の監査ログフレームワークを持たない。上記のログはランタイムのデバッグ・警告目的であり、永続的な監査ログ機能は提供されない。

## 備考

- Julia処理系はWebアプリケーションフレームワークではないため、XSS、CSRF、SQLインジェクション等のWeb固有の脅威は直接的には該当しない
- `eval`、`Meta.parse`、`include` による動的コード実行は言語設計上の仕様であり、サンドボックス機構は未提供。ユーザコードの実行環境のセキュリティはOS/インフラレベルで対処する必要がある
- `ccall` を通じたFFI（Foreign Function Interface）はメモリ安全性をユーザに委ねており、不正なポインタ操作によるメモリ破壊のリスクが存在する
- 外部依存ライブラリ（OpenSSL、libssh2、libcurl等）のセキュリティアップデートは JLL（Julia Library Loader）パッケージのバージョン更新により管理される
