#!/usr/bin/env python3
import argparse
import csv
import io
import sys


UTF8_BOM = b"\xef\xbb\xbf"
VIEWPOINTS = {"正常系", "境界値", "異常系"}


def _split_bom(data: bytes) -> tuple[bytes, bytes]:
    if data.startswith(UTF8_BOM):
        return UTF8_BOM, data[len(UTF8_BOM) :]
    return b"", data


def _detect_newline(data: bytes) -> str:
    if b"\r\n" in data:
        return "\r\n"
    if b"\r" in data:
        return "\r"
    return "\n"


def _fix_field_count(row: list[str], expected_len: int) -> list[str]:
    if len(row) == expected_len:
        return row
    if len(row) == expected_len - 1 and len(row) > 4:
        if row[4] in VIEWPOINTS:
            return row[:4] + [""] + row[4:]
        if len(row) > 7 and row[7].startswith("なし\""):
            before, after = row[7].split("\"", 1)
            return row[:7] + [before, after] + row[8:]
    if len(row) == expected_len + 1 and len(row) > 9:
        merged = row[8] + "," + row[9]
        return row[:8] + [merged] + row[10:]
    if len(row) == expected_len + 2 and len(row) > 10:
        merged = row[8] + "," + row[9] + "," + row[10]
        return row[:8] + [merged] + row[11:]
    return row


def main() -> int:
    parser = argparse.ArgumentParser(
        description="Re-number the No column in a CSV file."
    )
    parser.add_argument("csv_path", help="Target CSV path")
    parser.add_argument(
        "--no-renumber",
        action="store_true",
        help="Do not rewrite the No column",
    )
    args = parser.parse_args()

    with open(args.csv_path, "rb") as f:
        data = f.read()

    bom, body = _split_bom(data)
    newline = _detect_newline(body)

    text = body.decode("utf-8")
    reader = csv.reader(io.StringIO(text, newline=""))
    rows = list(reader)
    if not rows:
        return 0

    header = rows[0]
    expected_len = len(header)
    out_rows = [header]
    for index, row in enumerate(rows[1:], start=1):
        if not row:
            out_rows.append(row)
            continue
        fixed_row = _fix_field_count(list(row), expected_len)
        new_row = list(fixed_row)
        if not args.no_renumber:
            new_row[0] = str(index)
        out_rows.append(new_row)

    out_io = io.StringIO(newline="")
    writer = csv.writer(
        out_io,
        lineterminator=newline,
        quoting=csv.QUOTE_ALL,
    )
    writer.writerows(out_rows)

    out_bytes = bom + out_io.getvalue().encode("utf-8")
    with open(args.csv_path, "wb") as f:
        f.write(out_bytes)

    return 0


if __name__ == "__main__":
    sys.exit(main())
