# 機能設計書 1-プリミティブ型定義

## 概要

本ドキュメントは、Julia言語の型システムにおけるプリミティブ数値型の定義と管理に関する機能設計書である。

### 本機能の処理概要

Julia言語の基盤となるプリミティブ数値型（Bool, Int8/16/32/64/128, UInt8/16/32/64/128, Float16/32/64）を定義し、型階層を構築する機能である。

**業務上の目的・背景**：Julia言語において、すべての数値計算の基礎となるデータ型を提供するために必要な機能である。プリミティブ型はCPUのネイティブ命令に直接マッピングされ、高速な数値演算を実現する。Julia言語のパフォーマンスの根幹を支える型システムの最も低レベルな層であり、メモリ上のビット表現を直接定義する。

**機能の利用シーン**：すべてのJuliaプログラムにおいて暗黙的に使用される。数値リテラル（例: `42`, `3.14`, `true`）の型解決、関数引数の型アノテーション、多重ディスパッチにおける型マッチング等で利用される。

**主要な処理内容**：
1. 抽象型階層の定義（Number > Real > Integer/AbstractFloat > Signed/Unsigned）
2. `primitive type` キーワードによるビット幅を持つ型の宣言（例: `primitive type Int64 <: Signed 64 end`）
3. プラットフォーム依存の型エイリアス定義（`Int` / `UInt` の 32/64ビット選択）
4. 浮動小数点特殊定数（Inf, NaN, eps）の定義

**関連システム・外部連携**：Cランタイム（`src/jltypes.c`）における型オブジェクトの初期化、LLVM IRでの型マッピング（`src/codegen.cpp`）と連携する。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | プリミティブ型は画面に直接関連しないが、REPL上での型名表示に使用される |

## 機能種別

型定義・型システム基盤

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| type_name | Symbol | Yes | 型名（例: Int64） | 予約語と重複しないこと |
| bit_width | Int32 | Yes | ビット幅（8,16,32,64,128） | 8の倍数であること |
| supertype | DataType | Yes | 親抽象型 | 有効な抽象型であること |

### 入力データソース

Juliaブートストラップ処理（`base/boot.jl`）およびCランタイム初期化（`src/jltypes.c`）から定義される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| DataType | Core.DataType | 各プリミティブ型のDataTypeオブジェクト |
| 型階層 | 型グラフ | Number > Real > Integer/AbstractFloat の階層構造 |
| 型エイリアス | const定義 | Int=Int64/Int32, UInt=UInt64/UInt32 等のプラットフォーム依存エイリアス |

### 出力先

Core モジュールのグローバルスコープに型オブジェクトとしてエクスポートされる。

## 処理フロー

### 処理シーケンス

```
1. Cランタイム初期化（jl_init_types in src/jltypes.c）
   └─ Bool, Int8〜Int128, UInt8〜UInt128, Float16〜Float64のDataTypeオブジェクトを生成
2. boot.jl の読み込み
   └─ 抽象型階層を定義（Number, Real, AbstractFloat, Integer, Signed, Unsigned）
3. primitive type 宣言の処理
   └─ 各型のビット幅と親型を設定
4. プラットフォーム依存エイリアスの設定
   └─ Sys.WORD_SIZE に基づき Int/UInt を Int32/UInt32 または Int64/UInt64 に設定
5. 型のエクスポート
   └─ Core モジュールから export 文で公開
```

### フローチャート

```mermaid
flowchart TD
    A[Cランタイム初期化] --> B[基本DataType生成]
    B --> C[boot.jl 読み込み]
    C --> D[抽象型階層定義]
    D --> E{プラットフォーム判定}
    E -->|64bit| F[Int=Int64, UInt=UInt64]
    E -->|32bit| G[Int=Int32, UInt=UInt32]
    F --> H[型エクスポート]
    G --> H
    H --> I[完了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 型階層不変条件 | すべてのプリミティブ整数型はInteger <: Real <: Numberの部分型でなければならない | 常時 |
| BR-02 | ビット幅制約 | プリミティブ型のビット幅は8の倍数でなければならない | 型定義時 |
| BR-03 | プラットフォーム適応 | Int/UIntはSys.WORD_SIZEに応じて自動的に選択される | 起動時 |
| BR-04 | IEEE 754準拠 | Float16/32/64はIEEE 754規格に準拠したビット表現を持つ | 常時 |

### 計算ロジック

プリミティブ型自体に計算ロジックはないが、各型のビット表現はCPUアーキテクチャに直接対応する。

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（データベースを使用しない）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TypeError | TypeError | 型階層に矛盾がある場合 | 正しい親型を指定する |
| ErrorException | ErrorException | primitive typeの再定義を試みた場合 | 型の再定義は不可 |

### リトライ仕様

型定義は起動時の一回限りの処理であり、リトライは適用されない。

## トランザクション仕様

該当なし。型定義はアトミックな操作として実行される。

## パフォーマンス要件

プリミティブ型はCPUネイティブ命令に直接マッピングされるため、型アクセスはO(1)で実行される。起動時の型初期化は数ミリ秒以内に完了する。

## セキュリティ考慮事項

プリミティブ型の定義自体にセキュリティ上の懸念はないが、整数オーバーフロー（typemin(Int64)での符号反転等）は利用者が注意すべき点である。

## 備考

- Julia 1.11以降ではBFloat16型が追加された（`base/boot.jl` 259行目）
- 一部の型（Bool, UInt8, UInt16, Int32, UInt32, Int64, UInt64）はCランタイムで直接定義され、boot.jlではコメントアウトされた形で記載されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、型システムの階層構造とDataTypeの内部表現を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | boot.jl | `base/boot.jl` | DataType構造体のフィールド定義（18-30行目のコメント）を理解する |
| 1-2 | boot.jl | `base/boot.jl` | 抽象型階層Number > Real > AbstractFloat/Integer > Signed/Unsigned（248-253行目）を把握する |

**読解のコツ**: `base/boot.jl` の冒頭部分はコメントアウトされた型定義が並ぶ。これらはCランタイム（`src/jltypes.c`）で実装されており、Juliaコード上はドキュメント目的で記載されている。`#` で始まる行はCでの実装を示すリファレンスである。

#### Step 2: エントリーポイントを理解する

型システムの初期化はCランタイムから始まる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | jltypes.c | `src/jltypes.c` | `jl_init_types()` 関数でプリミティブ型のDataTypeオブジェクトが生成される |
| 2-2 | boot.jl | `base/boot.jl` | **248-274行目**: abstract type / primitive type によるJulia側の型宣言 |

**主要処理フロー**:
1. **248行目**: `abstract type Number end` - 数値型の最上位抽象型
2. **249行目**: `abstract type Real <: Number end` - 実数型
3. **250行目**: `abstract type AbstractFloat <: Real end` - 浮動小数点型
4. **251行目**: `abstract type Integer <: Real end` - 整数型
5. **252行目**: `abstract type Signed <: Integer end` - 符号付き整数型
6. **253行目**: `abstract type Unsigned <: Integer end` - 符号なし整数型
7. **255-257行目**: `primitive type Float16/32/64 <: AbstractFloat 16/32/64 end` - 浮動小数点型
8. **265-274行目**: `primitive type Int8/16/128 <: Signed` 等 - 整数プリミティブ型
9. **276-280行目**: プラットフォーム依存の `Int` / `UInt` エイリアス設定

#### Step 3: 型エクスポートを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | boot.jl | `base/boot.jl` | **207-243行目**: Coreモジュールからのexport文。数値型が公開される |

**主要処理フロー**:
- **216-219行目**: 数値型のエクスポート（Number, Real, Integer, Bool, Float16/32/64, Int8〜128, UInt8〜128等）

#### Step 4: 浮動小数点定数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | float.jl | `base/float.jl` | **1-97行目**: IEEEFloat型エイリアス、Inf/NaN各精度の定数定義 |
| 4-2 | float.jl | `base/float.jl` | **107-138行目**: ビットマスク関数（sign_mask, exponent_mask, significand_mask）の定義 |

### プログラム呼び出し階層図

```
Cランタイム (src/jltypes.c)
    │
    ├─ jl_init_types()
    │      ├─ jl_new_primitivetype() ... Bool, Int8〜UInt128, Float16〜64
    │      └─ jl_new_abstracttype() ... Number, Real, Integer 等
    │
    └─ boot.jl (base/boot.jl)
           ├─ abstract type 宣言 (248-253行目)
           ├─ primitive type 宣言 (255-274行目)
           ├─ Int/UInt エイリアス (276-280行目)
           └─ export 文 (207-243行目)
               │
               └─ float.jl (base/float.jl)
                      ├─ IEEEFloat 型エイリアス (3行目)
                      ├─ Inf/NaN 定数 (15-97行目)
                      └─ ビットマスク関数 (107-138行目)
```

### データフロー図

```
[入力]                    [処理]                        [出力]

Cランタイム初期化    ───▶ jl_init_types()          ───▶ DataType オブジェクト群
                                                        (jl_int64_type, jl_float64_type 等)

boot.jl 読み込み     ───▶ abstract/primitive type  ───▶ 型階層グラフ
                          宣言の評価                     (Number > Real > Integer > Signed...)

プラットフォーム情報  ───▶ Sys.WORD_SIZE 判定       ───▶ Int/UInt エイリアス
(32bit or 64bit)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| boot.jl | `base/boot.jl` | ソース | プリミティブ型と抽象型階層のJulia側定義 |
| float.jl | `base/float.jl` | ソース | 浮動小数点型の定数・ビット操作関数定義 |
| jltypes.c | `src/jltypes.c` | ソース（C） | 型システムのCランタイム実装 |
| codegen.cpp | `src/codegen.cpp` | ソース（C++） | LLVM IRへの型マッピング |
| essentials.jl | `base/essentials.jl` | ソース | 型関連ユーティリティ（Bottom, Callable等） |
