# 機能設計書 106-Markdown

## 概要

本ドキュメントは、Julia標準ライブラリ `Markdown` モジュールが提供するMarkdownテキストの解析・レンダリング機能の設計を記述する。Juliaのドキュメンテーションシステムの基盤であり、Markdownテキストを構造化オブジェクト（`MD`型）に変換し、プレーンテキスト、HTML、LaTeX、RST、ターミナル出力の複数フォーマットでレンダリングする。

### 本機能の処理概要

**業務上の目的・背景**：Juliaのdocstring（関数・型のドキュメント）はMarkdown形式で記述され、ヘルプモード（`?`）やドキュメント生成ツールで表示される。標準的なMarkdown記法に加え、GitHub拡張（テーブル、脚注）やIPython拡張（LaTeX数式）、Julia固有拡張（doctest、admonition）をサポートする。

**機能の利用シーン**：docstringの記述と表示、REPLヘルプモードでのドキュメント表示、Documenter.jlによるドキュメント生成、Markdown文字列リテラル（`md"..."`）の使用。

**主要な処理内容**：
1. Markdown解析（`parse` 関数）: テキストから`MD`オブジェクトへの変換
2. 文字列マクロ（`@md_str`, `@doc_str`）
3. 複数フレーバーのサポート（Common, GitHub, IPython, Julia）
4. 複数フォーマットへのレンダリング（plain, html, latex, rst, terminal）
5. StyledStringsとJuliaSyntaxHighlightingの統合
6. ターミナル用のフェイス定義（見出し、コードブロック、リンク等）

**関連システム・外部連携**：Base64（stringmime）、StyledStrings（スタイル付き文字列）、JuliaSyntaxHighlighting（シンタックスハイライト）、Base.Docs（docstringシステム）。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | ヘルプモード（help?>） | 主機能 | Markdownでフォーマットされたdocstringの表示 |

## 機能種別

テキスト変換 / レンダリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| markdown | AbstractString / IO | Yes | Markdownテキスト | - |
| flavor | Symbol | No | Markdownフレーバー（デフォルト`:julia`） | :julia, :common, :github 等 |

### 入力データソース

docstring、Markdownファイル、文字列リテラル。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| md_object | Markdown.MD | 構造化されたMarkdownドキュメント |
| rendered_text | String | レンダリングされたテキスト（plain/html/latex/rst/terminal） |

### 出力先

- `MD`オブジェクト（parse結果）
- IOストリーム（show/display経由）

## 処理フロー

### 処理シーケンス

```
1. parse(markdown_text, flavor=:julia)
   ├─ IOBuffer(markdown_text) でバッファ化
   ├─ フレーバー設定に基づくブロック要素パース
   │   ├─ 見出し（#）、段落、リスト、コードブロック、引用
   │   ├─ テーブル（GitHub拡張）
   │   ├─ LaTeX数式（IPython拡張）
   │   └─ Admonition（Julia拡張）
   └─ MD オブジェクト生成
2. show(io, MIME"text/plain"(), md) でレンダリング
   └─ ターミナルレンダラーでスタイル付き出力
```

### フローチャート

```mermaid
flowchart TD
    A["Markdown.parse(text)"] --> B["IOBuffer(text)"]
    B --> C["ブロック要素パース"]
    C --> D["インライン要素パース"]
    D --> E["MD オブジェクト生成"]
    E --> F{"出力先"}
    F -->|ターミナル| G["render/terminal/render.jl"]
    F -->|HTML| H["render/html.jl"]
    F -->|LaTeX| I["render/latex.jl"]
    F -->|RST| J["render/rst.jl"]
    F -->|プレーンテキスト| K["render/plain.jl"]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-106-1 | Juliaフレーバーデフォルト | parse時のデフォルトフレーバーは`:julia`（Common+GitHub+IPython+Julia拡張） | parse呼び出し時 |
| BR-106-2 | マージンサイズ | ターミナル出力のマージンは2文字 | ターミナルレンダリング時 |
| BR-106-3 | フェイス初期化 | `__init__`でMARKDOWN_FACESを登録 | モジュール初期化時 |
| BR-106-4 | catdoc統合 | `catdoc(md::MD...)` で複数MDを結合 | docstring結合時 |

### 計算ロジック

特になし。

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | Markdown構文エラーはパースエラーとならず、テキストとして扱われる | - |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- precompileワークロードで代表的なMarkdown要素の事前コンパイルを実施（139-170行目）
- ターミナルレンダリングにStyledStringsを使用し、効率的な色付き出力

## セキュリティ考慮事項

- HTML出力時のXSSリスクに注意（ユーザー入力をそのままHTMLに変換する場合）
- LaTeX出力時のインジェクションリスクはMarkdownパーサーレベルでは対処しない

## 備考

- `md"..."` 文字列マクロで直接MDオブジェクトを構築可能
- `doc"..."` 文字列マクロはメタ情報（ファイルパス、モジュール）を付加
- MARKDOWN_FACESでターミナル表示のスタイルをカスタマイズ可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | parse/parse.jl | `stdlib/Markdown/src/parse/parse.jl` | `MD` 構造体: Markdownドキュメントのルート型。`content` (要素配列) と `meta` (メタ情報辞書) を保持 |
| 1-2 | Common/Common.jl | `stdlib/Markdown/src/Common/Common.jl` | Common Markdown要素: Header, Paragraph, List, BlockQuote, Code, HorizontalRule, Bold, Italic, Link, Image |
| 1-3 | GitHub/GitHub.jl | `stdlib/Markdown/src/GitHub/GitHub.jl` | GitHub拡張要素: Table, Footnote, Strikethrough |

**読解のコツ**: Markdownの各要素（見出し、段落、リスト等）はそれぞれ独自の構造体として定義され、`MD.content` に配列として格納される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Markdown.jl | `stdlib/Markdown/src/Markdown.jl` | `parse(markdown::String; flavor=julia)`（61行目）: メインパース関数 |
| 2-2 | Markdown.jl | `stdlib/Markdown/src/Markdown.jl` | `@md_str` マクロ（101-103行目）: 文字列リテラルからMDオブジェクト生成 |
| 2-3 | Markdown.jl | `stdlib/Markdown/src/Markdown.jl` | `@doc_str` マクロ（131-133行目）: メタ情報付きMDオブジェクト生成 |

#### Step 3: レンダリングパイプラインを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | render/plain.jl | `stdlib/Markdown/src/render/plain.jl` | プレーンテキストレンダリング |
| 3-2 | render/html.jl | `stdlib/Markdown/src/render/html.jl` | HTMLレンダリング |
| 3-3 | render/terminal/render.jl | `stdlib/Markdown/src/render/terminal/render.jl` | ターミナルレンダリング（StyledStrings使用） |

### プログラム呼び出し階層図

```
Markdown.parse(text) / @md_str / @doc_str
    │
    ├─ IOBuffer(text)
    │
    ├─ ブロック要素パース (parse/parse.jl)
    │      ├─ Header (#)
    │      ├─ Code (```)
    │      ├─ BlockQuote (>)
    │      ├─ List (-, *)
    │      ├─ Table (GitHub)
    │      ├─ LaTeX ($$) (IPython)
    │      ├─ Admonition (!!! ) (Julia)
    │      └─ Paragraph
    │
    ├─ インライン要素パース
    │      ├─ Bold (**)
    │      ├─ Italic (*)
    │      ├─ InlineCode (`)
    │      ├─ Link ([](url))
    │      ├─ Image (![](url))
    │      └─ Footnote ([^ref])
    │
    └─ MD(content, meta) 生成

show(io, MIME"text/plain"(), md)
    └─ レンダラー選択
           ├─ plain.jl → テキスト
           ├─ html.jl → HTML
           ├─ latex.jl → LaTeX
           ├─ rst.jl → reStructuredText
           └─ terminal/render.jl → ターミナル（StyledStrings）
```

### データフロー図

```
[入力]                 [処理]                         [出力]

Markdown文字列 ──▶ parse() ──▶ MD オブジェクト
                                    │
                                    ▼
                              show(io, mime, md)
                                    │
                    ┌───────────┬───────────┬────────────┐
                    ▼           ▼           ▼            ▼
              text/plain    text/html   text/latex   terminal
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Markdown.jl | `stdlib/Markdown/src/Markdown.jl` | ソース | メインモジュール |
| parse/parse.jl | `stdlib/Markdown/src/parse/parse.jl` | ソース | MDパーサーコア |
| parse/config.jl | `stdlib/Markdown/src/parse/config.jl` | ソース | パーサー設定・フレーバー |
| parse/util.jl | `stdlib/Markdown/src/parse/util.jl` | ソース | パースユーティリティ |
| Common/Common.jl | `stdlib/Markdown/src/Common/Common.jl` | ソース | Common Markdown要素定義 |
| GitHub/GitHub.jl | `stdlib/Markdown/src/GitHub/GitHub.jl` | ソース | GitHub拡張（テーブル等） |
| IPython/IPython.jl | `stdlib/Markdown/src/IPython/IPython.jl` | ソース | IPython拡張（LaTeX数式） |
| Julia/Julia.jl | `stdlib/Markdown/src/Julia/Julia.jl` | ソース | Julia固有拡張（admonition等） |
| render/plain.jl | `stdlib/Markdown/src/render/plain.jl` | ソース | プレーンテキストレンダラー |
| render/html.jl | `stdlib/Markdown/src/render/html.jl` | ソース | HTMLレンダラー |
| render/latex.jl | `stdlib/Markdown/src/render/latex.jl` | ソース | LaTeXレンダラー |
| render/rst.jl | `stdlib/Markdown/src/render/rst.jl` | ソース | reStructuredTextレンダラー |
| render/terminal/render.jl | `stdlib/Markdown/src/render/terminal/render.jl` | ソース | ターミナルレンダラー |
