# 機能設計書 109-SHA

## 概要

本ドキュメントは、Julia標準ライブラリ `SHA` パッケージが提供するSHAハッシュ関数機能の設計を記述する。SHA-1およびSHA-2ファミリー（SHA-224, SHA-256, SHA-384, SHA-512, SHA-512/224, SHA-512/256）、SHA-3ファミリー（SHA3-224, SHA3-256, SHA3-384, SHA3-512）の暗号学的ハッシュ関数を提供する。

### 本機能の処理概要

**業務上の目的・背景**：パッケージの整合性検証、ファイルのチェックサム計算、暗号学的ハッシュに基づくデータ識別子生成等に使用される。Juliaのパッケージシステム（Pkg）においてアーティファクトやパッケージの整合性検証に内部的に使用される。

**機能の利用シーン**：パッケージダウンロードの整合性検証、ファイルのフィンガープリント生成、データのハッシュ値計算、HMAC計算。

**主要な処理内容**：
1. SHA-1ハッシュ計算（`sha1`）
2. SHA-2ファミリーハッシュ計算（`sha224`, `sha256`, `sha384`, `sha512`, `sha2_224`, `sha2_256`, `sha2_384`, `sha2_512`）
3. SHA-3ファミリーハッシュ計算（`sha3_224`, `sha3_256`, `sha3_384`, `sha3_512`）
4. HMAC計算（`hmac_sha1`, `hmac_sha224`, `hmac_sha256`, `hmac_sha384`, `hmac_sha512`, `hmac_sha3_224`, `hmac_sha3_256`, `hmac_sha3_384`, `hmac_sha3_512`）
5. ストリーミングAPI（`SHA1_CTX`, `SHA2_256_CTX` 等のコンテキストオブジェクト）
6. IO入力対応（ファイル/ストリームからの直接ハッシュ計算）

**関連システム・外部連携**：外部パッケージ `SHA.jl`（JuliaCrypto/SHA.jl）。stdlib/SHA.versionで管理される外部依存パッケージとして提供。

**権限による制御**：特になし。

## 関連画面

本機能に直接関連する画面はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | パッケージ管理や整合性検証のバックエンド処理として利用 |

## 機能種別

暗号処理 / データ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| data | AbstractBytes / AbstractString / IO | Yes | ハッシュ対象のデータ | - |
| key | Vector{UInt8} | Yes（HMAC時） | HMACの秘密鍵 | - |

### 入力データソース

バイト配列、文字列、IOストリーム（ファイル等）。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| hash_value | Vector{UInt8} | ハッシュ値（バイト配列） |

### 出力先

戻り値。

## 処理フロー

### 処理シーケンス

```
1. sha256(data) 等の関数呼び出し
   ├─ CTXオブジェクト生成（SHA2_256_CTX等）
   ├─ update!(ctx, data): データをブロック単位で処理
   │   └─ 64/128バイトブロックごとに圧縮関数を適用
   └─ digest!(ctx): 最終パディングとハッシュ値出力
2. IO入力の場合
   ├─ CTXオブジェクト生成
   ├─ while !eof(io): read + update!
   └─ digest!(ctx)
```

### フローチャート

```mermaid
flowchart TD
    A["sha256(data)"] --> B["SHA2_256_CTX() 生成"]
    B --> C["update!(ctx, data)"]
    C --> D{"データ残り?"}
    D -->|あり| C
    D -->|なし| E["digest!(ctx)"]
    E --> F["Vector{UInt8} 返却"]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-109-1 | SHA-1非推奨 | SHA-1は衝突攻撃が実用化されており、セキュリティ目的では非推奨 | SHA-1使用時 |
| BR-109-2 | HMAC仕様 | RFC 2104に準拠したHMAC計算 | hmac_*関数使用時 |
| BR-109-3 | 出力サイズ | SHA-1=20, SHA-224=28, SHA-256=32, SHA-384=48, SHA-512=64バイト | 各アルゴリズム |
| BR-109-4 | Pure Julia実装 | CライブラリやOpenSSLに依存しないPure Julia実装 | 常時 |

### 計算ロジック

- SHA-2: メッセージスケジュール拡張 + 64/80ラウンドの圧縮関数
- SHA-3: Keccak-f[1600]置換関数（24ラウンド）
- HMAC: H(K XOR opad, H(K XOR ipad, message))

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | MethodError | 非対応型の入力 | AbstractBytes/AbstractString/IOに変換 |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- Pure Julia実装のためJITコンパイルの最適化が効く
- ストリーミングAPIによりメモリ使用量を制限可能（大きなファイルのハッシュ計算）

## セキュリティ考慮事項

- SHA-1は衝突攻撃が実用化されているため、セキュリティ用途にはSHA-256以上を推奨
- HMAC計算はタイミング攻撃に対する耐性を持つ（定数時間比較は呼び出し側の責任）
- 本ライブラリは暗号学的な監査を受けたものではない

## 備考

- SHA.jlはJuliaCryptoオーガニゼーション配下の外部パッケージ
- stdlib/SHA.versionで管理され、バージョン`876bc0400f9a457eb2736388fc3d0fbe9460fc7d`（master）を使用
- リポジトリURL: https://github.com/JuliaCrypto/SHA.jl.git
- Pure Julia実装であるため、外部C/C++ライブラリへの依存がない

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

**注意**: SHA.jlは外部パッケージであり、ソースコードはstdlibディレクトリには含まれていない。`stdlib/SHA.version`ファイルに記載されたリポジトリURLとコミットハッシュを参照して外部リポジトリから取得する必要がある。

#### Step 1: バージョン・依存情報を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | SHA.version | `stdlib/SHA.version` | SHA_SHA1, SHA_GIT_URL: 外部リポジトリの参照情報 |

#### Step 2: 外部リポジトリのソースコードを理解する

外部リポジトリ（https://github.com/JuliaCrypto/SHA.jl）のソースコードを参照:

| 順序 | ファイル | リポジトリ | 読解ポイント |
|-----|---------|-----------|-------------|
| 2-1 | src/SHA.jl | JuliaCrypto/SHA.jl | メインモジュール: export一覧、include構造 |
| 2-2 | src/types.jl | JuliaCrypto/SHA.jl | CTX型定義: SHA1_CTX, SHA2_256_CTX等の構造体 |
| 2-3 | src/sha2.jl | JuliaCrypto/SHA.jl | SHA-2アルゴリズム実装 |
| 2-4 | src/sha3.jl | JuliaCrypto/SHA.jl | SHA-3（Keccak）アルゴリズム実装 |
| 2-5 | src/hmac.jl | JuliaCrypto/SHA.jl | HMAC計算実装 |

### プログラム呼び出し階層図

```
sha256(data) / sha1(data) / sha3_256(data)
    │
    ├─ CTX() 生成（SHA2_256_CTX等）
    │
    ├─ update!(ctx, data)
    │      └─ transform!() -- ブロック単位の圧縮処理
    │             ├─ SHA-2: メッセージスケジュール + 圧縮ラウンド
    │             └─ SHA-3: Keccak-f[1600]
    │
    └─ digest!(ctx)
           ├─ パディング追加
           └─ 最終ブロック処理 -> ハッシュ値出力

hmac_sha256(key, data)
    ├─ ipad = key XOR 0x36
    ├─ opad = key XOR 0x5c
    ├─ inner = sha256(ipad || data)
    └─ outer = sha256(opad || inner)
```

### データフロー図

```
[入力]                [処理]                      [出力]

バイトデータ ────▶ CTX() + update!() x N
                     │
                     ▼
               digest!() ──────────────▶ ハッシュ値 (Vector{UInt8})

IO ストリーム ──▶ while !eof: read + update!()
                     │
                     ▼
               digest!() ──────────────▶ ハッシュ値 (Vector{UInt8})
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| SHA.version | `stdlib/SHA.version` | バージョン管理 | 外部パッケージのバージョンとリポジトリURL |
| SHA.jl | JuliaCrypto/SHA.jl (外部) | ソース | SHAパッケージ本体 |
