# 機能設計書 110-CRC32c

## 概要

本ドキュメントは、Julia標準ライブラリ `CRC32c` モジュールが提供するCRC-32cチェックサム計算機能の設計を記述する。CRC-32c（Castagnoli）アルゴリズムに基づくチェックサム計算を、バイト配列、文字列、IOストリームに対して提供する。

### 本機能の処理概要

**業務上の目的・背景**：データの整合性検証（ファイル転送、ストレージ）に使用される高速なチェックサムアルゴリズムを提供する。CRC-32cはiSCSI、SCTP、ext4ファイルシステム等で広く使用されており、x86のSSE 4.2命令セットによるハードウェアアクセラレーションが可能な効率的なアルゴリズムである。Juliaのパッケージシステムでもアーティファクトの整合性検証に使用される。

**機能の利用シーン**：ファイルの整合性チェック、データ転送の検証、パッケージアーティファクトの検証、IOBufferのチェックサム計算。

**主要な処理内容**：
1. `crc32c(data)` -- バイト配列/文字列のCRC-32c計算
2. `crc32c(io, nb)` -- IOストリームからnbバイトのCRC-32c計算
3. `crc32c(io)` -- IOストリーム全体のCRC-32c計算
4. チェーン計算（`crc32c(data2, crc32c(data1))`）
5. Base._crc32cへの委譲（ランタイムC実装 `jl_crc32c` 使用）

**関連システム・外部連携**：Base._crc32c（コアランタイム）、jl_crc32c（C実装、SSE 4.2/NEON対応）。

**権限による制御**：特になし。

## 関連画面

本機能に直接関連する画面はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | データ整合性検証のバックエンド処理として利用 |

## 機能種別

計算処理 / データ検証

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| data | Array{UInt8} / AbstractVector{UInt8} / String / SubString{String} | Yes（配列/文字列版） | チェックサム対象のデータ | - |
| io | IO | Yes（IO版） | チェックサム対象のストリーム | - |
| nb | Integer | No | 読み取るバイト数（IO版、省略時は全体） | 正の整数 |
| crc | UInt32 | No | 初期CRC値（デフォルト0x00000000） | - |

### 入力データソース

バイト配列、文字列、IOストリーム。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| checksum | UInt32 | CRC-32cチェックサム値 |

### 出力先

戻り値。

## 処理フロー

### 処理シーケンス

```
1. crc32c(data, crc) -- Dense配列の場合
   └─ Base._crc32c(data, crc)
       └─ Base.unsafe_crc32c(data, length, crc)
           └─ ccall(:jl_crc32c, UInt32, ...) -- C実装呼び出し
2. crc32c(data::AbstractVector{UInt8}, crc) -- 非Dense配列の場合
   ├─ 24576バイトブロック単位でMemoryバッファにコピー
   ├─ Base.unsafe_crc32c(buf, n, crc) で各ブロック計算
   └─ チェーン計算で最終CRC値を算出
3. crc32c(io, crc) -- IOストリームの場合
   └─ Base._crc32c(io, crc) -- 内部でreadしながら計算
```

### フローチャート

```mermaid
flowchart TD
    A["crc32c(data, crc)"] --> B{"data の型?"}
    B -->|DenseUInt8OrInt8| C["Base._crc32c(data, crc)"]
    B -->|AbstractVector| D["24576バイトブロック分割"]
    B -->|String| E["Base._crc32c(s, crc)"]
    B -->|IO| F["Base._crc32c(io, crc)"]
    C --> G["unsafe_crc32c -> jl_crc32c"]
    D --> H["copyto! + unsafe_crc32c ループ"]
    E --> G
    F --> I["read + unsafe_crc32c ループ"]
    G --> J["UInt32 返却"]
    H --> J
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-110-1 | リトルエンディアン | リトルエンディアンのCRC-32cを計算 | 常時 |
| BR-110-2 | チェーン計算 | crc32c(data2, crc32c(data1)) == crc32c([data1; data2]) | CRC結合時 |
| BR-110-3 | ブロックサイズ | AbstractVector用のブロックサイズは24576バイト（8192*3）。3-way並列SIMD閾値 | AbstractVector入力時 |
| BR-110-4 | open互換 | open(crc32c, filename) でファイル全体のチェックサム計算が可能 | ファイルチェックサム時 |

### 計算ロジック

- CRC-32c多項式: 0x1EDC6F41 (iSCSI polynomial / Castagnoli)
- x86 SSE 4.2命令 (`CRC32` 命令) によるハードウェアアクセラレーション
- ARM NEON / CRC拡張によるハードウェアアクセラレーション
- 24576バイト以上の入力で3-way並列SIMD処理

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | EOFError | IOストリームが予期せず終端 | 十分なデータを確保 |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- ハードウェアCRC32命令（SSE 4.2/ARMv8-A CRC拡張）を使用する場合、数GB/sのスループットが可能
- ブロックサイズ24576バイト（8192*3）は3-way並列SIMD処理の閾値として最適化
- AbstractVector用のMemoryバッファ再利用でメモリ割り当てを最小化

## セキュリティ考慮事項

- CRC-32cは暗号学的ハッシュ関数ではなく、意図的な改ざんの検出には適さない
- データの偶発的な破損検出にのみ使用すべき

## 備考

- `open(crc32c, filename)` パターンでファイル全体のチェックサムを簡潔に計算可能
- `seekstart(buf)` と組み合わせてIOBufferのチェックサムを `take!` なしで計算可能
- bitstype配列のチェックサムは `crc32c(vec(reinterpret(UInt8, a)))` で計算可能（エンディアン依存）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CRC32c.jl | `stdlib/CRC32c/src/CRC32c.jl` | `crc32c(a::DenseUInt8OrInt8, crc)`（54-56行目）: Dense配列の直接パス。Base._crc32cに委譲 |
| 1-2 | CRC32c.jl | `stdlib/CRC32c/src/CRC32c.jl` | `crc32c(a::AbstractVector{UInt8}, crc)`（39-52行目）: 非Dense配列のブロック処理。24576バイトブロックでMemoryバッファにコピー |
| 1-3 | CRC32c.jl | `stdlib/CRC32c/src/CRC32c.jl` | `crc32c(s::String, crc)`（58行目）: 文字列のBase._crc32cへの委譲 |
| 1-4 | CRC32c.jl | `stdlib/CRC32c/src/CRC32c.jl` | `crc32c(io::IO, nb, crc)`（67行目）: IOストリームの処理 |
| 1-5 | CRC32c.jl | `stdlib/CRC32c/src/CRC32c.jl` | `crc32c(io::IO, crc)`（68行目）: IOストリーム全体の処理 |

**読解のコツ**: CRC32cモジュールは非常にコンパクト（70行）であり、ほとんどの処理をBase._crc32cに委譲している。AbstractVector{UInt8}用のメソッドのみが独自のブロック分割ロジックを持つ。

#### Step 2: コアランタイム実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | base/util.jl | `base/util.jl` | `Base._crc32c`/`Base.unsafe_crc32c`: ccallでjl_crc32cを呼び出し |
| 2-2 | src/support/crc32c.c | `src/support/crc32c.c` | `jl_crc32c`: SSE 4.2/ARM CRC拡張を使用したC実装 |

**主要処理フロー**:
1. **39-42行目**: AbstractVector用にMemory{UInt8}(undef, min(nb, 24576))のバッファを確保
2. **45-50行目**: 24576バイト以下のブロック単位でcopyto!+unsafe_crc32cを繰り返し
3. **54-56行目**: DenseUInt8OrInt8はBase._crc32cに直接委譲（最も高速なパス）

### プログラム呼び出し階層図

```
crc32c(data, crc)
    │
    ├─ DenseUInt8OrInt8
    │      └─ Base._crc32c(data, crc)
    │             └─ Base.unsafe_crc32c(data, n, crc)
    │                    └─ ccall(:jl_crc32c, UInt32, ...)
    │                           └─ SSE 4.2 CRC32 / ARM CRC拡張 / ソフトウェア実装
    │
    ├─ AbstractVector{UInt8}
    │      ├─ Memory{UInt8}(undef, 24576) -- バッファ確保
    │      ├─ copyto!(buf, 1, a, offset, n) -- ブロックコピー
    │      └─ Base.unsafe_crc32c(buf, n, crc) -- ブロックCRC計算
    │
    ├─ String / SubString{String}
    │      └─ Base._crc32c(s, crc) -> jl_crc32c
    │
    └─ IO
           └─ Base._crc32c(io, nb, crc)
                  └─ read + unsafe_crc32c ループ
```

### データフロー図

```
[入力]                  [処理]                      [出力]

バイト配列(Dense) ──▶ Base._crc32c() ──▶ jl_crc32c ──▶ UInt32
                                              │
                                    SSE 4.2 / ARM CRC / Software

バイト配列(非Dense) ──▶ copyto! → ブロックバッファ ──▶ jl_crc32c ──▶ UInt32
                           └─ 24576バイトブロック

文字列 ──────────▶ Base._crc32c() ──▶ jl_crc32c ──▶ UInt32

IOストリーム ──────▶ read + _crc32c ループ ──▶ UInt32
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CRC32c.jl | `stdlib/CRC32c/src/CRC32c.jl` | ソース | CRC32cモジュール全体（70行） |
| util.jl | `base/util.jl` | ソース | Base._crc32c / Base.unsafe_crc32c（ccallラッパー） |
| crc32c.c | `src/support/crc32c.c` | ソース | jl_crc32c C実装（SSE 4.2/ARM CRC対応） |
