# 機能設計書 111-UUIDs

## 概要

本ドキュメントは、Julia標準ライブラリ `UUIDs` モジュールの機能設計を記述する。UUID (Universally Unique Identifier) の生成およびパース機能を提供するモジュールである。

### 本機能の処理概要

**業務上の目的・背景**：UUIDは分散システムにおいてグローバルに一意な識別子を生成するために広く利用される標準規格（RFC 4122）である。Juliaのパッケージ管理システム（Pkg）をはじめ、分散処理やデータベースレコードの一意識別など、さまざまな場面で一意なIDが必要となる。UUIDsモジュールは、こうした要件に対応するために、UUID version 1（時刻ベース）およびversion 4（乱数ベース）の生成機能と、文字列からのパース機能を標準ライブラリとして提供する。

**機能の利用シーン**：パッケージの一意識別（Project.tomlのUUIDフィールド）、分散システムにおけるリモート参照ID、データベースの主キー生成、テストにおけるユニークなリソース名の生成など。

**主要な処理内容**：
1. UUID型の定義（128ビット値を保持する不変構造体）
2. UUID version 4 の生成（暗号論的乱数ベース）
3. UUID version 1 の生成（時刻・ノードベース、オプション）
4. 文字列形式（"xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx"）からのパース
5. UUID型からの文字列変換および表示

**関連システム・外部連携**：Base.Randomモジュール（乱数生成）、Baseの`SHA1`型との連携。Pkgモジュールはパッケージ識別にUUIDを使用する。

**権限による制御**：特になし。全てのユーザーが制限なく利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | REPL | 参照画面 | uuid1(), uuid4(), UUID()コンストラクタの対話的実行 |

## 機能種別

データ生成 / ユーティリティ

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| rng | AbstractRNG | No | uuid4()で使用する乱数生成器。デフォルトはタスクローカルRNG | RNGインタフェースを実装していること |
| s | AbstractString | Yes (パース時) | UUID文字列表現 | 8-4-4-4-12の16進数形式であること |

### 入力データソース

- 乱数生成器（uuid4の場合）
- システムクロック（uuid1の場合）
- ユーザー入力文字列（パースの場合）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| uuid | UUID | 128ビットのUUID値。valueフィールドにUInt128として格納 |
| 文字列表現 | String | "xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx" 形式の36文字の文字列 |

### 出力先

- 戻り値としてUUID型オブジェクトを返却
- IOストリームへの表示出力

## 処理フロー

### 処理シーケンス

```
1. uuid4()呼び出し
   └─ Random.UInt128で128ビット乱数を生成
   └─ version 4ビットおよびvariantビットを設定
   └─ UUID構造体を返却

2. uuid1()呼び出し
   └─ 現在時刻を取得（100ナノ秒精度）
   └─ クロックシーケンスを生成
   └─ ノードIDを生成（乱数ベース）
   └─ version 1ビットおよびvariantビットを設定
   └─ UUID構造体を返却

3. UUID(s::String)パース
   └─ 文字列を検証（長さ36、ハイフン位置）
   └─ 16進数部分を抽出・パース
   └─ UInt128値に変換
   └─ UUID構造体を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{生成 or パース?}
    B -->|uuid4| C[乱数128ビット生成]
    C --> D[version/variantビット設定]
    D --> E[UUID構造体を返却]
    B -->|uuid1| F[タイムスタンプ取得]
    F --> G[クロックシーケンス+ノードID生成]
    G --> D
    B -->|パース| H[文字列検証]
    H --> I[16進数→UInt128変換]
    I --> E
    E --> J[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-111-01 | UUID v4形式 | version 4のビットフィールド（ビット48-51が0100）とvariant（ビット64-65が10）を正しく設定する | uuid4()呼び出し時 |
| BR-111-02 | UUID v1形式 | version 1のタイムスタンプ、クロックシーケンス、ノードIDを正しく配置する | uuid1()呼び出し時 |
| BR-111-03 | 文字列形式 | "xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx" (8-4-4-4-12)の正規形式 | 文字列表示・パース時 |

### 計算ロジック

- UUID v4: 128ビット乱数を生成後、ビット操作でversionフィールド（4ビット=0100）とvariantフィールド（2ビット=10）を埋め込む
- UUID v1: グレゴリオ暦開始日（1582-10-15）からの100ナノ秒単位のタイムスタンプ、14ビットクロックシーケンス、48ビットノードID

## データベース操作仕様

該当なし（インメモリ処理のみ）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ArgumentError | バリデーション | 不正なUUID文字列形式をパースしようとした場合 | 正しい形式の文字列を提供する |

### リトライ仕様

リトライ不要（純粋な値生成処理）。

## トランザクション仕様

該当なし（ステートレスな値生成処理）。

## パフォーマンス要件

uuid4()はナノ秒オーダーで完了する軽量な処理。暗号論的乱数生成器の性能に依存する。

## セキュリティ考慮事項

- uuid4()はデフォルトで暗号論的に安全な乱数生成器（RandomDevice）を使用する。推測困難なUUIDが必要な場合はデフォルトを使用すること。
- uuid1()はタイムスタンプとノード情報を含むため、生成元の情報が漏洩する可能性がある。

## 備考

- Julia Base内部にもUUID型が定義されており、UUIDsモジュールはこれをre-exportしている。
- Pkgモジュールで使用されるUUIDはこのモジュールのUUID型と同一。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

UUID型は128ビット値を保持するシンプルな不変構造体である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | UUIDs.jl | `stdlib/UUIDs/src/UUIDs.jl` | UUID構造体の定義（Base.UUIDをre-export）、UInt128を内部値として保持 |

**読解のコツ**: Julia標準ライブラリのモジュールは、Base内部の型定義を再利用することが多い。UUIDs.jlの`UUID`型は`Base.UUID`そのものである。

#### Step 2: エントリーポイントを理解する

公開API（uuid1, uuid4, UUID文字列コンストラクタ）がエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | UUIDs.jl | `stdlib/UUIDs/src/UUIDs.jl` | モジュールのexport宣言（5行目付近）で公開APIを確認 |

**主要処理フロー**:
1. **5行目**: `export UUID, uuid1, uuid4, uuid5, uuid7, uuid_version` でエクスポートされるシンボルを定義
2. **8行目**: `import Base: UUID` でBase.UUIDを取り込み
3. **10-50行目付近**: uuid4()の実装 -- 乱数を生成し、versionビットとvariantビットを操作
4. **uuid1()**: タイムスタンプベースのUUID生成ロジック

#### Step 3: UUID生成ロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | UUIDs.jl | `stdlib/UUIDs/src/UUIDs.jl` | uuid4()のビット操作ロジック: UInt128乱数に対してversion=4, variant=10のビットマスクを適用 |
| 3-2 | UUIDs.jl | `stdlib/UUIDs/src/UUIDs.jl` | uuid1()のタイムスタンプ計算: 1582-10-15からの100ns単位時刻を算出 |

**主要処理フロー**:
- **uuid4()**: `u = rand(rng, UInt128)` で乱数生成 -> ビットマスクで version 4 / variant 10 を設定 -> `UUID(u)` 返却
- **uuid1()**: `time_now()` でタイムスタンプ取得 -> clock_seq生成 -> node ID生成 -> ビットフィールド組み立て

### プログラム呼び出し階層図

```
UUIDs.uuid4(rng)
    |
    +-- Random.rand(rng, UInt128)       # 128ビット乱数生成
    |
    +-- ビットマスク操作                 # version=4, variant=10
    |
    +-- UUID(value::UInt128)            # 構造体生成

UUIDs.uuid1(rng)
    |
    +-- time_ns()                       # タイムスタンプ取得
    |
    +-- rand(rng, UInt128)              # clock_seq/node生成
    |
    +-- ビットフィールド組み立て
    |
    +-- UUID(value::UInt128)
```

### データフロー図

```
[入力]               [処理]                    [出力]

乱数(RNG)    ───▶  uuid4()              ───▶  UUID型オブジェクト
                    ビットマスク操作

時刻+乱数    ───▶  uuid1()              ───▶  UUID型オブジェクト
                    タイムスタンプ+ビット操作

文字列       ───▶  UUID(s)パース        ───▶  UUID型オブジェクト
                    16進数→UInt128変換
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| UUIDs.jl | `stdlib/UUIDs/src/UUIDs.jl` | ソース | メインモジュール。UUID生成・パースの全実装 |
| Project.toml | `stdlib/UUIDs/Project.toml` | 設定 | パッケージ依存関係定義 |
| runtests.jl | `stdlib/UUIDs/test/runtests.jl` | テスト | ユニットテスト |
