# 機能設計書 112-Logging

## 概要

本ドキュメントは、Julia標準ライブラリ `Logging` モジュールの機能設計を記述する。構造化されたログレベル制御、カスタムロガーの実装基盤、およびAbstractLoggerインタフェースを提供するモジュールである。

### 本機能の処理概要

**業務上の目的・背景**：ソフトウェア開発・運用において、アプリケーションの動作状態を把握するためのログ出力は不可欠である。Loggingモジュールは、Juliaの標準的なログ基盤として、ログレベル（Debug, Info, Warn, Error）による出力制御、カスタムロガーによる出力先の柔軟な切り替え、およびメタデータ（ファイル名、行番号、モジュール名）の自動付与を提供する。これにより、開発時のデバッグからプロダクション環境の監視まで統一的なログインタフェースを実現する。

**機能の利用シーン**：デバッグ情報の出力（@debug）、一般的な情報通知（@info）、警告の出力（@warn）、エラーの記録（@error）、ログ出力先の切り替え（with_logger）、ログレベルによるフィルタリング（min_enabled_level）、カスタムロガーの実装。

**主要な処理内容**：
1. ログレベルの定義と階層構造の管理（BelowMinLevel < Debug < Info < Warn < Error < AboveMaxLevel）
2. ログマクロ（@debug, @info, @warn, @error, @logmsg）による構造化ログ出力
3. AbstractLoggerインタフェースの定義（handle_message, shouldlog, min_enabled_level, catch_exceptions）
4. ConsoleLoggerによるデフォルトの標準エラー出力
5. NullLoggerによるログ抑制
6. SimpleLoggerによるシンプルなIO出力
7. グローバルロガーおよびタスクローカルロガーの管理（global_logger, with_logger, current_logger）

**関連システム・外部連携**：Base.CoreLoggingモジュール（コアログ基盤）、標準エラー出力（stderr）。

**権限による制御**：特になし。全てのユーザーが制限なく利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | REPL / ターミナル | 主画面 | ログ出力の表示、ロガー切り替え |

## 機能種別

ログ制御 / フレームワーク基盤

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| level | LogLevel | Yes | ログレベル（Debug/Info/Warn/Error） | LogLevel型であること |
| message | Any | Yes | ログメッセージ（遅延評価） | 任意の値（string変換される） |
| _module | Module | No | ログ発生モジュール（マクロが自動設定） | - |
| _group | Symbol | No | ロググループ（デフォルトはファイルベース名） | - |
| _id | Symbol | No | ログメッセージの一意識別子 | - |
| _file | String | No | ソースファイルパス（マクロが自動設定） | - |
| _line | Int | No | ソース行番号（マクロが自動設定） | - |
| kwargs... | Any | No | 追加のキーバリューペア | - |

### 入力データソース

- ソースコード中のログマクロ呼び出し
- 実行時のモジュール・ファイル・行番号情報

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ログ出力 | テキスト | フォーマットされたログメッセージ（レベル、モジュール、ファイル:行番号、メッセージ、追加キーバリュー） |

### 出力先

- ConsoleLogger: stderr（デフォルト）
- SimpleLogger: 指定されたIOストリーム
- NullLogger: 出力なし（全てのログを破棄）
- カスタムロガー: ユーザー定義の出力先

## 処理フロー

### 処理シーケンス

```
1. ログマクロ（@info等）が呼び出される
   └─ コンパイル時にメタデータ（module, file, line, id, group）を取得
2. 現在のロガーを取得（current_logger()）
   └─ タスクローカルロガー → グローバルロガーの優先順で検索
3. shouldlog()で出力可否を判定
   └─ ログレベル、モジュール、グループ、IDに基づくフィルタリング
4. handle_message()でログを処理
   └─ メッセージの遅延評価（shouldlog=trueの場合のみ）
   └─ フォーマットと出力先への書き込み
5. 例外処理（catch_exceptionsがtrueの場合）
   └─ handle_message内の例外をキャッチしてログ出力失敗を通知
```

### フローチャート

```mermaid
flowchart TD
    A["@info/@warn等マクロ呼び出し"] --> B["メタデータ取得（module,file,line）"]
    B --> C["current_logger()でロガー取得"]
    C --> D{"min_enabled_level チェック"}
    D -->|レベル不足| E["スキップ（何もしない）"]
    D -->|レベル十分| F{"shouldlog() チェック"}
    F -->|false| E
    F -->|true| G["メッセージを遅延評価"]
    G --> H["handle_message() 呼び出し"]
    H --> I{"catch_exceptions?"}
    I -->|true| J["try-catch で例外処理"]
    I -->|false| K["直接出力"]
    J --> L["終了"]
    K --> L
    E --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-112-01 | ログレベル階層 | BelowMinLevel < Debug(-1000) < Info(0) < Warn(1000) < Error(2000) < AboveMaxLevel | 全てのログ出力時 |
| BR-112-02 | 遅延評価 | メッセージ式はshouldlog()がtrueを返した場合のみ評価される | ログマクロ使用時 |
| BR-112-03 | ロガー優先順位 | タスクローカルロガーがグローバルロガーより優先される | with_logger使用時 |
| BR-112-04 | デフォルト出力先 | ConsoleLoggerがデフォルトロガーとしてstderrに出力 | 明示的なロガー設定がない場合 |

### 計算ロジック

- ログレベルは整数値で比較される。LogLevel(level)のlevelフィールドがInt32で定義されている。
- shouldlog()はmin_enabled_level()の結果と個別のshouldlog()判定の2段階で評価される。

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ログ出力例外 | handle_message内でIO例外が発生 | catch_exceptions=trueの場合は例外をキャッチしstderrに通知 |
| - | ロガー未設定 | current_loggerがnothingを返す場合 | global_loggerがデフォルトとして機能 |

### リトライ仕様

ログ出力の失敗に対するリトライは行わない。catch_exceptions=trueの場合、例外はキャッチされ警告が出力される。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- shouldlog()がfalseを返す場合、メッセージの文字列化は行われない（遅延評価）
- min_enabled_levelによる早期リターンでオーバーヘッドを最小化
- ログマクロはコンパイル時にメタデータを解決するため、実行時のオーバーヘッドが小さい

## セキュリティ考慮事項

- ログに機密情報（パスワード、トークン等）が含まれないよう注意が必要
- カスタムロガーの実装時、ログ出力先のアクセス制御を適切に設定すること

## 備考

- Loggingモジュールの基盤はBase.CoreLoggingに定義されており、Loggingモジュールはこれをre-exportしている
- ConsoleLoggerは色付き出力に対応し、ターミナル幅に応じてフォーマットを調整する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

LogLevelの定義とAbstractLoggerインタフェースを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Logging.jl | `stdlib/Logging/src/Logging.jl` | LogLevel型の定義、AbstractLoggerの抽象インタフェース、NullLogger/SimpleLoggerの実装 |

**読解のコツ**: LoggingモジュールはBase.CoreLoggingからほぼ全てをre-exportしている。実際の実装の多くはBase内部にある。

#### Step 2: エントリーポイントを理解する

ログマクロ（@info等）とロガー管理関数がエントリーポイント。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Logging.jl | `stdlib/Logging/src/Logging.jl` | 8-15行目: Base.CoreLoggingからのインポート宣言で公開APIの全体像を把握 |

**主要処理フロー**:
1. **8-15行目**: `import Base.CoreLogging:` で@debug, @info, @warn, @error等のマクロと、AbstractLogger, ConsoleLogger等の型をインポート
2. **16行目**: `using Base.CoreLogging` で全シンボルを利用可能に

#### Step 3: ConsoleLoggerの実装を理解する

デフォルトのログ出力先であるConsoleLoggerの動作を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | logging.jl | `base/logging/logging.jl` | CoreLogging内のAbstractLoggerプロトコル定義、handle_message/shouldlog/min_enabled_level |

**主要処理フロー**:
- ConsoleLoggerはstderrへの色付きログ出力を行う
- ターミナル幅に基づいてメッセージの折り返しを制御
- ログレベルに応じた色付けを実施

### プログラム呼び出し階層図

```
@info "message" key=value
    |
    +-- CoreLogging.@logmsg(Info, "message", key=value)
        |
        +-- current_logger()                  # ロガー取得
        |   +-- task_local_storage(:CURRENT_LOGGER)
        |   +-- global_logger()               # fallback
        |
        +-- min_enabled_level(logger)         # 早期フィルタ
        |
        +-- shouldlog(logger, level, ...)     # 詳細フィルタ
        |
        +-- handle_message(logger, level, message, ...)  # 出力
            |
            +-- ConsoleLogger: write to stderr
            +-- NullLogger: no-op
            +-- SimpleLogger: write to io
```

### データフロー図

```
[入力]               [処理]                          [出力]

ソースコード内    ───▶  ログマクロ展開           ───▶  メタデータ付きログ
@info "msg"            + メタデータ取得                レコード
                       + ロガー選択
                       + shouldlogフィルタ
                       + handle_message出力

ロガー設定       ───▶  with_logger/              ───▶  タスクローカル
(Logger型)             global_logger                    ストレージ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Logging.jl | `stdlib/Logging/src/Logging.jl` | ソース | メインモジュール。CoreLoggingのre-export |
| logging.jl | `base/logging/logging.jl` | ソース | CoreLogging実装。ログマクロ、AbstractLogger、ロガー管理 |
| ConsoleLogger.jl | `base/logging/ConsoleLogger.jl` | ソース | ConsoleLoggerの実装 |
| Project.toml | `stdlib/Logging/Project.toml` | 設定 | パッケージ依存関係定義 |
| runtests.jl | `stdlib/Logging/test/runtests.jl` | テスト | ユニットテスト |
