# 機能設計書 12-多倍長浮動小数点（BigFloat）

## 概要

本ドキュメントは、Julia言語における多倍長浮動小数点（BigFloat）機能の設計を記述する。BigFloatはMPFR（Multiple Precision Floating-Point Reliable）ライブラリを基盤とした任意精度浮動小数点演算を提供する。

### 本機能の処理概要

BigFloat型は、固定長浮動小数点型（Float64等）では精度が不足する場面で、ユーザーが指定した任意の精度（デフォルト256ビット）で浮動小数点演算を行うための型である。MPFRライブラリのmpfr系関数をccallで直接呼び出し、正確な丸めを保証した高精度演算を実現する。

**業務上の目的・背景**：科学技術計算、数値解析、数学定数の高精度計算などでは、Float64の約15-16桁の有効桁数では不十分な場合がある。BigFloatは任意精度の浮動小数点演算を提供し、計算精度をユーザーが制御可能にすることで、数値的安定性の検証や超高精度計算を可能にする。MPFRライブラリはIEEE 754準拠の正確な丸めを保証する。

**機能の利用シーン**：（1）円周率や自然対数の底などの数学定数の高精度計算、（2）数値解析アルゴリズムの精度検証、（3）区間演算の端点計算、（4）暗号学的な浮動小数点演算、（5）setprecision/setrroundingによる精度と丸めモードの動的制御。

**主要な処理内容**：
1. BigFloat型の定義（Memory{Limb}ベースのインライン格納方式）
2. 精度制御（setprecision/precision）と丸めモード制御（setrounding/rounding）
3. 基本算術演算（+, -, *, /, ^）のMPFR経由実装
4. 数学関数（三角関数、指数関数、対数関数等）のMPFR経由実装
5. 各種数値型（Integer, Float64, BigInt, Rational, String）からの変換
6. ScopedValueによるタスクローカルな精度・丸めモード管理

**関連システム・外部連携**：MPFR（Multiple Precision Floating-Point Reliable）ライブラリのlibmpfrと直接連携。GMPライブラリにも依存（BigIntとの相互変換時）。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | Juliaプロンプト（julia>） | 参照画面 | REPL上でBigFloat演算を対話的に実行・結果表示 |
| 3 | ヘルプモード（help?>） | 参照画面 | BigFloat関連のdocstring表示 |

## 機能種別

計算処理（任意精度浮動小数点演算）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| x | Real / AbstractString | Yes | BigFloatに変換する値 | - |
| rounding | RoundingMode | No | 丸めモード（デフォルト: RoundNearest） | MPFRRoundingModeに変換可能な値 |
| precision | Integer | No | ビット精度（デフォルト: 256） | 1以上 |

### 入力データソース

Julia式からの直接入力。parse(BigFloat, s)による文字列パース。big"..."文字列マクロ。widen(Float64)による型昇格。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | BigFloat | 演算結果の任意精度浮動小数点数 |
| string_repr | String | string/show による文字列表現 |

### 出力先

演算結果はメモリ上のBigFloatオブジェクトとして返却。

## 処理フロー

### 処理シーケンス

```
1. BigFloatオブジェクト生成
   └─ Memory{Limb}を確保し、precision/sign/exp/dポインタを初期化（NaNで初期化）
2. 値の設定
   └─ 型に応じてmpfr_set_si/set_d/set_z/set_str等を呼び出し
3. 演算実行
   └─ ccallでMPFRのmpfr系関数を呼び出し（丸めモード付き）
4. 結果の返却
   └─ 新規BigFloatオブジェクトに結果を格納して返却
```

### フローチャート

```mermaid
flowchart TD
    A[BigFloat生成要求] --> B{入力型の判定}
    B -->|Clong/Culong| C[mpfr_set_si/ui]
    B -->|Float64| D[精度判定]
    D -->|precision <= 52| E[mpfr_set_d]
    D -->|precision > 52| F[手動ビット設定]
    B -->|BigInt| G[mpfr_set_z]
    B -->|String| H[mpfr_set_str]
    B -->|Rational| I[num/den個別変換後除算]
    C --> J[BigFloatオブジェクト返却]
    E --> J
    F --> J
    G --> J
    H --> J
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 正確な丸め保証 | MPFRによるIEEE 754準拠の正確な丸め | 全BigFloat演算 |
| BR-02 | デフォルト精度 | 精度のデフォルトは256ビット | BigFloat()コンストラクタ |
| BR-03 | グローバル精度設定 | setprecisionで全体の精度を変更可能 | setprecision呼び出し時 |
| BR-04 | 丸めモード | RoundNearest/ToZero/Up/Down/FromZeroを選択可能 | 全BigFloat演算 |
| BR-05 | NaN初期化 | 新規BigFloatはNaNで初期化される | コンストラクタ呼び出し時 |
| BR-06 | ScopedValue精度 | タスクローカルな精度・丸めモード設定が可能 | CURRENT_PRECISION/CURRENT_ROUNDING_MODE |

### 計算ロジック

- BigFloat内部表現: prec（精度ビット数）、sign（符号）、exp（指数）、d（仮数部のlimb配列）
- Memory{Limb}にインラインで全データを格納する方式（offset_prec, offset_sign, offset_exp, offset_dで各フィールドの位置を管理）
- Float64からの変換で precision > 52 の場合、mpfr_set_dのオーバーヘッドを避けて手動でビット設定

## データベース操作仕様

該当なし（本機能はデータベースを使用しない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | DomainError | precision < 1 | 1以上の精度を指定 |
| - | ArgumentError | 不正な丸めモードコード | 有効なRoundingModeを指定 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし（インメモリ計算のみ）

## パフォーマンス要件

- Float64からの変換で precision > significand_bits(Float64) の場合は手動ビット設定で高速化
- Memory{Limb}へのインライン格納により、別途のGCオブジェクト管理が不要
- ScopedValueによるタスクローカルな精度管理でスレッドセーフな精度切り替え

## セキュリティ考慮事項

- unsafe_load/unsafe_store!によるポインタ操作はBigFloat内部のMemory{Limb}に限定
- unsafe_convert(Ref{BigFloat})で内部ポインタの整合性を検証

## 備考

- BigFloatはstruct（immutable）だが、sign/expフィールドはsetproperty!で変更可能（アルゴリズム実装上の理由）
- rawbigfloats.jlがincludeされており、RawBigFloat等の低レベルインタフェースも提供
- widen(Float64) = BigFloat により、浮動小数点型昇格チェーンの終端を担う

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

BigFloatの内部レイアウトとMemory{Limb}ベースの格納方式を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | mpfr.jl | `base/mpfr.jl` | BigFloatLayout構造体（124-131行目）とオフセット定数（132-137行目） |
| 1-2 | mpfr.jl | `base/mpfr.jl` | BigFloat構造体定義（144-172行目）：Memory{Limb}によるインライン格納 |
| 1-3 | mpfr.jl | `base/mpfr.jl` | BigFloatData構造体（178-180行目）とアクセサ（183-243行目） |

**読解のコツ**: BigFloatは通常の構造体フィールドではなくMemory{Limb}にMPFR mpfr_t構造体をインラインで格納する。getproperty/setproperty!でフィールドアクセスをエミュレートし、offset_prec等の定数でバイトオフセットを管理する。

#### Step 2: 精度・丸めモード管理を理解する

グローバルおよびタスクローカルな精度・丸めモード管理。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | mpfr.jl | `base/mpfr.jl` | MPFRRoundingMode列挙型（66-73行目） |
| 2-2 | mpfr.jl | `base/mpfr.jl` | ROUNDING_MODE/DEFAULT_PRECISION定数（112-115行目） |
| 2-3 | mpfr.jl | `base/mpfr.jl` | ScopedValueによるタスクローカル管理（113-115行目） |

#### Step 3: コンストラクタと型変換を理解する

各種数値型からBigFloatへの変換。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | mpfr.jl | `base/mpfr.jl` | BigFloat(::BigFloat)コンストラクタ（303-312行目） |
| 3-2 | mpfr.jl | `base/mpfr.jl` | BigFloat(::Clong/Culong)（321-329行目）：mpfr_set_si/ui |
| 3-3 | mpfr.jl | `base/mpfr.jl` | BigFloat(::Float64)（331-373行目）：精度に応じた最適化分岐 |
| 3-4 | mpfr.jl | `base/mpfr.jl` | BigFloat(::BigInt)（375-379行目）：mpfr_set_z |
| 3-5 | mpfr.jl | `base/mpfr.jl` | BigFloat(::Rational)（394-399行目）：分子分母個別変換 |

### プログラム呼び出し階層図

```
BigFloat(x) [コンストラクタ]
    |
    +-- BigFloat(;precision=...)     [Memory{Limb}割り当て、NaN初期化]
    |       +-- mpfr_custom_get_size(precision)
    |       +-- Memory{Limb}(undef, nl)
    |       +-- unsafe_store! (prec, sign, exp, d)
    |
    +-- ccall(mpfr_set_*, ...)       [値の設定]
    |       +-- mpfr_set_si / mpfr_set_d / mpfr_set_z / mpfr_set_str
    |
    +-- 算術演算: +(x, y)
    |       +-- ccall(mpfr_add, ...)
    |
    +-- 数学関数: sin(x)
    |       +-- ccall(mpfr_sin, ...)
    |
    +-- show(io, x)
            +-- _string(x) -> ccall(mpfr_get_str, ...)
```

### データフロー図

```
[入力]                    [処理]                              [出力]

Integer          ──────> mpfr_set_si/ui (ccall MPFR) ──────> BigFloat
Float64          ──────> 手動ビット設定 or mpfr_set_d ──────> BigFloat
BigInt           ──────> mpfr_set_z (ccall MPFR) ──────────> BigFloat
String           ──────> mpfr_set_str (ccall MPFR) ────────> BigFloat
BigFloat a, b    ──────> mpfr_add/sub/mul/div (ccall) ─────> BigFloat
BigFloat x       ──────> mpfr_sin/cos/exp/log (ccall) ─────> BigFloat
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| mpfr.jl | `base/mpfr.jl` | ソース | BigFloat型定義、MPFR関数ラッパー、算術演算、数学関数、型変換 |
| rawbigfloats.jl | `base/rawbigfloats.jl` | ソース | RawBigFloat低レベルインタフェース |
| gmp.jl | `base/gmp.jl` | ソース | BigInt定義（BigFloatとの相互変換に使用） |
| rounding.jl | `base/rounding.jl` | ソース | RoundingMode定義 |
| MPFR_jll | `stdlib/MPFR_jll/` | JLL | libmpfrバイナリの提供 |
| GMP_jll | `stdlib/GMP_jll/` | JLL | libgmpバイナリの提供（MPFR依存） |
