# 機能設計書 121-LinearAlgebra

## 概要

本ドキュメントは、Julia標準ライブラリの LinearAlgebra パッケージに関する機能設計書である。LinearAlgebra は行列演算・行列分解（LU/QR/SVD/Eigen）・BLAS/LAPACK ラッパーを提供する外部パッケージとして Julia に同梱されている。

### 本機能の処理概要

LinearAlgebra は、科学技術計算・機械学習・統計解析などで必要となる線形代数演算の包括的なライブラリである。高性能な行列演算をJuliaの多重ディスパッチシステムを活用して提供する。

**業務上の目的・背景**：線形代数はほぼすべての科学技術計算の基盤であり、行列演算の高速かつ正確な実装はプログラミング言語の計算能力を決定づける。Julia は数値計算を主要なユースケースとしており、LinearAlgebra はその中核的な標準ライブラリである。BLAS/LAPACK による最適化されたネイティブ実装を活用することで、C/Fortran と同等の性能を Julia から利用できるようにする。

**機能の利用シーン**：連立一次方程式の求解、最小二乗法、固有値問題、特異値分解、行列のノルム計算、線形回帰分析、主成分分析（PCA）、物理シミュレーションの行列計算など、科学技術計算のあらゆる場面で利用される。

**主要な処理内容**：
1. 行列演算（加減乗除、転置、共役転置、トレース、行列式、逆行列）
2. 行列分解（LU分解、QR分解、コレスキー分解、SVD、固有値分解、シュール分解、ヘッセンベルク分解）
3. BLAS レベル1/2/3ルーチンのラッパー（dot, gemv, gemm, trsm 等）
4. LAPACK ルーチンのラッパー（getrf, geqrf, gesvd, syev 等）
5. 特殊行列型（対称行列、エルミート行列、三角行列、対角行列、帯行列、三重対角行列）
6. ノルム・条件数の計算
7. 線形方程式系の求解（`\` 演算子）

**関連システム・外部連携**：OpenBLAS_jll によるBLAS/LAPACK実装、SuiteSparse_jll による疎行列向けルーチン。SparseArrays パッケージと密接に連携する。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に直接関連する操作はない（ライブラリとして利用） |

## 機能種別

計算処理（線形代数演算ライブラリ）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| A | AbstractMatrix | Yes | 演算対象の行列 | 次元の整合性チェック |
| B | AbstractVecOrMat | 条件付き | 第二オペランドの行列またはベクトル | Aとの次元整合性 |
| uplo | Char | No | 上三角('U')/下三角('L')の指定 | 'U'または'L' |
| pivot | PivotingStrategy | No | ピボット戦略の指定 | NoPivot/RowMaximum/ColumnNorm等 |

### 入力データソース

Julia のプログラムコードからの直接呼び出し。行列データはメモリ上の Array{T,2} オブジェクト。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Factorization | LU/QR/SVD/Eigen等 | 行列分解の結果オブジェクト |
| Matrix | Matrix{T} | 演算結果の行列 |
| Vector | Vector{T} | 固有値・特異値等のベクトル |
| Scalar | Number | 行列式・ノルム・条件数等のスカラ値 |

### 出力先

呼び出し元のJuliaプログラムへの戻り値として返却。

## 処理フロー

### 処理シーケンス

```
1. 型ディスパッチによる最適アルゴリズムの選択
   └─ 行列の型（Dense/Sparse/Symmetric/Triangular等）に応じた分岐
2. 入力検証（次元チェック、特殊行列属性の確認）
   └─ DimensionMismatch等の例外をスロー
3. BLAS/LAPACKルーチンの呼び出し（該当する場合）
   └─ ccallによるFortranルーチンの呼び出し
4. 結果のJuliaオブジェクトへのラッピング
   └─ Factorizationオブジェクトの構築
5. 結果の返却
   └─ 分解オブジェクトまたは計算結果の返却
```

### フローチャート

```mermaid
flowchart TD
    A[入力行列 A] --> B{行列型の判定}
    B -->|Dense| C[BLAS/LAPACK呼び出し]
    B -->|Symmetric| D[対称行列専用ルーチン]
    B -->|Triangular| E[三角行列専用ルーチン]
    B -->|Diagonal| F[対角行列高速パス]
    C --> G[結果をFactorizationにラップ]
    D --> G
    E --> G
    F --> G
    G --> H[結果返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-121-01 | 次元整合性 | 行列演算時に次元の整合性をチェック | すべての行列演算 |
| BR-121-02 | 型昇格 | 異なる要素型の行列演算時にpromote_typeで型を統一 | 異なる要素型の行列間演算 |
| BR-121-03 | 特殊行列最適化 | 対称・三角・対角行列等の構造を活用した最適化 | 特殊行列型が検出された場合 |
| BR-121-04 | in-place演算 | `!`付き関数による破壊的操作でメモリ割り当て削減 | パフォーマンスが重要な場合 |

### 計算ロジック

- LU分解: PA = LU（部分ピボット付き）
- QR分解: A = QR（ハウスホルダー反射法）
- SVD: A = USV'（特異値分解）
- 固有値分解: AV = VD（固有値・固有ベクトル）
- コレスキー分解: A = LL'（正定値対称行列）
- `\` 演算子: 行列型に応じた最適ソルバの自動選択

## データベース操作仕様

### 操作別データベース影響一覧

データベースは使用しない。すべての操作はメモリ上で完結する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | DimensionMismatch | 行列の次元が演算に不適合 | 入力行列の次元を確認 |
| - | SingularException | 特異行列に対するLU分解・逆行列計算 | 正則行列を入力するか擬似逆行列を使用 |
| - | PosDefException | 非正定値行列に対するコレスキー分解 | 入力行列の正定値性を確認 |
| - | LAPACKException | LAPACK内部エラー | info値に基づくエラー判定 |
| - | ArgumentError | 不正な引数（uplo指定等） | 引数の妥当性を確認 |

### リトライ仕様

リトライは不要。数値計算エラーは入力データの問題であるため。

## トランザクション仕様

トランザクションは使用しない。メモリ上の計算処理のみ。

## パフォーマンス要件

BLAS/LAPACK による最適化された実装を使用。マルチスレッドBLAS（OpenBLAS）による並列化に対応。大規模行列の演算ではBLAS_NUM_THREADSによるスレッド数制御が可能。

## セキュリティ考慮事項

特筆すべきセキュリティ要件はない。メモリ上の数値演算のみ。

## 備考

- LinearAlgebra は外部パッケージ（https://github.com/JuliaLang/LinearAlgebra.jl）として管理されている
- Julia本体には `stdlib/LinearAlgebra.version` ファイルでバージョン・SHA1が管理されている
- コミットSHA1: 0eaade81fae264f9802df0e79b24ebc625f21305
- `using LinearAlgebra` で読み込み可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: パッケージ参照設定を理解する

LinearAlgebra は外部パッケージであるため、Julia本体側ではバージョン管理ファイルのみが存在する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | LinearAlgebra.version | `stdlib/LinearAlgebra.version` | ブランチ・SHA1・Git URLの管理方法を理解する |

**読解のコツ**: `.version` ファイルは Makefile 変数形式で記述されている。`*_SHA1` がビルドに含まれるコミットを特定する。

#### Step 2: ビルドシステムとの統合を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdlib.mk | `stdlib/stdlib.mk` | 外部stdlibのダウンロード・展開・ビルドの仕組みを理解する |

**主要処理フロー**:
1. `stdlib/LinearAlgebra.version` からSHA1を読み取り
2. GitHub APIからtarballをダウンロード
3. `stdlib/LinearAlgebra-{SHA1}/` に展開
4. システムイメージまたはパッケージイメージとしてプリコンパイル

#### Step 3: ドキュメント統合を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | make.jl | `doc/make.jl` | **49行目**: `EXT_STDLIB_DOCS`にPkgのみ含まれ、LinearAlgebraは通常のSTDLIB_DOCSとして処理される |

### プログラム呼び出し階層図

```
using LinearAlgebra
    |
    +-- Base.require("LinearAlgebra")
    |       +-- パッケージイメージの読み込み
    |       +-- モジュール初期化
    |
    +-- LinearAlgebra モジュール
            |
            +-- BLAS サブモジュール
            |       +-- ccall(:dgemm_, ...) [Level 3 BLAS]
            |       +-- ccall(:dgemv_, ...) [Level 2 BLAS]
            |       +-- ccall(:ddot_, ...)  [Level 1 BLAS]
            |
            +-- LAPACK サブモジュール
            |       +-- ccall(:dgetrf_, ...) [LU分解]
            |       +-- ccall(:dgeqrf_, ...) [QR分解]
            |       +-- ccall(:dgesvd_, ...) [SVD]
            |       +-- ccall(:dsyev_, ...)  [固有値分解]
            |
            +-- 特殊行列型
            |       +-- Symmetric / Hermitian
            |       +-- UpperTriangular / LowerTriangular
            |       +-- Diagonal / Bidiagonal / Tridiagonal
            |
            +-- Factorization型
                    +-- LU / QR / Cholesky
                    +-- SVD / Eigen / Schur / Hessenberg
```

### データフロー図

```
[入力]                   [処理]                        [出力]

Matrix{T}       ──> 型ディスパッチ               ──> Factorizationオブジェクト
                       |
                       +-- BLAS/LAPACK ccall    ──> 分解結果の行列/ベクトル
                       |
                       +-- Julia純粋実装         ──> スカラ値（det, norm等）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| LinearAlgebra.version | `stdlib/LinearAlgebra.version` | 設定 | パッケージバージョン・SHA1管理 |
| make.jl | `doc/make.jl` | ビルドスクリプト | ドキュメント生成時のLinearAlgebra統合 |
| OpenBLAS_jll | `stdlib/OpenBLAS_jll/` | JLLパッケージ | BLAS/LAPACK実装の提供 |
| SuiteSparse_jll | `stdlib/SuiteSparse_jll/` | JLLパッケージ | 疎行列ルーチンの提供 |
