# 機能設計書 122-SparseArrays

## 概要

本ドキュメントは、Julia標準ライブラリの SparseArrays パッケージに関する機能設計書である。SparseArrays は疎行列（SparseMatrixCSC）と SuiteSparse ラッパーを提供する外部パッケージとして Julia に同梱されている。

### 本機能の処理概要

SparseArrays は、非ゼロ要素が少ない大規模行列を効率的に格納・操作するためのデータ構造とアルゴリズムを提供する。CSC（Compressed Sparse Column）形式を主要なストレージフォーマットとして採用し、疎行列特有の最適化された演算を実装している。

**業務上の目的・背景**：大規模な科学技術計算・グラフ解析・有限要素法・ネットワーク分析などでは、行列のほとんどの要素がゼロである疎行列が頻出する。密行列として格納すると膨大なメモリが必要となるため、非ゼロ要素のみを効率的に格納する疎行列データ構造が不可欠である。SuiteSparse ライブラリとの連携により、疎行列の直接法ソルバ（UMFPACK/CHOLMOD）を利用可能にする。

**機能の利用シーン**：有限要素法の剛性行列の組立・解法、グラフのラプラシアン行列演算、大規模疎線形方程式系の求解、ネットワーク解析、推薦システムの行列分解など。

**主要な処理内容**：
1. SparseMatrixCSC 型による疎行列の生成・格納
2. sparse/sparsevec/spzeros/sprand 等の疎行列生成関数
3. 疎行列の算術演算（加減乗除）
4. 疎行列のインデックスアクセス・スライス
5. SuiteSparse（UMFPACK/CHOLMOD）を用いた疎行列分解
6. findnz / nnz / nonzeros / rowvals 等の疎行列ユーティリティ
7. 密行列との相互変換

**関連システム・外部連携**：SuiteSparse_jll（UMFPACK/CHOLMOD）、LinearAlgebra パッケージとの連携。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に直接関連する操作はない（ライブラリとして利用） |

## 機能種別

計算処理（疎行列演算ライブラリ）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| I | Vector{Int} | Yes | 非ゼロ要素の行インデックス | 1以上、行数以下 |
| J | Vector{Int} | Yes | 非ゼロ要素の列インデックス | 1以上、列数以下 |
| V | Vector{T} | Yes | 非ゼロ要素の値 | I, J と同じ長さ |
| m | Int | No | 行数（省略時はmax(I)） | 正の整数 |
| n | Int | No | 列数（省略時はmax(J)） | 正の整数 |

### 入力データソース

Julia のプログラムコードからの直接呼び出し。COO（Coordinate）形式またはCSC形式のデータ。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| SparseMatrixCSC | SparseMatrixCSC{Tv,Ti} | CSC形式の疎行列 |
| SparseVector | SparseVector{Tv,Ti} | 疎ベクトル |
| Factorization | UMFPACK.UmfpackLU等 | 疎行列分解の結果 |
| nnz | Int | 非ゼロ要素数 |

### 出力先

呼び出し元のJuliaプログラムへの戻り値として返却。

## 処理フロー

### 処理シーケンス

```
1. COO形式の入力データ（I, J, V）を受け取る
   └─ sparse(I, J, V, m, n) の呼び出し
2. COO形式からCSC形式への変換
   └─ 列ポインタ・行インデックス・値の配列を構築
3. 重複エントリの結合（combine関数の適用）
   └─ デフォルトは加算（+）
4. SparseMatrixCSCオブジェクトの返却
   └─ colptr, rowval, nzval の3配列で構成
```

### フローチャート

```mermaid
flowchart TD
    A[COO入力 I,J,V] --> B[ソート・重複検出]
    B --> C[CSC形式変換]
    C --> D[colptr配列構築]
    D --> E[rowval配列構築]
    E --> F[nzval配列構築]
    F --> G[SparseMatrixCSCオブジェクト]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-122-01 | CSC形式 | Compressed Sparse Column形式でデータを格納 | すべての疎行列 |
| BR-122-02 | 重複結合 | 同一位置の重複エントリはcombine関数で結合 | sparse関数の呼び出し時 |
| BR-122-03 | ゼロ除去 | dropzeros!で明示的なゼロエントリを除去可能 | 演算後のクリーンアップ時 |
| BR-122-04 | LinearAlgebra連携 | `\`演算子でUMFPACK/CHOLMODを自動選択 | 疎行列の線形方程式求解時 |

### 計算ロジック

- CSC形式: colptr[j]:colptr[j+1]-1 の範囲が列jの非ゼロ要素を示す
- 疎行列乗算: 非ゼロパターンの積集合に基づく効率的な演算

## データベース操作仕様

### 操作別データベース影響一覧

データベースは使用しない。すべての操作はメモリ上で完結する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | DimensionMismatch | 疎行列演算の次元不整合 | 入力の次元を確認 |
| - | BoundsError | 範囲外インデックスアクセス | インデックスの妥当性を確認 |
| - | ArgumentError | 不正なCSCデータ構造 | colptr/rowval/nzvalの整合性を確認 |
| - | SingularException | 特異疎行列の分解 | 行列の正則性を確認 |

### リトライ仕様

リトライは不要。

## トランザクション仕様

トランザクションは使用しない。

## パフォーマンス要件

疎行列演算はO(nnz)の時間計算量で動作する。SuiteSparse（UMFPACK/CHOLMOD）による最適化された直接法ソルバを使用。

## セキュリティ考慮事項

特筆すべきセキュリティ要件はない。

## 備考

- SparseArrays は外部パッケージ（https://github.com/JuliaSparse/SparseArrays.jl）として管理されている
- Julia本体には `stdlib/SparseArrays.version` ファイルでバージョン管理されている
- コミットSHA1: 26c80c8b45dc2dca92788332a40a99b6c360d05a
- ドキュメントビルド時にLinearAlgebraとセットでDocTestSetupが設定される（doc/make.jl 364行目）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: パッケージ参照設定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | SparseArrays.version | `stdlib/SparseArrays.version` | ブランチ(main)・SHA1・Git URL(JuliaSparse/SparseArrays.jl)の管理方法 |

**読解のコツ**: SparseArraysはJuliaLangではなくJuliaSparse組織のリポジトリで管理されている点に注意。

#### Step 2: ドキュメントビルドとの統合を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | make.jl | `doc/make.jl` | **361-366行目**: SparseArraysのDocTestSetupでLinearAlgebraも同時にusingされる設定 |

**主要処理フロー**:
1. **361行目**: `DocMeta.setdocmeta!(SparseArrays, ...)` でSparseArraysのドキュメントテスト設定
2. **364行目**: `using SparseArrays, LinearAlgebra` がDocTestSetupとして設定

#### Step 3: ビルドシステムでの外部パッケージ取得を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdlib.mk | `stdlib/stdlib.mk` | 外部パッケージのtar取得・展開処理のMakefileルール |

### プログラム呼び出し階層図

```
using SparseArrays
    |
    +-- Base.require("SparseArrays")
    |       +-- パッケージイメージの読み込み
    |
    +-- SparseArrays モジュール
            |
            +-- SparseMatrixCSC{Tv,Ti} 型
            |       +-- colptr::Vector{Ti}
            |       +-- rowval::Vector{Ti}
            |       +-- nzval::Vector{Tv}
            |
            +-- sparse / sparsevec / spzeros / sprand
            |
            +-- UMFPACK サブモジュール
            |       +-- UmfpackLU (LU分解)
            |
            +-- CHOLMOD サブモジュール
            |       +-- CholmodFactor (コレスキー分解)
            |
            +-- findnz / nnz / nonzeros / rowvals
```

### データフロー図

```
[入力]                   [処理]                        [出力]

COO (I,J,V)     ──> sparse()変換             ──> SparseMatrixCSC
密行列 Matrix    ──> sparse()変換             ──> SparseMatrixCSC
SparseMatrixCSC  ──> UMFPACK LU分解          ──> UmfpackLU
SparseMatrixCSC  ──> Array()変換             ──> Matrix (密行列)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| SparseArrays.version | `stdlib/SparseArrays.version` | 設定 | パッケージバージョン管理 |
| make.jl | `doc/make.jl` | ビルドスクリプト | ドキュメント生成設定 |
| SuiteSparse_jll | `stdlib/SuiteSparse_jll/` | JLLパッケージ | UMFPACK/CHOLMOD実装 |
