# 機能設計書 124-Distributed

## 概要

本ドキュメントは、Julia標準ライブラリの Distributed パッケージに関する機能設計書である。Distributed はマルチプロセス並列計算（addprocs / @distributed / pmap / RemoteChannel）を提供する外部パッケージとして Julia に同梱されている。

### 本機能の処理概要

Distributed は、複数のJuliaプロセス（ワーカー）を起動・管理し、プロセス間通信を通じた分散並列計算を実現するライブラリである。共有メモリではなくメッセージパッシングベースの並列計算モデルを採用し、単一マシン内およびクラスタ環境での並列実行を可能にする。

**業務上の目的・背景**：計算量の大きなタスクを複数のプロセスに分散して高速化する。マルチスレッド（Threads）とは異なり、各ワーカーが独立したメモリ空間を持つため、GIL（Global Interpreter Lock）的な制約がなく、完全な並列実行が可能。クラスタ環境での分散計算にも対応し、大規模な科学技術計算やデータ処理に適している。

**機能の利用シーン**：大規模モンテカルロシミュレーション、パラメータスイープ、データ並列処理、MapReduce計算、分散機械学習、クラスタでの並列科学計算など。

**主要な処理内容**：
1. ワーカープロセスの起動・管理（addprocs / rmprocs / workers / nworkers）
2. リモート関数実行（@spawnat / remotecall / fetch）
3. 分散ループ（@distributed for / pmap）
4. リモートチャネル（RemoteChannel）によるプロセス間通信
5. 共有参照（Future / RemoteRef）
6. クラスタマネージャ（SSHManager / LocalManager）
7. データ転送とシリアライゼーション

**関連システム・外部連携**：SSH接続（クラスタ環境）、Serialization パッケージ、Sockets パッケージ。

**権限による制御**：リモートプロセス起動にはSSH認証が必要。ローカルプロセスは特別な権限不要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に直接関連する操作はない（ライブラリとして利用） |

## 機能種別

計算処理（分散並列計算ライブラリ）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| np | Int | Yes（addprocs） | 追加するワーカー数 | 正の整数 |
| machines | Vector{String} | No | リモートマシンのホスト名リスト | SSH接続可能なホスト |
| f | Function | Yes（pmap等） | 並列実行する関数 | 呼び出し可能オブジェクト |
| data | Any | Yes（pmap等） | 処理対象のデータ | シリアライズ可能 |

### 入力データソース

Julia のプログラムコードからの直接呼び出し。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Future | Future | リモート計算結果への参照 |
| result | Any | fetch()で取得した計算結果 |
| pids | Vector{Int} | ワーカープロセスIDリスト |
| RemoteChannel | RemoteChannel | リモートチャネルオブジェクト |

### 出力先

呼び出し元のJuliaプログラムへの戻り値として返却。

## 処理フロー

### 処理シーケンス

```
1. ワーカープロセスの起動（addprocs）
   └─ ClusterManager経由でプロセスを起動
2. タスクの分配
   └─ @distributed / pmap でタスクをワーカーに割り当て
3. データのシリアライズ・転送
   └─ Serialization.serialize でデータをバイト列化
4. ワーカーでの計算実行
   └─ remotecall_fetch / @spawnat で実行
5. 結果の収集
   └─ fetch / reduce で結果を集約
```

### フローチャート

```mermaid
flowchart TD
    A[addprocs でワーカー起動] --> B[タスク分配]
    B --> C[@distributed / pmap]
    C --> D[データシリアライズ]
    D --> E[ワーカーで計算実行]
    E --> F[結果シリアライズ]
    F --> G[結果収集 fetch]
    G --> H[結果の集約]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-124-01 | ワーカーID | マスタープロセスはID 1、ワーカーは2以降 | すべての分散計算 |
| BR-124-02 | データ転送 | ワーカーに送信するデータはシリアライズ可能であること | remotecall等 |
| BR-124-03 | エラー伝播 | ワーカーでのエラーはFutureのfetch時に再スロー | リモート実行エラー時 |
| BR-124-04 | プロセス独立性 | 各ワーカーは独立したメモリ空間を持つ | すべての分散計算 |

### 計算ロジック

pmap はデータのチャンクを各ワーカーに配分し、結果を順序を保持して返却する。@distributed for はイテレーション範囲を均等分割してワーカーに配分する。

## データベース操作仕様

### 操作別データベース影響一覧

データベースは使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ProcessExitedException | ワーカープロセスが異常終了 | ワーカーの状態確認・再起動 |
| - | RemoteException | ワーカーでの実行時エラー | エラー内容を確認し修正 |
| - | ArgumentError | 不正なワーカー指定 | 有効なワーカーIDを指定 |
| - | SSH接続エラー | リモートマシンへの接続失敗 | SSH設定・ネットワークを確認 |

### リトライ仕様

pmap にはretry_delays / retry_check オプションによるリトライ機構がある。

## トランザクション仕様

トランザクションは使用しない。分散計算の結果は個別に管理。

## パフォーマンス要件

ワーカー間のデータ転送がボトルネックとなり得る。大きなデータの転送を最小化し、計算量に対してデータ転送量が小さいタスクで効果的。

## セキュリティ考慮事項

- リモートワーカーへの接続にSSH認証が必要
- ワーカープロセスは起動元と同じ権限で実行される
- ネットワーク通信はデフォルトでは暗号化されない（SSH経由の場合は暗号化される）

## 備考

- Distributed は外部パッケージ（https://github.com/JuliaLang/Distributed.jl）として管理されている
- Julia本体には `stdlib/Distributed.version` ファイルでバージョン管理されている
- コミットSHA1: cd9219573d736b036077dff3cadddf369516d495
- `using Distributed` で読み込み可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: パッケージ参照設定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Distributed.version | `stdlib/Distributed.version` | ブランチ(master)・SHA1・Git URL(JuliaLang/Distributed.jl)の管理方法 |

**読解のコツ**: Distributed は Julia 1.x 以降外部パッケージ化されている。ただしGit URLは`.git`拡張子なしの形式で記述されている点に注意。

#### Step 2: ドキュメント統合を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | make.jl | `doc/make.jl` | Distributedは通常のSTDLIB_DOCSとして処理される。EXT_STDLIB_DOCSには含まれない |

#### Step 3: 関連ドキュメントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | distributed-computing.md | `doc/src/manual/distributed-computing.md` | Julia公式マニュアルの分散計算章。Distributedの使用法・概念を解説 |

### プログラム呼び出し階層図

```
using Distributed
    |
    +-- Base.require("Distributed")
    |
    +-- Distributed モジュール
            |
            +-- addprocs(n)
            |       +-- ClusterManager (LocalManager / SSHManager)
            |       +-- プロセス起動・接続
            |       +-- ワーカー初期化
            |
            +-- @distributed for
            |       +-- イテレーション範囲分割
            |       +-- 各ワーカーに送信
            |       +-- 結果収集・集約
            |
            +-- pmap(f, data)
            |       +-- データチャンク分割
            |       +-- ワーカーへの配分
            |       +-- 結果の順序保持収集
            |
            +-- RemoteChannel / Future
            |       +-- put! / take! / fetch
            |       +-- シリアライゼーション
            |
            +-- @spawnat / remotecall
                    +-- ワーカー指定実行
                    +-- 結果Future返却
```

### データフロー図

```
[入力]                   [処理]                        [出力]

タスク定義        ──> ClusterManager            ──> ワーカープロセス群
データ           ──> シリアライズ・転送           ──> ワーカーローカルデータ
ワーカー結果      ──> シリアライズ・返送          ──> マスター側Future
Future           ──> fetch()                   ──> 計算結果
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Distributed.version | `stdlib/Distributed.version` | 設定 | パッケージバージョン管理 |
| distributed-computing.md | `doc/src/manual/distributed-computing.md` | ドキュメント | 分散計算マニュアル |
| make.jl | `doc/make.jl` | ビルドスクリプト | ドキュメント生成設定 |
