# 機能設計書 125-Statistics

## 概要

本ドキュメントは、Julia標準ライブラリの Statistics パッケージに関する機能設計書である。Statistics は mean / median / std / var / cor / cov 等の基本的な統計関数を提供する外部パッケージとして Julia に同梱されている。

### 本機能の処理概要

Statistics は、データ分析・統計解析で必要となる基本的な記述統計量の計算関数を提供する。平均、中央値、標準偏差、分散、相関係数、共分散などの統計関数を、Juliaの多重ディスパッチシステムを活用して様々なデータ型に対して効率的に計算する。

**業務上の目的・背景**：統計解析はデータサイエンス・機械学習・品質管理・財務分析など幅広い分野で不可欠である。基本的な記述統計量の計算はほぼすべてのデータ分析ワークフローの最初のステップであり、正確かつ高速な実装が求められる。Statistics パッケージはこれらの基本関数を標準的なインタフェースで提供する。

**機能の利用シーン**：データの概要把握（平均・分散の計算）、外れ値の検出、変数間の関係性分析（相関係数）、データの正規化前処理、回帰分析の前段階など。

**主要な処理内容**：
1. 平均値計算（mean）
2. 中央値計算（median / median!）
3. 分散計算（var）
4. 標準偏差計算（std）
5. 相関係数計算（cor）
6. 共分散計算（cov）
7. 分位数計算（quantile）
8. 次元指定の統計量計算（dims引数）

**関連システム・外部連携**：LinearAlgebra（共分散行列・相関行列の計算）、特になし。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に直接関連する操作はない（ライブラリとして利用） |

## 機能種別

計算処理（統計計算ライブラリ）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| x | AbstractArray | Yes | 統計量を計算するデータ | 空でないこと |
| y | AbstractArray | 条件付き | 第2データ系列（cor/cov用） | xと同じ長さ |
| corrected | Bool | No | 不偏推定量を使用するか（デフォルト: true） | true/false |
| dims | Int | No | 集約する次元 | 有効な次元番号 |
| weights | AbstractWeights | No | 重み付き統計量の重みベクトル | xと同じ長さ |

### 入力データソース

Julia のプログラムコードからの直接呼び出し。数値配列データ。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| mean | Number | 算術平均 |
| median | Number | 中央値 |
| var | Number | 分散 |
| std | Number | 標準偏差 |
| cor | Number/Matrix | 相関係数（スカラまたは相関行列） |
| cov | Number/Matrix | 共分散（スカラまたは共分散行列） |

### 出力先

呼び出し元のJuliaプログラムへの戻り値として返却。

## 処理フロー

### 処理シーケンス

```
1. 入力データの検証
   └─ 空配列チェック、型チェック
2. 次元引数の解釈
   └─ dims指定がある場合は次元方向の集約
3. 統計量の計算
   └─ 適切なアルゴリズムの適用
4. 結果の返却
   └─ スカラ値または配列の返却
```

### フローチャート

```mermaid
flowchart TD
    A[入力データ x] --> B{dims指定?}
    B -->|No| C[全体の統計量計算]
    B -->|Yes| D[次元方向の集約計算]
    C --> E[スカラ値返却]
    D --> F[配列返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-125-01 | 不偏推定量 | var/stdはデフォルトでN-1除算（不偏分散） | corrected=true（デフォルト） |
| BR-125-02 | 標本分散 | corrected=falseでN除算（標本分散） | corrected=false指定時 |
| BR-125-03 | 空配列処理 | 空配列に対する統計量はエラー | mean/var等の引数が空配列の場合 |
| BR-125-04 | NaN伝播 | NaN値は計算結果に伝播する | 入力にNaN含む場合 |

### 計算ロジック

- mean(x) = sum(x) / length(x)
- var(x; corrected=true) = sum((x .- mean(x)).^2) / (length(x) - 1)
- std(x) = sqrt(var(x))
- cor(x, y) = cov(x, y) / (std(x) * std(y))
- cov(x, y) = sum((x .- mean(x)) .* (y .- mean(y))) / (length(x) - 1)

## データベース操作仕様

### 操作別データベース影響一覧

データベースは使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentError | 空配列への統計量計算 | 非空のデータを入力 |
| - | DimensionMismatch | cor/covの入力長さ不一致 | 入力長さを揃える |
| - | MethodError | 非数値型への統計量計算 | 数値型のデータを入力 |

### リトライ仕様

リトライは不要。

## トランザクション仕様

トランザクションは使用しない。

## パフォーマンス要件

mean/var/std等の基本統計量はO(n)の時間計算量。median はO(n log n)（ソートベース）。数値安定性のためにWelfordのアルゴリズム等が使用される場合がある。

## セキュリティ考慮事項

特筆すべきセキュリティ要件はない。

## 備考

- Statistics は外部パッケージ（https://github.com/JuliaStats/Statistics.jl）として管理されている
- Julia本体には `stdlib/Statistics.version` ファイルでバージョン管理されている
- コミットSHA1: 22dee82f9824d6045e87aa4b97e1d64fe6f01d8d
- JuliaLangではなくJuliaStats組織のリポジトリで管理されている点に注意

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: パッケージ参照設定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Statistics.version | `stdlib/Statistics.version` | ブランチ(master)・SHA1・Git URL(JuliaStats/Statistics.jl)の管理方法 |

**読解のコツ**: Statistics はJuliaStats組織のリポジトリで管理されている。JuliaLang組織ではない点に注意。

#### Step 2: Baseの統計関数との関係を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | reduce.jl | `base/reduce.jl` | sumやprodなどの集約関数がBaseに定義されている。Statisticsはこれらの上に構築される |

**主要処理フロー**:
- Base.sum がリダクション演算の基盤
- Statistics.mean は sum / length に基づく
- Statistics.var は mean を使用して計算

### プログラム呼び出し階層図

```
using Statistics
    |
    +-- Base.require("Statistics")
    |
    +-- Statistics モジュール
            |
            +-- mean(x)
            |       +-- Base.sum(x) / Base.length(x)
            |
            +-- var(x; corrected=true)
            |       +-- mean(x)
            |       +-- sum of squared deviations
            |
            +-- std(x)
            |       +-- sqrt(var(x))
            |
            +-- median(x)
            |       +-- sort / partialsort
            |
            +-- cor(x, y)
            |       +-- cov(x, y) / (std(x) * std(y))
            |
            +-- cov(x, y)
                    +-- mean-centered products
```

### データフロー図

```
[入力]                   [処理]                        [出力]

AbstractArray    ──> 入力検証                  ──> 検証済みデータ
検証済みデータ    ──> 統計量計算               ──> スカラ値/配列
dims指定         ──> 次元方向集約              ──> 低次元配列
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Statistics.version | `stdlib/Statistics.version` | 設定 | パッケージバージョン管理 |
| reduce.jl | `base/reduce.jl` | ソース | sum等の基盤リダクション演算 |
| make.jl | `doc/make.jl` | ビルドスクリプト | ドキュメント生成設定 |
