# 機能設計書 128-Future

## 概要

本ドキュメントは、Julia標準ライブラリの Future モジュールに関する機能設計書である。Future は将来のJuliaバージョンとの互換性ユーティリティを提供するモジュールであり、`stdlib/Future/` に定義されている。

### 本機能の処理概要

Future モジュールは、既存の関数の将来バージョンでの振る舞い変更を先取りして提供するモジュールである。将来のJuliaリリースで現在のバージョンを置き換える予定の関数実装を、現在のバージョンで利用可能にする。これにより段階的な移行パスを提供する。

**業務上の目的・背景**：プログラミング言語のバージョンアップに伴い、既存関数の振る舞いが変更されることがある。Future モジュールは、こうした変更を事前に利用可能にすることで、ユーザーコードの段階的な移行を支援する。破壊的変更に対するソフトランディング機構として機能する。

**機能の利用シーン**：将来のJuliaバージョンでの動作を先取りして使用したい場合、互換性テストの実施、新しいAPIへの段階的移行など。

**主要な処理内容**：
1. `Future.copy!` - Base.copy! の将来バージョン互換ラッパー（Julia 1.1 でBaseに移動済み）
2. `Future.randjump` - MersenneTwister の状態をステップ数分前進させる関数

**関連システム・外部連携**：Random パッケージ（MersenneTwister / DSFMT）。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に直接関連する操作はない（ユーティリティライブラリ） |

## 機能種別

ユーティリティ（互換性ヘルパー）

## 入力仕様

### 入力パラメータ

#### Future.copy!

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| dst | AbstractSet/AbstractDict/AbstractArray | Yes | コピー先 | 型が適合すること |
| src | AbstractSet/AbstractDict/AbstractArray | Yes | コピー元 | dstと同じ基底型 |

#### Future.randjump

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| r | MersenneTwister | Yes | 状態をジャンプさせるRNG | MersenneTwister型 |
| steps | Integer | Yes | ジャンプするステップ数 | 整数値 |

### 入力データソース

Julia のプログラムコードからの直接呼び出し。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| dst | copy!の場合: コピー先オブジェクト | 値がコピーされたdst |
| rng | randjumpの場合: MersenneTwister | ステップ数分前進した新しいRNG状態 |

### 出力先

呼び出し元のJuliaプログラムへの戻り値として返却。

## 処理フロー

### 処理シーケンス

#### Future.copy!
```
1. 引数の型に基づきBase.copy!にディスパッチ
   └─ AbstractSet, AbstractDict, AbstractArray の3パターン
2. Base.copy!(dst, src) の呼び出し
   └─ 実際のコピー処理を委譲
3. dst の返却
```

#### Future.randjump
```
1. MersenneTwisterの状態を受け取り
   └─ r パラメータ
2. DSFMT.calc_jump(steps) でジャンプ多項式を計算
   └─ 大きなステップ数の場合の内部多項式計算
3. Random._randjump(r, jump_poly) で状態をジャンプ
   └─ 新しいMersenneTwister状態を構築
4. adv_jump フィールドを更新
   └─ BigInt変換でオーバーフロー防止
5. 新しいMersenneTwisterを返却
```

### フローチャート

```mermaid
flowchart TD
    A[Future.randjump 呼び出し] --> B[DSFMT.calc_jump でジャンプ多項式計算]
    B --> C[Random._randjump で状態ジャンプ]
    C --> D[adv_jump フィールド更新]
    D --> E[新しいMersenneTwister返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-128-01 | Base委譲 | Future.copy! は Base.copy! に処理を委譲 | copy!呼び出し時 |
| BR-128-02 | BigInt変換 | randjump のステップ数はBigIntに変換してオーバーフロー防止 | randjump呼び出し時 |
| BR-128-03 | 非破壊的 | randjump は元のRNGを変更せず新しいRNGを返す | randjump呼び出し時 |
| BR-128-04 | 事前計算 | steps=big(10)^20 のジャンプ多項式は事前計算済み | 該当ステップ数指定時 |

### 計算ロジック

randjump: 1ステップは2つのFloat64乱数の生成に相当する。DSFMTの特性多項式に基づくジャンプ多項式を計算し、RNG状態を効率的に前進させる。

## データベース操作仕様

### 操作別データベース影響一覧

データベースは使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | MethodError | 未対応の型に対するcopy! | 対応する型を使用 |
| - | ArgumentError | 不正なステップ数 | 有効な整数値を指定 |

### リトライ仕様

リトライは不要。

## トランザクション仕様

トランザクションは使用しない。

## パフォーマンス要件

Future.copy! は Base.copy! と同等のパフォーマンス（委譲のみ）。randjump のジャンプ多項式計算は大きなステップ数で時間がかかる場合がある（事前計算済みの場合は高速）。

## セキュリティ考慮事項

特筆すべきセキュリティ要件はない。

## 備考

- Future モジュールは `stdlib/Future/` に定義されている内部パッケージ
- UUID: 9fa8497b-333b-5362-9e8d-4d0656e87820
- バージョン: 1.11.0
- Random パッケージに依存
- Future.copy! は Julia 1.1 以降は非推奨（Base.copy! を直接使用推奨）
- テストは closure boxes チェックと docstring チェックのみ（最小限）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Future モジュールは非常に小さなモジュールであり、独自のデータ構造は定義していない。依存するRandomパッケージのMersenneTwister型を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Project.toml | `stdlib/Future/Project.toml` | 依存関係（Random）とUUID、バージョンの確認 |

**読解のコツ**: Project.toml の `[deps]` セクションでRandomへの依存を確認。`[extras]` と `[targets]` はテスト用依存。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Future.jl | `stdlib/Future/src/Future.jl` | モジュール全体の定義（45行の小規模モジュール） |

**主要処理フロー**:
1. **7行目**: `using Random` でRandomパッケージの読み込み
2. **23-25行目**: `copy!` の3つのメソッド定義（AbstractSet, AbstractDict, AbstractArray）。すべてBase.copy!に委譲
3. **39-43行目**: `randjump` 関数の定義。`Random._randjump` と `DSFMT.calc_jump` を使用
4. **41行目**: `j.adv_jump += 2*big(steps)` でBigInt変換によるオーバーフロー防止

#### Step 3: テストを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | runtests.jl | `stdlib/Future/test/runtests.jl` | **6行目**: closure boxes検出テスト。**8-10行目**: docstring網羅性テスト |

**主要処理フロー**:
- **6行目**: `isempty(Test.detect_closure_boxes(Future))` でクロージャボックス検出
- **9行目**: `isempty(Docs.undocumented_names(Future))` で全公開名がドキュメント化されていることを検証

### プログラム呼び出し階層図

```
Future モジュール
    |
    +-- Future.copy!(dst, src)
    |       +-- Base.copy!(dst::AbstractSet, src::AbstractSet)
    |       +-- Base.copy!(dst::AbstractDict, src::AbstractDict)
    |       +-- Base.copy!(dst::AbstractArray, src::AbstractArray)
    |
    +-- Future.randjump(r, steps)
            +-- Random.DSFMT.calc_jump(steps)
            |       +-- ジャンプ多項式の計算
            +-- Random._randjump(r, jump_poly)
            |       +-- MersenneTwister状態のジャンプ
            +-- j.adv_jump += 2*big(steps)
                    +-- BigInt変換でオーバーフロー防止
```

### データフロー図

```
[入力]                   [処理]                        [出力]

MersenneTwister  ──> DSFMT.calc_jump          ──> ジャンプ多項式
ジャンプ多項式    ──> Random._randjump         ──> 新MersenneTwister
steps (Integer)  ──> big(steps) 変換           ──> adv_jump更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Future.jl | `stdlib/Future/src/Future.jl` | ソース | モジュール本体（45行） |
| Project.toml | `stdlib/Future/Project.toml` | 設定 | パッケージ設定・依存関係 |
| runtests.jl | `stdlib/Future/test/runtests.jl` | テスト | closure boxes・docstringテスト |
