# 機能設計書 149-OpenSSL_jll

## 概要

本ドキュメントは、Julia標準ライブラリに含まれるOpenSSL_jllパッケージの機能設計を記述する。OpenSSL_jllは、OpenSSLライブラリのJLLバイナリラッパーであり、SSL/TLS暗号化通信と暗号化機能をJuliaから利用可能にする。

### 本機能の処理概要

OpenSSL_jllは、OpenSSL Cライブラリ（libcrypto, libssl）のプリビルドバイナリをJuliaのパッケージシステムを通じて提供するJLLラッパーパッケージである。2つのライブラリ（暗号化基盤のlibcryptoとSSL/TLSプロトコルのlibssl）を提供し、プラットフォーム別のライブラリ名マッピングを行う。Windows環境ではx86_64固有のDLL命名規則に対応する。

**業務上の目的・背景**：OpenSSLはSSL/TLSプロトコルとさまざまな暗号化アルゴリズムを実装するライブラリであり、Juliaのネットワーク通信セキュリティの基盤として使用される。非Windows環境ではLibCURL_jll、LibSSH2_jll、LibGit2_jll等の暗号化基盤として不可欠である。HTTPS通信、SSH認証、パッケージレジストリの安全な取得などに利用される。

**機能の利用シーン**：HTTPS通信によるパッケージダウンロード、SSH経由のGitリポジトリアクセス、SSL/TLS対応のネットワーク通信等で利用される。非Windows環境でJuliaの通信セキュリティの根幹を担う。

**主要な処理内容**：
1. プラットフォーム（Windows/macOS/Linux/FreeBSD）に応じたlibcrypto/libssl共有ライブラリパスの解決
2. Windows x86_64環境でのアーキテクチャ固有DLL名の判定
3. libsslのlibcryptoへの依存関係の定義
4. `LazyLibrary`メカニズムを用いた遅延ライブラリロード

**関連システム・外部連携**：外部JLL依存なし（libsslはlibcryptoに依存）。LibCURL_jll、LibSSH2_jll、LibGit2_jllから依存される側のライブラリ。BinaryPlatformsを使用してホストアーキテクチャを判定。上流のOpenSSLプロジェクト（https://www.openssl.org/）のバージョン3.5.5に対応。

**権限による制御**：特になし。全ユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はUI画面を持たない。他のJLLパッケージの暗号化基盤として使用される |

## 機能種別

データ連携（ネイティブライブラリバイナリ配布・動的リンク管理・暗号化通信基盤）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Sys.KERNEL | Symbol | Yes | 実行プラットフォームのカーネル識別子 | Windows/macOS/Linux/FreeBSDのいずれか |
| HostPlatform() | Platform | Yes | ホストプラットフォーム情報（Windows x86_64判定用） | 有効なプラットフォーム |

### 入力データソース

- プラットフォーム情報: `Sys.iswindows()`, `Sys.isapple()`, `Sys.islinux()`, `Sys.isfreebsd()`
- アーキテクチャ情報: `Base.BinaryPlatforms.arch(HostPlatform())` （Windows環境でのDLL名判定）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| libcrypto | LazyLibrary | libcrypto共有ライブラリへのハンドル（遅延ロード） |
| libssl | LazyLibrary | libssl共有ライブラリへのハンドル（遅延ロード） |
| libcrypto_path | String | libcryptoの絶対ファイルパス |
| libssl_path | String | libsslの絶対ファイルパス |
| artifact_dir | String | アーティファクトディレクトリ |

### 出力先

- メモリ内グローバル変数としてモジュール内に保持

## 処理フロー

### 処理シーケンス

```
1. モジュール定義（baremodule）
   └─ Base, Libdl, Base.BinaryPlatforms をusing
2. libcrypto LazyLibrary定義
   └─ Windows x86_64: "libcrypto-3-x64.dll"
   └─ Windows 他: "libcrypto-3.dll"
   └─ macOS: "libcrypto.3.dylib"
   └─ Linux/FreeBSD: "libcrypto.so.3"
3. libssl LazyLibrary定義（libcryptoに依存）
   └─ Windows x86_64: "libssl-3-x64.dll"
   └─ Windows 他: "libssl-3.dll"
   └─ macOS: "libssl.3.dylib"
   └─ Linux/FreeBSD: "libssl.so.3"
4. __init__()実行
   └─ libcrypto_path, libssl_path, artifact_dir, LIBPATH の設定
```

### フローチャート

```mermaid
flowchart TD
    A[モジュールロード] --> B[libcrypto定義]
    B --> C{Windowsプラットフォーム?}
    C -->|Yes| D{x86_64?}
    D -->|Yes| E["libcrypto-3-x64.dll"]
    D -->|No| F["libcrypto-3.dll"]
    C -->|macOS| G["libcrypto.3.dylib"]
    C -->|Linux/FreeBSD| H["libcrypto.so.3"]
    E --> I[libssl定義]
    F --> I
    G --> I
    H --> I
    I --> J["libssl deps: libcrypto"]
    J --> K["__init__() 実行"]
    K --> L[libcrypto_path, libssl_path 設定]
    L --> M[初期化完了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-149-01 | 2ライブラリ構成 | libcrypto（暗号化基盤）とlibssl（SSL/TLSプロトコル）の2つを提供 | 全環境 |
| BR-149-02 | 内部依存 | libsslはlibcryptoに依存する | 全環境 |
| BR-149-03 | Windows x86_64命名 | x86_64ではDLL名にサフィックス"-x64"を付与 | Windows x86_64 |
| BR-149-04 | OpenSSL 3.x | メジャーバージョン3系のOpenSSLを使用 | 全環境 |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ErrorException | 未対応プラットフォーム | `error("OpenSSL_jll: Library 'libcrypto/libssl' is not available for $(Sys.KERNEL)")` |
| - | DLError | ライブラリファイルが見つからない | dlopen失敗時のOSエラーメッセージ表示 |

### リトライ仕様

リトライは行わない。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- LazyLibraryメカニズムにより遅延ロード
- プリコンパイル対応（eager_mode, is_available）

## セキュリティ考慮事項

- OpenSSL 3.5.5は最新のセキュリティ修正を含むバージョン
- SSL/TLS暗号化通信の基盤としてJulia全体のネットワークセキュリティに直結
- OpenSSLの脆弱性発見時は迅速なバージョン更新が必要

## 備考

- バージョン: OpenSSL 3.5.5+0
- UUID: 458c3c95-2e84-50aa-8efc-19380b2a3a95
- JuliaBinaryWrappersの上流リポジトリ: https://github.com/JuliaBinaryWrappers/OpenSSL_jll.jl
- BinaryPlatformsモジュールを使用してWindows x86_64のアーキテクチャ判定を行う特殊なJLLパッケージ
- Juliaの依存ツリーにおいて、非Windows環境での暗号化通信の根幹を担う

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | libdl.jl | `base/libdl.jl` | LazyLibrary構造体のdependenciesフィールド。ライブラリ間依存（libssl -> libcrypto）の定義方法 |
| 1-2 | binaryplatforms.jl | `base/binaryplatforms.jl` | HostPlatform()とarch()関数。プラットフォーム検出の仕組み |

**読解のコツ**: OpenSSL_jllの特殊な点は、(1) 2つのライブラリを提供すること、(2) BinaryPlatformsモジュールを使ってWindows x86_64固有のDLL名を決定すること。他のJLLにはこのパターンは少ない。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OpenSSL_jll.jl | `stdlib/OpenSSL_jll/src/OpenSSL_jll.jl` | モジュール全体の定義（71行） |

**主要処理フロー**:
1. **5行目**: `baremodule OpenSSL_jll` - モジュール定義開始
2. **6行目**: `using Base, Libdl, Base.BinaryPlatforms` - BinaryPlatformsの追加使用
3. **8行目**: `export libcrypto, libssl` - 2つのライブラリシンボルをエクスポート
4. **18-32行目**: `libcrypto` LazyLibrary定数。Windows x86_64で"-x64"サフィックス付きDLL名
5. **20-24行目**: Windowsでarch(HostPlatform())による条件分岐
6. **35-50行目**: `libssl` LazyLibrary定数。libcryptoへの依存を指定（49行目）
7. **52-55行目**: eager_mode()。libcrypto -> libsslの順でdlopen
8. **58-64行目**: __init__()。両ライブラリのパスとartifact_dirを設定

### プログラム呼び出し階層図

```
OpenSSL_jll.__init__()
    |
    +-- libcrypto.path (LazyLibrary.path アクセス)
    |       |
    |       +-- BundledLazyLibraryPath("libcrypto.so.3")
    |
    +-- libssl.path (LazyLibrary.path アクセス)
    |       |
    |       +-- BundledLazyLibraryPath("libssl.so.3")
    |
    +-- artifact_dir = dirname(Sys.BINDIR)
    |
    +-- LIBPATH[] = dirname(libssl_path)

OpenSSL_jll.eager_mode()
    |
    +-- dlopen(libcrypto) [先にロード]
    |
    +-- dlopen(libssl) [libcryptoに依存]

被依存関係:
    LibCURL_jll ----> OpenSSL_jll (libssl, libcrypto) [非Windows]
    LibSSH2_jll ----> OpenSSL_jll (libcrypto) [非Windows]
    LibGit2_jll ----> OpenSSL_jll (libssl, libcrypto) [Linux/FreeBSD]
```

### データフロー図

```
[入力]                              [処理]                              [出力]

Sys.KERNEL (OS種別) -------------> プラットフォーム判定 ----------> ライブラリファイル名
HostPlatform() (アーキテクチャ) -> Windows x86_64判定 -----------> DLL名サフィックス
                                      |
                                      v
                               dlopen(libcrypto) --------------> 暗号化基盤
                               dlopen(libssl) -----------------> SSL/TLS通信機能
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OpenSSL_jll.jl | `stdlib/OpenSSL_jll/src/OpenSSL_jll.jl` | ソース | メインモジュール定義。libcrypto/libsslのLazyLibrary定義 |
| Project.toml | `stdlib/OpenSSL_jll/Project.toml` | 設定 | パッケージメタデータ・バージョン情報 |
| binaryplatforms.jl | `base/binaryplatforms.jl` | ソース | HostPlatform()によるアーキテクチャ検出 |
| libdl.jl | `base/libdl.jl` | ソース | LazyLibrary基盤実装 |
