# 機能設計書 15-Array（多次元配列）

## 概要

本ドキュメントは、Julia言語におけるArray（多次元配列）機能の設計を記述する。N次元配列の生成・アクセス・変形・コピーなどの基本操作を提供する。

### 本機能の処理概要

Array{T,N}はJuliaの中核的なデータ構造であり、N次元の密な配列を提供する。C言語レベルでのメモリ管理と連携し、連続メモリ上に要素を格納する高性能な配列実装を実現している。AbstractArray{T,N}を頂点とする型階層により、多態的な配列操作を可能にしている。

**業務上の目的・背景**：数値計算・データ処理の基盤として、高速なメモリアクセスとSIMD最適化を可能にする連続メモリ配列は不可欠である。JuliaのArrayはPython/NumPyやMATLABの配列と同等の使いやすさを持ちつつ、C/Fortran並みの性能を実現する。型パラメータT,Nにより、コンパイル時に要素型と次元数が確定し、型安定な操作が可能。

**機能の利用シーン**：（1）科学技術計算での行列・ベクトル演算、（2）データ分析でのテーブルデータ格納、（3）画像処理での多次元データ操作、（4）機械学習でのテンソル操作、（5）一般的なプログラミングでのリスト操作。

**主要な処理内容**：
1. Array{T,N}型定義とVector{T}/Matrix{T}型エイリアス
2. 配列生成（Array(undef, dims), zeros, ones, fill, similar）
3. インデックスアクセス（getindex, setindex!, checkbounds）
4. 配列変形（reshape, vec）
5. 配列連結（vcat, hcat, cat, hvcat, hvncat）
6. 配列コピー（copy, copyto!）
7. AbstractArray抽象インタフェース（size, axes, length, ndims, eltype, IndexStyle）
8. push!/pop!/append!/prepend!等の動的サイズ変更操作

**関連システム・外部連携**：Cランタイム（src/array.c）でのメモリ割り当て。GenericMemory（base/genericmemory.jl）を内部バッファとして使用。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | Juliaプロンプト（julia>） | 参照画面 | REPL上で配列操作を対話的に実行 |
| 8 | RadioMenu | 補助機能 | options::Array{String,1}による選択肢管理 |
| 9 | MultiSelectMenu | 補助機能 | options::Array{String,1}による選択肢管理 |

## 機能種別

データ管理（多次元配列データ構造）/ CRUD操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| T | Type | Yes | 要素型 | 具体型であること |
| dims | NTuple{N,Int} | Yes | 各次元のサイズ | 各次元が0以上 |
| init | UndefInitializer/値 | No | 初期化方法 | - |

### 入力データソース

Julia式からの直接入力。リテラル構文 [1,2,3] や内包表記 [f(x) for x in iter]。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| array | Array{T,N} | 生成・操作された配列 |
| element | T | getindexによる要素取得結果 |

### 出力先

メモリ上のArrayオブジェクトとして返却。

## 処理フロー

### 処理シーケンス

```
1. 配列生成
   └─ Memory{T}バッファ確保 → Arrayオブジェクト生成
2. インデックスアクセス
   └─ checkboundsで境界チェック → getindex/setindex!
3. 配列変形
   └─ 新しいサイズのArrayを生成、同じバッファを参照（reshape）
4. 配列連結
   └─ 結果サイズ計算 → 新Array確保 → copyto!でデータコピー
```

### フローチャート

```mermaid
flowchart TD
    A[配列操作要求] --> B{操作種別}
    B -->|生成| C[Memory割り当て → Array構築]
    B -->|アクセス| D[checkbounds → getindex/setindex!]
    B -->|変形| E[reshape: 同一バッファで新Array]
    B -->|連結| F[サイズ計算 → 新Array → copyto!]
    B -->|動的変更| G[push!/pop!: ccall jl_array_grow_end等]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 1-based インデックス | デフォルトは1始まりのインデックス | 全Array操作 |
| BR-02 | column-major順序 | 列優先（Fortran順）のメモリレイアウト | 多次元配列 |
| BR-03 | 境界チェック | @boundscheck/@inboundsで制御可能 | getindex/setindex! |
| BR-04 | 型安定性 | eltype(A)が実行時に一定 | 全Array |
| BR-05 | DimensionMismatch | サイズ不一致の操作でエラー | 配列演算時 |

### 計算ロジック

- IndexLinearスタイル: 線形インデックスによる高速アクセス
- IndexCartesianスタイル: 多次元インデックスによるアクセス
- 配列生成: Array{T,N}(undef, dims...) は内部でGenericMemory{:not_atomic, T}を確保

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | BoundsError | インデックスが範囲外 | 有効なインデックスを使用 |
| - | DimensionMismatch | 配列サイズ不一致の操作 | サイズを合わせる |
| - | ArgumentError | 不正な次元サイズ（負の値等） | 非負のサイズを指定 |
| - | InexactError | 型変換不可能な要素の代入 | 互換性のある型を使用 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 連続メモリ配置によるキャッシュ効率的なアクセス
- SIMD最適化が可能なメモリレイアウト
- @inboundsによる境界チェック省略で内部ループ高速化
- column-major順序によるFortran互換のBLAS/LAPACK呼び出し

## セキュリティ考慮事項

- @inboundsの誤用によるメモリ安全性の喪失に注意
- Array(undef, ...)で初期化されていないメモリの読み取りに注意

## 備考

- Julia 1.11以降、ArrayはGenericMemory{:not_atomic, T}を内部バッファとして使用
- DenseArrayはメモリ連続性を保証する抽象型
- StridedArrayはstride情報を持つ配列の型エイリアス

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

配列型の型階層と基本型定義を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | abstractarray.jl | `base/abstractarray.jl` | AbstractArray{T,N}の定義と基本インタフェース（1-100行目） |
| 1-2 | array.jl | `base/array.jl` | Array{T,N}, Vector{T}, Matrix{T}の定義（1-91行目） |

**読解のコツ**: AbstractArray{T,N}はJuliaの配列型階層の頂点。size, axes, length, ndims, eltype等のインタフェース関数が定義されている。Array{T,N}はCoreで定義されたビルトイン型で、array.jlでは型エイリアスとドキュメントが提供される。

#### Step 2: 配列生成と基本操作を理解する

コンストラクタとインデックスアクセス。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | array.jl | `base/array.jl` | Array生成関数（zeros, ones, fill, similar等） |
| 2-2 | abstractarray.jl | `base/abstractarray.jl` | size, axes, length, ndims等の基本関数（20-100行目） |

#### Step 3: インデックスアクセスを理解する

getindex, setindex!, checkboundsの実装。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | abstractarray.jl | `base/abstractarray.jl` | checkboundsとインデックス変換 |
| 3-2 | array.jl | `base/array.jl` | Array向けgetindex/setindex!の実装 |

### プログラム呼び出し階層図

```
Array{T,N}(undef, dims...)
    |
    +-- GenericMemory{:not_atomic, T}(undef, prod(dims))  [バッファ確保]
    |
    +-- getindex(A, i...)
    |       +-- checkbounds(A, i...)
    |       +-- _getindex(IndexStyle(A), A, i...)
    |               +-- @inbounds memoryref(A.ref, idx)
    |
    +-- setindex!(A, v, i...)
    |       +-- checkbounds(A, i...)
    |       +-- _setindex!(IndexStyle(A), A, v, i...)
    |
    +-- push!(A, item)
    |       +-- ccall(:jl_array_grow_end)
    |       +-- setindex!(A, item, length(A))
    |
    +-- reshape(A, dims...)
            +-- 新Array（同一バッファ参照）
```

### データフロー図

```
[入力]                    [処理]                         [出力]

dims, T          ──────> Memory確保 → Array構築  ──────> Array{T,N}
Array, indices   ──────> checkbounds → getindex   ──────> 要素 T
Array, val, idx  ──────> checkbounds → setindex!  ──────> Array（変更済み）
Array, dims      ──────> reshape（バッファ共有）   ──────> Array{T,M}
Arrays...        ──────> cat/vcat/hcat → copyto!  ──────> Array{T,N}
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| array.jl | `base/array.jl` | ソース | Array型定義、基本操作、push!/pop!等 |
| abstractarray.jl | `base/abstractarray.jl` | ソース | AbstractArray抽象インタフェース |
| genericmemory.jl | `base/genericmemory.jl` | ソース | Memory{T}バッファ管理 |
| multidimensional.jl | `base/multidimensional.jl` | ソース | 多次元インデックス操作 |
| reshapedarray.jl | `base/reshapedarray.jl` | ソース | reshape操作 |
| indices.jl | `base/indices.jl` | ソース | インデックス型定義 |
| src/array.c | `src/array.c` | Cソース | 配列メモリ管理のCランタイム実装 |
