# 機能設計書 2-複合型（構造体）定義

## 概要

本ドキュメントは、Julia言語における `struct` / `mutable struct` によるユーザー定義複合型（構造体）の宣言と生成に関する機能設計書である。

### 本機能の処理概要

Julia言語のユーザー定義型システムの中核をなす複合型（構造体）の定義機能を提供する。`struct` キーワードによる不変構造体、`mutable struct` キーワードによる可変構造体の宣言・生成・フィールドアクセスを実現する。

**業務上の目的・背景**：プリミティブ型だけではビジネスドメインのデータを表現できないため、複数のフィールドを持つユーザー定義型が必要である。Juliaの複合型はコンパイラによる型推論・最適化の対象となり、C構造体並みのメモリ効率と速度を実現する。不変構造体（struct）はスレッドセーフかつメモリ効率が高く、値セマンティクスを持つデータの表現に適している。

**機能の利用シーン**：データ構造の定義（座標、色、日時、設定情報等）、型安全なドメインモデリング、多重ディスパッチの型引数としての利用、パッケージAPIにおけるデータ型の公開等で使用される。

**主要な処理内容**：
1. `struct` 宣言のパース・型生成（不変構造体）
2. `mutable struct` 宣言のパース・型生成（可変構造体）
3. コンストラクタ（内部・外部）の自動生成
4. `getfield` / `setfield!` によるフィールドアクセス
5. 型パラメータ付き構造体のサポート

**関連システム・外部連携**：Cランタイム（`src/datatype.c`）での DataType オブジェクト生成、コンパイラによるフィールドレイアウト最適化と連携する。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 複合型は画面に直接関連しないが、REPLでのshow/display表示に使用される |

## 機能種別

型定義・データ構造管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| type_name | Symbol | Yes | 構造体名 | 有効な識別子であること |
| fields | Vector{Pair{Symbol,Type}} | Yes | フィールド名と型のペア | 型はすべて具体型または抽象型であること |
| supertype | DataType | No | 親抽象型（デフォルト: Any） | abstracttypeであること |
| mutable | Bool | Yes | 可変性フラグ | true=mutable struct, false=struct |
| type_params | Vector{TypeVar} | No | 型パラメータ | 有効なTypeVarであること |

### 入力データソース

Juliaソースコードの `struct` / `mutable struct` 宣言文から、パーサー（JuliaSyntax.jl）を通じて取得される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| DataType | Core.DataType | 構造体のDataTypeオブジェクト |
| コンストラクタ | Function | デフォルトコンストラクタおよびユーザー定義コンストラクタ |
| フィールド情報 | Tuple{Symbol...} | フィールド名のタプル |

### 出力先

定義されたモジュールのスコープに型オブジェクトとしてバインドされる。

## 処理フロー

### 処理シーケンス

```
1. struct/mutable struct 宣言のパース
   └─ JuliaSyntaxパーサーが宣言をExprに変換
2. 型パラメータの解決
   └─ TypeVar の生成と UnionAll ラッピング
3. フィールド型の解決
   └─ 各フィールドの型を評価・バリデーション
4. DataType オブジェクトの生成
   └─ jl_new_datatype() (Cランタイム) でメモリ割り当て
5. フィールドレイアウトの計算
   └─ jl_compute_field_offsets() でアライメント・パディング計算
6. デフォルトコンストラクタの生成
   └─ new() キーワードを使用する内部コンストラクタの生成
7. 型のバインド
   └─ モジュールスコープへの登録
```

### フローチャート

```mermaid
flowchart TD
    A[struct宣言パース] --> B[型パラメータ解決]
    B --> C[フィールド型解決]
    C --> D{mutable?}
    D -->|Yes| E[可変DataType生成]
    D -->|No| F[不変DataType生成]
    E --> G[フィールドレイアウト計算]
    F --> G
    G --> H[デフォルトコンストラクタ生成]
    H --> I[モジュールスコープにバインド]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 不変性制約 | structのフィールドはsetfield!で変更不可 | struct（非mutable）の場合 |
| BR-02 | 単一継承制約 | 構造体は1つの抽象型のみ継承可能 | 常時 |
| BR-03 | 具体型非継承 | 具体的な構造体を親型として指定することは不可 | 常時 |
| BR-04 | new制約 | new()は内部コンストラクタ内でのみ使用可能 | コンストラクタ定義時 |
| BR-05 | const修飾 | mutable structのフィールドにconstを付けると変更不可 | mutable struct定義時 |

### 計算ロジック

フィールドレイアウトの計算はCランタイムで行われ、各フィールドのサイズとアライメントに基づいてオフセットとパディングが決定される。

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（データベースを使用しない）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TypeError | TypeError | 不正な親型指定（具体型を継承しようとした場合） | 抽象型を指定する |
| ErrorException | ErrorException | 型の再定義 | 新しいセッションで再定義する |
| FieldError | FieldError | 存在しないフィールドへのアクセス | 正しいフィールド名を使用する |

### リトライ仕様

型定義は一回限りの処理であり、リトライは適用されない。

## トランザクション仕様

該当なし。型定義はアトミックな操作として実行される。

## パフォーマンス要件

- 不変構造体（struct）はスタック割り当てまたはインライン化が可能で、ヒープ割り当てを回避できる
- フィールドアクセスはgetfield組み込み関数により O(1) で実行される
- 型パラメータ付き構造体は特殊化によりボックス化を回避できる

## セキュリティ考慮事項

- mutable structのフィールドは複数スレッドからの同時アクセスで競合が発生する可能性がある
- `@atomic` フィールド修飾子を使用してアトミックアクセスを確保できる

## 備考

- boot.jlのコメント（18-30行目）にDataType構造体の内部フィールドがリファレンスとして記載されている
- Task, Module, String 等のCore型もmutable structとしてCランタイムで定義されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

DataType自体がどのような構造を持つかを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | boot.jl | `base/boot.jl` | **18-30行目**: DataType構造体のフィールド定義（コメント形式）。name, super, parameters, names, types, size, abstract, mutableフィールドを持つ |
| 1-2 | boot.jl | `base/boot.jl` | **231-232行目**: getfield/setfield!/swapfield!/modifyfield!/replacefield!/setfieldonce! のexport |

**読解のコツ**: DataTypeはJuliaの型システムのメタ型であり、すべての具体型・抽象型はDataTypeのインスタンスとして表現される。boot.jlのコメントアウトされた定義はCランタイムの実装を示すリファレンスである。

#### Step 2: エントリーポイントを理解する

struct宣言がどのように処理されるかを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | boot.jl | `base/boot.jl` | **245-246行目**: getproperty/setproperty! がgetfield/setfield!にエイリアスされている |
| 2-2 | boot.jl | `base/boot.jl` | **496-499行目**: Box型の定義例（mutable structの実例） |

**主要処理フロー**:
1. パーサーがstruct宣言をExpr(:struct, ...)に変換
2. Cランタイムのjl_new_datatype()がDataTypeオブジェクトを生成
3. フィールドレイアウトが計算される
4. デフォルトコンストラクタ（new()使用）が自動生成される

#### Step 3: 型関連ユーティリティを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | essentials.jl | `base/essentials.jl` | 型関連のユーティリティ関数群 |
| 3-2 | reflection.jl | `base/reflection.jl` | fieldnames, fieldtypes等のリフレクション関数 |

### プログラム呼び出し階層図

```
パーサー (JuliaSyntax.jl)
    │
    ├─ struct宣言 → Expr(:struct, mutable, name, body)
    │
    └─ Cランタイム
           ├─ jl_new_datatype()
           │      ├─ フィールド型の検証
           │      └─ jl_compute_field_offsets()
           │             └─ アライメント・パディング計算
           │
           └─ コンストラクタ生成
                  ├─ 内部コンストラクタ（new()使用）
                  └─ 外部コンストラクタ（型変換等）
```

### データフロー図

```
[入力]                      [処理]                          [出力]

struct宣言          ───▶ パーサー                    ───▶ Expr(:struct, ...)
  (ソースコード)           │
                          ▼
Expr                ───▶ jl_new_datatype()          ───▶ DataTypeオブジェクト
                          │
                          ▼
フィールド型情報    ───▶ jl_compute_field_offsets()  ───▶ メモリレイアウト
                          │
                          ▼
型定義              ───▶ コンストラクタ生成          ───▶ new()関数
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| boot.jl | `base/boot.jl` | ソース | DataType構造体の定義リファレンス、getfield/setfield!のexport |
| essentials.jl | `base/essentials.jl` | ソース | 型関連ユーティリティ |
| reflection.jl | `base/reflection.jl` | ソース | fieldnames, fieldtypes等のリフレクション |
| jltypes.c | `src/jltypes.c` | ソース（C） | DataType生成のCランタイム実装 |
| datatype.c | `src/datatype.c` | ソース（C） | フィールドレイアウト計算 |
