# 機能設計書 26-イテレータ

## 概要

本ドキュメントは、Julia Base ライブラリが提供する Iterators モジュールの設計について記述する。Iterators モジュールは、既存のイテラブルオブジェクトを変換・合成・制御するための遅延イテレータ構築関数群を提供する。

### 本機能の処理概要

Iterators モジュールは、`enumerate` / `zip` / `map` / `filter` / `flatten` / `product` / `take` / `drop` / `takewhile` / `dropwhile` / `cycle` / `repeated` / `partition` / `rest` / `countfrom` / `Stateful` / `accumulate` / `reverse` / `peel` / `flatmap` などの遅延イテレータ構築関数を提供する。これらはすべて遅延評価であり、要素の実体化は iterate 呼び出し時まで行われない。

**業務上の目的・背景**：大規模データの処理において、中間配列を生成せずにパイプライン的にデータ変換を行うことは、メモリ効率とパフォーマンスの両面で重要である。Iterators モジュールは、map / filter / flatten 等の操作を遅延的に合成し、最終的な collect や for ループでの消費時にのみ要素を生成する。これにより、無限イテレータの表現や、メモリに収まらないデータの処理が可能となる。

**機能の利用シーン**：大規模データの遅延変換パイプライン、無限数列の表現（countfrom, cycle, repeated）、複数イテレータの同期的走査（zip）、インデックス付きイテレーション（enumerate）、直積の列挙（product）、ウィンドウ処理（partition）で利用される。

**主要な処理内容**：
1. `enumerate(iter)` - カウンタ付きイテレーション
2. `zip(iters...)` - 複数イテレータの同期走査
3. `Iterators.map(f, iters...)` - 遅延マッピング
4. `Iterators.filter(f, iter)` - 遅延フィルタリング
5. `Iterators.flatten(iter)` - ネストされたイテレータの平坦化
6. `Iterators.product(iters...)` - 直積（デカルト積）
7. `take(iter, n)` / `drop(iter, n)` - 先頭n個の取得 / スキップ
8. `takewhile(f, iter)` / `dropwhile(f, iter)` - 条件付き取得 / スキップ
9. `cycle(iter)` / `repeated(x)` - 循環・繰り返しイテレータ
10. `partition(iter, n)` - n要素ごとのグループ化
11. `Iterators.Stateful(iter)` - 状態付きイテレータラッパー
12. `Iterators.reverse(iter)` - 逆順イテレーション
13. `Iterators.accumulate(op, iter)` - 遅延累積演算
14. `pairs(A)` - インデックスと値のペアイテレータ

**関連システム・外部連携**：なし。Base モジュール内で完結する。

**権限による制御**：なし。すべてのユーザーが利用可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面関連なし（ライブラリ関数） |

## 機能種別

計算処理（遅延データ変換・合成）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| iter / iters | Any (iterable) | Yes | 変換元のイテラブルオブジェクト | iterate プロトコルを実装 |
| f | Function | No | マッピング/フィルタリング関数 | 呼び出し可能であること |
| n | Integer | No | take/drop/partition の要素数 | 正の整数 |
| op | Function | No | accumulate の二項演算子 | 呼び出し可能であること |

### 入力データソース

Julia プログラム内のコレクション、Range、Generator、その他イテラブルオブジェクト。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| iterator | 各種イテレータ型 | 遅延イテレータオブジェクト（Enumerate, Zip, Reverse 等） |

### 出力先

呼び出し元への戻り値（for ループや collect で消費される）。

## 処理フロー

### 処理シーケンス

```
1. Iterators.XXX(args...) でイテレータ構築
   └─ 対応するラッパー構造体を作成（遅延、この時点で要素は生成しない）
2. for x in iterator / collect(iterator) でイテレーション開始
   └─ iterate(wrapper) が呼ばれる
   └─ 内部イテレータの iterate を呼び、変換を適用して返す
3. iterate(wrapper, state) で次の要素を取得
   └─ 内部イテレータの state を使って次の要素を取得
   └─ 変換（map/filter/flatten等）を適用
   └─ nothing が返されたら終了
```

### フローチャート

```mermaid
flowchart TD
    A[Iterators.XXX 構築] --> B[ラッパー構造体作成]
    B --> C[for/collect による消費]
    C --> D[iterate 呼び出し]
    D --> E{イテレータ種別}
    E -->|enumerate| F[カウンタ付加]
    E -->|zip| G[複数イテレータ同期]
    E -->|filter| H[条件判定+スキップ]
    E -->|flatten| I[内部イテレータ展開]
    E -->|take| J[カウンタ減算+終了判定]
    E -->|product| K[辞書式順序で直積]
    F --> L[次のiterate呼び出し]
    G --> L
    H --> L
    I --> L
    J --> L
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 遅延評価 | すべてのイテレータ構築は遅延的で、要素は消費時にのみ生成 | 常時 |
| BR-02 | zip終了条件 | zip は最も短いイテレータが尽きた時点で終了 | zip 使用時 |
| BR-03 | filter消費 | filter は条件を満たさない要素をスキップし、次の条件一致要素まで内部イテレータを消費 | filter 使用時 |
| BR-04 | product順序 | product は辞書式順序（最も右のイテレータが最速で回転） | product 使用時 |
| BR-05 | Stateful副作用 | Stateful は mutable で、iterate 呼び出しが内部状態を変更する | Stateful 使用時 |

### 計算ロジック

- `enumerate`: state = (counter, inner_state)、counter は 1 から開始し +1 ずつ増加
- `zip`: 各イテレータの iterate を順に呼び出し、1つでも nothing なら終了
- `product`: 辞書式順序で全組み合わせを列挙。要素数 = prod(length.(iters))

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（インメモリ計算のみ）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | MethodError | iterate が未定義の型にイテレータ操作 | iterate メソッドを定義 |
| - | ArgumentError | partition に n <= 0 を指定 | 正の整数を指定 |

### リトライ仕様

リトライは不要。

## トランザクション仕様

Stateful を除き、イテレータは関数型で副作用なし。Stateful は mutable な状態を持つ。

## パフォーマンス要件

- O(1) のイテレータ構築（ラッパー構造体の作成のみ）
- O(1) の追加メモリ（元のイテレータへの参照のみ）
- Transducer パターンにより、map/filter/flatten の合成が reduce 内で中間配列なしに実行可能

## セキュリティ考慮事項

特になし。

## 備考

- `Iterators.map(f, itr)` は `(f(x) for x in itr)` と同等の Generator を生成
- `zip()` は引数なしで呼ぶと無限の空タプルイテレータを返す
- `Iterators.Stateful` は一度消費した要素を再取得できないワンパスイテレータ
- Transducer パターン（reduce.jl の _xfadjoint）により、Generator/Filter/Flatten は reduce 内で自動的にリダクション関数の合成に変換される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

各イテレータのラッパー構造体を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | iterators.jl | `base/iterators.jl` | Reverse, Enumerate, Zip, Pairs 等の構造体定義 |

**読解のコツ**: 各イテレータは構造体として定義され、内部イテレータへの参照を保持する。iterate メソッドが構造体ごとにオーバーロードされており、これが遅延評価の実体である。`IteratorSize` / `IteratorEltype` トレイトにより、サイズ・要素型の情報が伝播される。

- **6行目**: `baremodule Iterators` - baremodule として定義（Base からの明示的インポート）
- **113-115行目**: `Reverse{T}` 構造体 - 逆順イテレーションのラッパー
- **150-152行目**: `Enumerate{I}` 構造体 - カウンタ付きラッパー
- **320-322行目**: `Zip{Is<:Tuple}` 構造体 - 複数イテレータの同期ラッパー

#### Step 2: enumerate と zip を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | iterators.jl | `base/iterators.jl` | enumerate, zip の iterate 実装 |

**主要処理フロー**:
1. **191行目**: `enumerate(iter)` - Enumerate(iter) を作成
2. **195-200行目**: `iterate(e::Enumerate, state)` - state = (counter, inner_state)
3. **324-327行目**: `zip(iters...)` - Zip(iters) を作成
4. zip の iterate は各サブイテレータを順に iterate し、いずれかが nothing なら終了

#### Step 3: pairs を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | iterators.jl | `base/iterators.jl` | pairs(IndexLinear/IndexCartesian, A) の実装 |

**主要処理フロー**:
- **265行目**: `pairs(::IndexLinear, A)` - Pairs(A, LinearIndices(A))
- **275行目**: `pairs(::IndexCartesian, A)` - Pairs(A, CartesianIndices(axes(A)))
- **282-291行目**: `iterate(p::Pairs)` - keys の iterate + data[idx]

#### Step 4: Reverse を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | iterators.jl | `base/iterators.jl` | Reverse の iterate 実装 |

**主要処理フロー**:
- **111行目**: `reverse(itr)` - Reverse(itr) を作成
- **124-129行目**: `iterate(A::Reverse{<:AbstractArray})` - reverse(eachindex(A.itr)) を使用
- **140行目**: `reverse(R::AbstractRange) = Base.reverse(R)` - Range はコピーが安価なので即座に反転

### プログラム呼び出し階層図

```
enumerate(iter) / zip(iters...) / Iterators.filter(f, iter) / ...
    |
    +-- ラッパー構造体作成（Enumerate / Zip / Filter / ...）
    |
    +-- iterate(wrapper) [for ループ / collect による消費]
    |       |
    |       +-- [Enumerate] iterate(e.itr, rest...) + カウンタ
    |       +-- [Zip] map(iterate, iters) → Tuple
    |       +-- [Filter] iterate + pred チェック + スキップ
    |       +-- [Flatten] 外部 iterate + 内部 iterate
    |       +-- [Reverse] reverse(eachindex(itr)) + getindex
    |       +-- [Product] 辞書式順序で組み合わせ生成
    |
    +-- [Transducer パターン] reduce 内で自動合成
            +-- _xfadjoint: Generator → MappingRF
            +-- _xfadjoint: Filter → FilteringRF
            +-- _xfadjoint: Flatten → FlatteningRF
```

### データフロー図

```
[入力]                      [処理]                              [出力]

iter (iterable)     ---->  Iterators.map(f, iter)
                           = Generator(f, iter)                ---->  遅延イテレータ

iter1, iter2        ---->  zip(iter1, iter2)
                           = Zip((iter1, iter2))               ---->  (a, b) タプル列

iter (iterable)     ---->  enumerate(iter)
                           = Enumerate(iter)                   ---->  (i, x) タプル列

iter (nested)       ---->  Iterators.flatten(iter)
                           = Flatten(iter)                     ---->  平坦化された要素列
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| iterators.jl | `base/iterators.jl` | ソース | Iterators モジュールの全実装（enumerate, zip, map, filter, flatten, product, take, drop, cycle, repeated, partition, Stateful, reverse, accumulate, pairs 等） |
| generator.jl | `base/generator.jl` | ソース | Generator 型の定義（Iterators.map の戻り値） |
| reduce.jl | `base/reduce.jl` | ソース | Transducer パターン（_xfadjoint）によるイテレータ変換のリダクション合成 |
