# 機能設計書 29-文字列基本操作

## 概要

本ドキュメントは、Julia Base ライブラリが提供する文字列基本操作機能の設計について記述する。String 型と SubString 型の生成・連結・繰り返し・長さ取得・インデックスアクセス・イテレーションなどの基本操作を提供する。

### 本機能の処理概要

文字列基本操作機能は、Julia の `String`（UTF-8 エンコード）と `SubString`（コピーなし部分文字列ビュー）を中心に、文字列の生成・連結（`*` 演算子）・繰り返し（`repeat` / `^`）・長さ取得（`length` / `ncodeunits`）・インデックスアクセス（`s[i]`）・イテレーション・コード単位操作（`codeunit`）・有効インデックス判定（`isvalid`）・インデックス算術（`thisind` / `nextind` / `prevind`）を提供する。

**業務上の目的・背景**：テキストデータの処理はプログラミングにおける最も基本的な操作の一つである。Julia は UTF-8 をネイティブエンコーディングとして採用しており、Unicode テキストを効率的に処理できる。文字列はイミュータブルであり、SubString によるコピーなし部分参照で効率的なスライス操作を実現している。インデックスはバイト位置ベースであり、可変長エンコーディングのため有効なインデックスのみでの文字アクセスが可能。

**機能の利用シーン**：テキストの連結・分割、ユーザー入力の処理、ファイルパスの構築、ログメッセージの生成、データのシリアライズ、テンプレートベースの文字列生成で利用される。

**主要な処理内容**：
1. `String` の生成（文字列リテラル、string 関数、コンストラクタ）
2. `*` 演算子による文字列連結
3. `repeat(s, n)` / `s^n` による文字列の繰り返し
4. `length(s)` による文字数の取得
5. `ncodeunits(s)` によるバイト数の取得
6. `s[i]` / `s[i:j]` によるインデックスアクセス
7. `SubString(s, i, j)` / `@view s[i:j]` によるコピーなし部分文字列
8. `codeunit(s, i)` によるバイトレベルアクセス
9. `isvalid(s, i)` による有効インデックス判定
10. `thisind` / `nextind` / `prevind` によるインデックス算術
11. `iterate(s)` による文字列のイテレーション
12. `reverse(s)` による文字列の反転

**関連システム・外部連携**：なし。Base モジュール内で完結する。

**権限による制御**：なし。すべてのユーザーが利用可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面関連なし（ライブラリ関数） |

## 機能種別

計算処理（文字列操作）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| s | AbstractString | Yes | 操作対象の文字列 | 有効な文字列 |
| i | Integer | No | バイトインデックス | 1 <= i <= ncodeunits(s) |
| n | Integer | No | 繰り返し回数 | 0以上の整数 |

### 入力データソース

Julia プログラム内の文字列リテラル、変数、I/O入力。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | String / SubString | 操作結果の文字列 |
| char | Char | インデックスアクセスの結果 |
| length | Int | 文字数 / バイト数 |
| valid | Bool | isvalid の結果 |

### 出力先

呼び出し元への戻り値。

## 処理フロー

### 処理シーケンス

```
1. 文字列インデックスアクセス s[i]
   └─ isvalid(s, i) で有効なインデックスか確認
   └─ 有効: UTF-8 バイト列からChar をデコード
   └─ 無効: StringIndexError を投げる
2. 文字列連結 s1 * s2
   └─ 各文字列のバイト数を合算
   └─ 新しいメモリを確保してバイト列をコピー
3. SubString(s, i, j)
   └─ 元の文字列への参照、オフセット、長さを保持
   └─ データのコピーは行わない
4. length(s)
   └─ UTF-8 バイト列を走査
   └─ 有効な文字の開始バイトをカウント
5. iterate(s, i)
   └─ i 位置の UTF-8 バイト列をデコード
   └─ (Char, 次のインデックス) を返す
```

### フローチャート

```mermaid
flowchart TD
    A[文字列操作] --> B{操作種別}
    B -->|s_i アクセス| C{isvalid?}
    C -->|Yes| D[UTF-8 デコード → Char]
    C -->|No| E[StringIndexError]
    B -->|s1 * s2 連結| F[バイト数合算 + メモリ確保 + コピー]
    B -->|SubString| G[参照+オフセット+長さ保持]
    B -->|length| H[UTF-8 バイト走査 + カウント]
    B -->|iterate| I[UTF-8 デコード + 次インデックス計算]
    B -->|nextind| J[次の有効インデックスを計算]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | UTF-8 エンコーディング | Julia の String は UTF-8 でエンコードされる | 常時 |
| BR-02 | バイトインデックス | 文字列インデックスはバイト位置を指す | 常時 |
| BR-03 | 有効インデックス | 文字の先頭バイトのインデックスのみが有効 | s[i] アクセス時 |
| BR-04 | イミュータブル | String は不変で setindex! は不可 | 常時 |
| BR-05 | SubString共有 | SubString は元文字列のデータを共有（コピーなし） | SubString 作成時 |
| BR-06 | 自己同期化 | UTF-8 は自己同期化コードで isvalid は O(1) | isvalid 呼び出し時 |

### 計算ロジック

- `ncodeunits(s)`: String のバイト長を返す（O(1)）
- `length(s)`: UTF-8 バイト列を走査して文字数をカウント（O(n)）
- `nextind(s, i)`: バイト位置 i の次の有効インデックスを返す
- `thisind(s, i)`: バイト位置 i を含む文字の先頭インデックスを返す

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（インメモリ計算のみ）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | StringIndexError | 無効なインデックスで s[i] | isvalid で事前確認、または nextind/prevind を使用 |
| - | BoundsError | 範囲外のインデックス | 1:ncodeunits(s) の範囲内で使用 |

### リトライ仕様

リトライは不要。

## トランザクション仕様

String はイミュータブルであり、トランザクション管理は不要。

## パフォーマンス要件

- ncodeunits: O(1)
- length: O(n)（UTF-8 の可変長のため全走査が必要）
- s[i]: O(1)（バイト位置から直接アクセス）
- `*` (連結): O(m+n)（両文字列のバイト数の合計）
- SubString: O(1)（コピーなし、参照のみ）
- iterate: 1文字あたり O(1)

## セキュリティ考慮事項

- 不正な UTF-8 バイト列を含む文字列の処理に対して安全（isvalid で検証可能）

## 備考

- Julia では `*` が文字列連結演算子（数学における自由群の乗法に由来）
- `string(x...)` 関数は任意のオブジェクトを文字列に変換して連結
- GenericString は AbstractString のテスト用汎用実装
- `codeunits(s)` はバイト配列としてのビューを返す（CodeUnits 型）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

AbstractString の要件と String / SubString の内部表現を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | strings/basic.jl | `base/strings/basic.jl` | AbstractString の抽象インタフェース（ncodeunits, codeunit, isvalid, iterate） |

**読解のコツ**: Julia の文字列は「コード単位（code unit）」によるバイト配列として表現される。AbstractString は4つの必須メソッド（ncodeunits, codeunit, isvalid, iterate）を要求する。インデックスはバイト位置であり、文字位置ではない点が重要。

- **5-41行目**: `AbstractString` の docstring - 文字列のエンコーディング仮定
- **46-68行目**: `ncodeunits(s)` - コード単位数を返す
- **70-82行目**: `codeunit(s)` - コード単位型を返す
- **86-108行目**: `codeunit(s, i)` - i番目のコード単位を返す
- **110-144行目**: `isvalid(s, i)` - 有効インデックス判定
- **146-150行目**: `iterate(s, i)` - 文字とNextインデックスを返す

#### Step 2: String / SubString を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | strings/string.jl | `base/strings/string.jl` | String 型の定義と UTF-8 デコード |
| 2-2 | strings/substring.jl | `base/strings/substring.jl` | SubString のビュー実装 |

**主要処理フロー**:
- String の iterate: UTF-8 のリーディングバイトから文字長を判定し、Char をデコード
- SubString: 親文字列への参照 + オフセット + ncodeunits で部分文字列を表現

#### Step 3: 文字列連結・繰り返しを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | strings/basic.jl | `base/strings/basic.jl` | `*` 演算子、repeat の実装 |

### プログラム呼び出し階層図

```
s[i] / s1 * s2 / length(s) / iterate(s) / SubString(s, i, j)
    |
    +-- s[i] (getindex)
    |       +-- isvalid(s, i) チェック
    |       +-- UTF-8 バイト列デコード → Char
    |
    +-- s1 * s2 (連結)
    |       +-- ncodeunits(s1) + ncodeunits(s2) → 合計バイト数
    |       +-- メモリ確保 + unsafe_copyto!
    |
    +-- length(s)
    |       +-- UTF-8 バイト走査
    |       +-- 有効文字開始バイトをカウント
    |
    +-- iterate(s, i)
    |       +-- codeunit(s, i) → リーディングバイト取得
    |       +-- UTF-8 文字長判定
    |       +-- Char デコード + nextind 計算
    |
    +-- SubString(s, i, j)
            +-- 参照: s.string (= s itself or s.string)
            +-- オフセット: i - 1
            +-- 長さ: j - i + 1
```

### データフロー図

```
[入力]                       [処理]                            [出力]

s (String)           ---->  s[i]
                            +-- UTF-8 デコード                ---->  Char

s1, s2 (String)      ---->  s1 * s2
                            +-- バイト列結合                  ---->  String (新規)

s (String), i, j     ---->  SubString(s, i, j)
                            +-- 参照+オフセット保持            ---->  SubString (ビュー)

s (String)           ---->  length(s)
                            +-- UTF-8 走査                    ---->  Int (文字数)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| strings/basic.jl | `base/strings/basic.jl` | ソース | AbstractString インタフェース、基本操作（ncodeunits, codeunit, isvalid, iterate） |
| strings/string.jl | `base/strings/string.jl` | ソース | String 型の定義と UTF-8 実装 |
| strings/substring.jl | `base/strings/substring.jl` | ソース | SubString 型のビュー実装 |
| strings/io.jl | `base/strings/io.jl` | ソース | 文字列の入出力（string 関数、print 等） |
| char.jl | `base/char.jl` | ソース | Char 型の定義（Unicode 文字） |
