# 機能設計書 3-抽象型・Union型

## 概要

本ドキュメントは、Julia言語における `abstract type` / `Union{}` による型階層の構築と多重ディスパッチ用の型集合定義に関する機能設計書である。

### 本機能の処理概要

Julia言語の型階層を構築するための抽象型（abstract type）と、複数型の和集合を表現するUnion型を定義する機能を提供する。抽象型はインスタンス化できない型であり型階層のノードとして機能し、Union型は有限個の具体型の和集合として多重ディスパッチのパターンマッチングに使用される。

**業務上の目的・背景**：Juliaの多重ディスパッチ機構において、共通のインターフェースを提供する抽象型と、複数の具体型をまとめて扱うUnion型は不可欠である。抽象型は継承階層を構築し、Union型はad-hocな型グルーピングを可能にする。これにより、型安全で拡張可能なコード設計が実現される。

**機能の利用シーン**：ライブラリAPIにおける型制約の定義（例: `function foo(x::Number)`）、型パラメータの上限・下限指定、条件分岐による型に基づくディスパッチ、`Union{Nothing, T}` によるオプショナル値の表現等で使用される。

**主要な処理内容**：
1. `abstract type` 宣言による抽象型の定義と型階層の構築
2. `Union{T1, T2, ...}` による型の和集合の生成
3. 型の部分型判定（`<:` 演算子）における抽象型・Union型の処理
4. `Union{}` (Bottom型) の定義と空集合としての振る舞い

**関連システム・外部連携**：Cランタイム（`src/jltypes.c`, `src/subtype.c`）における型判定処理、コンパイラの型推論エンジン（`Compiler/src/typelattice.jl`）と連携する。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 抽象型・Union型は画面に直接関連しないが、REPLのヘルプモードで型情報が表示される |

## 機能種別

型定義・型システム基盤

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| type_name | Symbol | Yes（abstract type） | 抽象型名 | 有効な識別子であること |
| supertype | DataType | No | 親抽象型（デフォルト: Any） | abstract typeであること |
| union_types | Vector{Type} | Yes（Union） | Union構成型 | 各型が有効な型であること |

### 入力データソース

Juliaソースコードの `abstract type` 宣言および `Union{...}` 式から、パーサーを通じて取得される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| DataType | Core.DataType | 抽象型のDataTypeオブジェクト（abstract=true） |
| Union | Core.Union | Union型オブジェクト（a, bフィールドで二分木表現） |
| Bottom | Union{} | 空のUnion（すべての型のサブタイプ） |

### 出力先

定義されたモジュールのスコープに型オブジェクトとしてバインドされる。

## 処理フロー

### 処理シーケンス

```
1. abstract type 宣言のパース
   └─ パーサーがExpr(:abstract, ...)に変換
2. 親型の検証
   └─ 指定された親型がabstract typeであることを確認
3. DataTypeオブジェクトの生成（abstract=true）
   └─ Cランタイムで抽象型DataTypeを生成
4. Union{...} 式の評価
   └─ 指定された型のリストからUnionオブジェクトを構築
5. 型の正規化
   └─ 重複排除、ネスト解消、ソート
```

### フローチャート

```mermaid
flowchart TD
    A[abstract type宣言] --> B[親型検証]
    B --> C{親型がabstract?}
    C -->|Yes| D[DataType生成 abstract=true]
    C -->|No| E[TypeError]
    D --> F[モジュールにバインド]

    G["Union{T1,T2,...}式"] --> H[型リスト評価]
    H --> I[正規化・重複排除]
    I --> J{型が1つ?}
    J -->|Yes| K[その型を返す]
    J -->|No| L[Unionオブジェクト生成]
    L --> M[二分木構造で格納]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 抽象型非インスタンス化 | abstract typeはインスタンス化できない | 常時 |
| BR-02 | 単一継承 | abstract typeは1つの親抽象型のみ持てる | abstract type定義時 |
| BR-03 | Union正規化 | Union{A, Union{B, C}} は Union{A, B, C} に正規化される | Union構築時 |
| BR-04 | Union単純化 | Union{T} は T に、Union{} は Bottom に等しい | Union構築時 |
| BR-05 | Bottom最小性 | Union{}（Bottom）はすべての型のサブタイプである | 型判定時 |
| BR-06 | Union二分木表現 | Union内部は (a, b) フィールドの二分木として表現される | Unionオブジェクト生成時 |

### 計算ロジック

型の部分型判定（`<:`）はCランタイムの `jl_subtype()` で実装され、抽象型に対しては型階層を走査、Union型に対しては構成要素すべてについて判定が行われる。

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（データベースを使用しない）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TypeError | TypeError | 具体型を親型として指定した場合 | abstract typeを親型に指定する |
| ArgumentError | ArgumentError | Union{}のインスタンス化を試みた場合 | Union{}は値を持てない |

### リトライ仕様

型定義は一回限りの処理であり、リトライは適用されない。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- 型の部分型判定（`<:`）はキャッシュ機構により高速化されている
- Union型は小さい型集合（概ね4型以下）で特に効率的にディスパッチされる

## セキュリティ考慮事項

特になし。

## 備考

- boot.jlの5-8行目にAny, Type{T}, Varargが抽象型としてコメント記載されている
- boot.jlの32-35行目にUnion構造体が (a, b) フィールドの二分木として定義されている
- `Bottom = Union{}` は `base/essentials.jl` 7行目で定義されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Union型とDataTypeの内部構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | boot.jl | `base/boot.jl` | **5-8行目**: Any, Type{T}, Varargの抽象型定義（コメント） |
| 1-2 | boot.jl | `base/boot.jl` | **32-35行目**: Union構造体の定義。a, bフィールドによる二分木表現 |
| 1-3 | essentials.jl | `base/essentials.jl` | **7行目**: `const Bottom = Union{}` - 空のUnion型 |

**読解のコツ**: Union型はJuliaの型格子（type lattice）においてjoin操作に対応する。`Union{A, B}` は型格子上のAとBの最小上界に近い概念である。内部的には二分木 `Union(a, b)` として格納される。

#### Step 2: エントリーポイントを理解する

抽象型・Union型がどのように定義・利用されるかを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | boot.jl | `base/boot.jl` | **248-253行目**: Number, Real, AbstractFloat, Integer, Signed, Unsignedの抽象型階層定義 |
| 2-2 | int.jl | `base/int.jl` | **26-40行目**: BitSigned, BitUnsigned, BitInteger等のUnion型定義 |

**主要処理フロー**:
1. **boot.jl 248行目**: `abstract type Number end` - 最上位数値抽象型
2. **int.jl 34行目**: `const BitSigned = Union{BitSigned_types...}` - 符号付き整数のUnion
3. **int.jl 36行目**: `const BitInteger = Union{BitInteger_types...}` - 全整数のUnion

#### Step 3: 型判定と型結合を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | promotion.jl | `base/promotion.jl` | **20-141行目**: typejoin関数 - 2つの型の最近共通祖先を計算 |
| 3-2 | promotion.jl | `base/promotion.jl` | **147-158行目**: typesplit関数 - Union型から特定の型を除去 |

### プログラム呼び出し階層図

```
abstract type 宣言
    │
    └─ Cランタイム (jl_new_abstracttype)
           └─ DataType生成 (abstract=true)

Union{T1, T2, ...} 式
    │
    ├─ 型リストの評価
    │      ├─ 各型の解決
    │      └─ 正規化（重複排除・フラット化）
    │
    └─ jl_type_union() (Cランタイム)
           └─ Union(a, b) 二分木生成

typejoin(a, b) (base/promotion.jl)
    │
    ├─ TypeVar の処理
    ├─ UnionAll の処理
    ├─ Union の処理（構成要素を再帰的にjoin）
    └─ DataType の処理（型階層を走査）
```

### データフロー図

```
[入力]                          [処理]                      [出力]

abstract type宣言       ───▶ jl_new_abstracttype()  ───▶ DataType (abstract=true)

Union{T1, T2, ...}      ───▶ jl_type_union()        ───▶ Union(a, b)

型A, 型B                ───▶ typejoin(A, B)         ───▶ 最近共通祖先型

型A, 型B                ───▶ A <: B (jl_subtype)    ───▶ Bool
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| boot.jl | `base/boot.jl` | ソース | 抽象型階層とUnion型のCリファレンス定義 |
| essentials.jl | `base/essentials.jl` | ソース | Bottom (Union{}) の定義 |
| promotion.jl | `base/promotion.jl` | ソース | typejoin, typesplit関数 |
| int.jl | `base/int.jl` | ソース | BitSigned等のUnion型エイリアス定義 |
| jltypes.c | `src/jltypes.c` | ソース（C） | 型システムのCランタイム実装 |
| subtype.c | `src/subtype.c` | ソース（C） | 部分型判定のCランタイム実装 |
| typelattice.jl | `Compiler/src/typelattice.jl` | ソース | コンパイラの型格子定義 |
