# 機能設計書 4-パラメトリック型

## 概要

本ドキュメントは、Julia言語における型パラメータ（TypeVar / UnionAll）による汎用型の定義に関する機能設計書である。

### 本機能の処理概要

Julia言語のパラメトリック型（Parametric Types）機能を提供する。型パラメータを持つ構造体・抽象型・プリミティブ型の定義と、TypeVar（型変数）およびUnionAll（全称量化型）による汎用的な型表現を実現する。

**業務上の目的・背景**：型安全かつ汎用的なデータ構造（例: `Array{T,N}`, `Dict{K,V}`, `Complex{T}`）を定義するために不可欠な機能である。パラメトリック型はコンパイラによる特殊化（specialization）の基盤となり、ジェネリックプログラミングと高性能コードの両立を実現する。各型パラメータの具体化ごとに特殊化されたコードが生成されるため、動的言語でありながらC/Fortran並みの性能を達成できる。

**機能の利用シーン**：コンテナ型の定義（`Vector{Int}`, `Dict{String, Any}`等）、数値型のパラメトリック拡張（`Complex{Float64}`等）、カスタムデータ構造のジェネリック定義、メソッドシグネチャにおける型制約（`where T <: Number`等）で使用される。

**主要な処理内容**：
1. TypeVar（型変数）の生成と上下界の設定
2. UnionAll型の構築（`Array{T,N} where {T, N}` の表現）
3. パラメトリック型の特殊化（`Array{Int,1}` の生成）
4. `where` 句による型パラメータ制約の処理

**関連システム・外部連携**：Cランタイム（`src/jltypes.c`）でのUnionAll構築、コンパイラ型推論（`Compiler/src/abstractinterpretation.jl`）での型パラメータ解決と連携する。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | パラメトリック型は画面に直接関連しないが、REPLでの型表示に使用される |

## 機能種別

型定義・ジェネリクス基盤

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| name | Symbol | Yes（TypeVar） | 型変数名（例: :T） | 有効な識別子であること |
| lb | Type | No | 型変数の下界（デフォルト: Union{}） | 有効な型であること |
| ub | Type | No | 型変数の上界（デフォルト: Any） | lb <: ub であること |
| var | TypeVar | Yes（UnionAll） | 全称量化する型変数 | 有効なTypeVarであること |
| body | Type | Yes（UnionAll） | UnionAllの本体型 | 型変数を含む型であること |

### 入力データソース

Juliaソースコードの型パラメータ宣言（`{T}`, `{T<:Number}`, `where T`等）から、パーサーを通じて取得される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| TypeVar | Core.TypeVar | 型変数オブジェクト（name, lb, ubフィールド） |
| UnionAll | Core.UnionAll | 全称量化型オブジェクト（var, bodyフィールド） |
| 特殊化型 | DataType | 型パラメータが具体化されたDataType |

### 出力先

型オブジェクトとしてJuliaの型システム内に生成される。

## 処理フロー

### 処理シーケンス

```
1. 型パラメータの宣言パース
   └─ {T}, {T<:Number}, where T 等の構文をパース
2. TypeVarの生成
   └─ _typevar(name, lb, ub) で型変数オブジェクトを生成
3. UnionAllの構築
   └─ ccall(:jl_type_unionall, ...) で全称量化型を構築
4. 型の特殊化（利用時）
   └─ 具体的な型パラメータでDataTypeを生成
5. 型キャッシュの確認
   └─ 既に同じパラメータで特殊化された型があればそれを再利用
```

### フローチャート

```mermaid
flowchart TD
    A["型パラメータ宣言 {T<:Number}"] --> B[TypeVar生成]
    B --> C["UnionAll構築 (body where T)"]
    C --> D[型定義完了]

    E["型特殊化 MyType{Int}"] --> F{キャッシュ確認}
    F -->|Hit| G[キャッシュ済み型を返す]
    F -->|Miss| H[新規DataType生成]
    H --> I[キャッシュに登録]
    I --> J[特殊化型を返す]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 上下界制約 | TypeVarの下界は上界のサブタイプでなければならない（lb <: ub） | TypeVar生成時 |
| BR-02 | 不変性 | Juliaのパラメトリック型は不変（invariant）であり、`Vector{Int} <: Vector{Number}` は成立しない | 型判定時 |
| BR-03 | 共変where | `Vector{T} where T<:Number` は `Vector{Int}`, `Vector{Float64}` 等の和集合を表す | where句使用時 |
| BR-04 | 型キャッシュ | 同じ型パラメータの組み合わせは一意なDataTypeオブジェクトにキャッシュされる | 型特殊化時 |

### 計算ロジック

パラメトリック型の特殊化はCランタイムの `jl_apply_type()` で実行され、型パラメータの適用・境界チェック・キャッシュ検索が行われる。

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（データベースを使用しない）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TypeError | TypeError | 型パラメータが上界を満たさない場合 | 制約を満たす型を指定する |
| ArgumentError | ArgumentError | パラメータ数が一致しない場合 | 正しい数の型パラメータを指定する |

### リトライ仕様

型定義は一回限りの処理であり、リトライは適用されない。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- 型パラメータの具体化ごとにコンパイラが特殊化したコードを生成し、ボックス化のオーバーヘッドを除去する
- 型キャッシュにより同一パラメータの型は一度だけ生成される

## セキュリティ考慮事項

特になし。

## 備考

- boot.jlの37-46行目にTypeVarとUnionAll構造体がコメント形式で定義されている
- boot.jlの335-338行目にTypeVarとUnionAllのコンストラクタが定義されている
- `has_free_typevars()` 関数（boot.jl 354行目）で型が自由な型変数を含むか判定できる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

TypeVarとUnionAllの内部構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | boot.jl | `base/boot.jl` | **37-41行目**: TypeVar構造体の定義（name::Symbol, lb::Type, ub::Type）|
| 1-2 | boot.jl | `base/boot.jl` | **43-46行目**: UnionAll構造体の定義（var::TypeVar, body）|

**読解のコツ**: UnionAllは「forall T. body」を表す全称量化型である。例えば `Vector{T} where T` は `UnionAll(TypeVar(:T, Union{}, Any), Array{T,1})` と表現される。ネストされたwhere句（`A{S,T} where S where T`）は外側のUnionAllが先に適用される。

#### Step 2: エントリーポイントを理解する

TypeVarとUnionAllの生成処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | boot.jl | `base/boot.jl` | **335行目**: `TypeVar(@nospecialize(n)) = _typevar(n::Symbol, Union{}, Any)` - デフォルト上下界 |
| 2-2 | boot.jl | `base/boot.jl` | **336行目**: `TypeVar(@nospecialize(n), @nospecialize(ub))` - 上界指定 |
| 2-3 | boot.jl | `base/boot.jl` | **337行目**: `TypeVar(@nospecialize(n), @nospecialize(lb), @nospecialize(ub))` - 上下界指定 |
| 2-4 | boot.jl | `base/boot.jl` | **338行目**: `UnionAll(@nospecialize(v), @nospecialize(t)) = ccall(:jl_type_unionall, ...)` |

**主要処理フロー**:
1. **335行目**: TypeVar(:T) で下界=Union{}, 上界=Any の型変数生成
2. **338行目**: UnionAll(tv, body) でccallを通じてCランタイムで全称量化型構築
3. **354行目**: `has_free_typevars()` で型変数の未束縛判定

#### Step 3: 型パラメータの利用例を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | complex.jl | `base/complex.jl` | **13-16行目**: `struct Complex{T<:Real} <: Number` - パラメトリック型の実例 |
| 3-2 | rational.jl | `base/rational.jl` | **9-15行目**: `struct Rational{T<:Integer} <: Real` - パラメトリック型の実例 |
| 3-3 | essentials.jl | `base/essentials.jl` | TypeVar関連ユーティリティ |

### プログラム呼び出し階層図

```
TypeVar コンストラクタ (base/boot.jl:335-337)
    │
    └─ _typevar(name, lb, ub) ... Cランタイム
           └─ TypeVarオブジェクト生成

UnionAll コンストラクタ (base/boot.jl:338)
    │
    └─ ccall(:jl_type_unionall, ...)
           └─ UnionAllオブジェクト生成

型特殊化 (例: Complex{Float64})
    │
    └─ jl_apply_type() (Cランタイム)
           ├─ TypeVarの境界チェック
           ├─ 型キャッシュ検索
           └─ 新規DataType生成（キャッシュミス時）
```

### データフロー図

```
[入力]                              [処理]                         [出力]

型パラメータ宣言             ───▶ TypeVar生成               ───▶ TypeVarオブジェクト
  ({T<:Number})                    (_typevar)

TypeVar + body型             ───▶ UnionAll構築              ───▶ UnionAllオブジェクト
                                   (jl_type_unionall)

UnionAll + 具体型            ───▶ 型特殊化                  ───▶ 具体DataType
  (Complex{Float64})               (jl_apply_type)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| boot.jl | `base/boot.jl` | ソース | TypeVar/UnionAll定義、コンストラクタ |
| essentials.jl | `base/essentials.jl` | ソース | 型関連ユーティリティ |
| complex.jl | `base/complex.jl` | ソース | パラメトリック型の実例（Complex{T}） |
| rational.jl | `base/rational.jl` | ソース | パラメトリック型の実例（Rational{T}） |
| jltypes.c | `src/jltypes.c` | ソース（C） | UnionAll構築・型特殊化のCランタイム |
| subtype.c | `src/subtype.c` | ソース（C） | パラメトリック型の部分型判定 |
