# 機能設計書 43-ファイル情報

## 概要

Julia Base ライブラリにおけるファイルのメタデータ（サイズ、パーミッション、タイムスタンプ、種別等）の取得を提供する機能の設計書である。

### 本機能の処理概要

**業務上の目的・背景**：ファイル処理を行う前に、対象ファイルの存在確認、種別判定（通常ファイル / ディレクトリ / シンボリックリンク等）、サイズ取得、パーミッション確認などのメタデータ取得が必要となる。本機能は `stat` システムコールを Julia から利用可能にし、ファイルに関する全情報を構造化して提供する。

**機能の利用シーン**：ファイルの存在確認（`isfile`, `isdir`, `ispath`）、ファイルサイズの取得（`filesize`）、パーミッション確認（`isreadable`, `iswritable`, `isexecutable`）、ファイル種別判定（`islink`, `isblockdev`, `ischardev`, `isfifo`, `issocket`）、タイムスタンプ取得（`mtime`, `ctime`）など。

**主要な処理内容**：
1. `stat` / `lstat` によるファイル情報の取得（StatStruct への格納）
2. `isfile` / `isdir` / `ispath` / `islink` によるファイル種別判定
3. `filesize` によるファイルサイズ取得
4. `filemode` によるファイルモード取得
5. `mtime` / `ctime` によるタイムスタンプ取得
6. `isreadable` / `iswritable` / `isexecutable` による権限判定
7. `uperm` / `gperm` / `operm` によるパーミッションビット取得
8. `issetuid` / `issetgid` / `issticky` による特殊ビット判定
9. `samefile` による同一ファイル判定

**関連システム・外部連携**：OS の `stat(2)` / `lstat(2)` システムコールを libuv 経由で呼び出す。

**権限による制御**：`stat` の実行自体にはファイルの読み取り権限は不要だが、親ディレクトリの実行権限が必要。`isreadable` / `iswritable` / `isexecutable` は `access(2)` またはパーミッションビットから判定する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | CLI / REPL | 主画面 | stat 関数の対話的実行 |

## 機能種別

情報取得 / ユーティリティ

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| path | AbstractString | Yes | 対象ファイルのパス | - |
| fd | RawFD / IO | No | 既に開いているファイルディスクリプタ | - |

### 入力データソース

ユーザーコードからファイルパス文字列またはファイルディスクリプタとして渡される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| StatStruct | StatStruct | ファイルの全メタデータを格納する構造体 |
| size | Int64 | ファイルサイズ（バイト） |
| mode | UInt32 | ファイルモード（パーミッション + 種別） |
| mtime | Float64 | 最終変更時刻（Unix エポック秒） |
| ctime | Float64 | 最終ステータス変更時刻 |
| device/inode/nlink/uid/gid | 各種整数型 | ファイルシステム固有の情報 |

### 出力先

関数の戻り値としてユーザーコードに返される。

## 処理フロー

### 処理シーケンス

```
1. パスまたは FD の受け取り
   └─ AbstractString または RawFD/IO として引数を受け取る
2. stat/lstat システムコール実行
   └─ ccall 経由で libuv の jl_stat / jl_lstat を呼び出す
3. 結果の StatStruct への格納
   └─ C 構造体のバイト列から Julia 構造体へフィールドを抽出
4. 述語関数の判定
   └─ StatStruct のフィールドに基づいてブール値を返す
```

### フローチャート

```mermaid
flowchart TD
    A[パス/FD入力] --> B[stat/lstat 呼出]
    B --> C[StatStruct 生成]
    C --> D{判定関数?}
    D -->|isfile| E[S_ISREG チェック]
    D -->|isdir| F[S_ISDIR チェック]
    D -->|islink| G[lstat + S_ISLNK チェック]
    D -->|filesize| H[st_size 返却]
    D -->|mtime| I[st_mtime 返却]
    D -->|filemode| J[st_mode 返却]
    E --> K[Bool 返却]
    F --> K
    G --> K
    H --> L[数値返却]
    I --> L
    J --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-43-01 | stat vs lstat | stat はシンボリックリンクを追従、lstat はリンク自体の情報を返す | リンクファイルへの操作時 |
| BR-43-02 | 存在しないファイル | 存在しないパスに対する stat は全フィールドがゼロの StatStruct を返す | ispath(path) == false の場合 |
| BR-43-03 | isfile と islink | isfile は stat（追従）、islink は lstat（非追従）を使用する | シンボリックリンク判定時 |
| BR-43-04 | filesize の対象 | ディレクトリに対する filesize は OS 依存の値を返す | isdir(path) == true の場合 |

### 計算ロジック

パーミッション判定（`uperm`, `gperm`, `operm`）はファイルモードの該当ビットフィールドを抽出して返す。`isreadable` 等の述語は `access(2)` に相当するチェックを行う。

## データベース操作仕様

### 操作別データベース影響一覧

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | (暗黙) | 存在しないパスへの stat | ゼロ初期化された StatStruct が返される（エラーにならない） |
| IOError | SystemError | filesize 等で FD が不正 | 有効な FD を使用する |

### リトライ仕様

stat は読み取り専用操作であり、リトライは不要。

## トランザクション仕様

該当なし。stat は読み取り専用でアトミック。

## パフォーマンス要件

stat システムコールは通常マイクロ秒オーダーで完了する。ただしネットワークファイルシステムの場合は遅延が生じ得る。

## セキュリティ考慮事項

- stat と実際のファイル操作の間に TOCTOU（Time of Check, Time of Use）競合が発生する可能性がある
- ファイルのパーミッション情報はセキュリティ上の判断に使用されるが、確認と使用の間にパーミッションが変更される可能性がある

## 備考

- StatStruct は `show` メソッドが定義されており、`ls -la` 風の表示が可能
- `stat` の結果はキャッシュされないため、毎回ファイルシステムにアクセスする
- `samefile` はデバイスID とinode番号の比較で同一ファイルを判定する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

StatStruct がファイル情報の中心的なデータ構造である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stat.jl | `base/stat.jl` | **1-65行目**: StatStruct の定義。C の `struct stat` に対応するフィールド（device, inode, mode, nlink, uid, gid, rdev, size, blksize, blocks, mtime, ctime）を持つ |

**読解のコツ**: StatStruct は `struct stat` の Julia 表現であり、各フィールドは対応する C 構造体のメンバーと同じ意味を持つ。`mode` フィールドが最も重要で、ファイル種別とパーミッションの両方を含む。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stat.jl | `base/stat.jl` | **67-90行目**: `stat(path)` / `lstat(path)` 関数。ccall で `jl_stat` / `jl_lstat` を呼び出し、結果バッファから StatStruct を構築する |

**主要処理フロー**:
1. **67-75行目**: `stat(path::AbstractString)` - パスから StatStruct を取得
2. **77-85行目**: `lstat(path::AbstractString)` - シンボリックリンク非追従版
3. **87-95行目**: `stat(fd::RawFD)` - ファイルディスクリプタから StatStruct を取得

#### Step 3: 述語関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stat.jl | `base/stat.jl` | **97-200行目**: ファイル種別判定関数群（isfile, isdir, islink, isblockdev, ischardev, isfifo, issocket, ispath）。いずれも StatStruct の mode フィールドをビットマスクでチェック |
| 3-2 | stat.jl | `base/stat.jl` | **202-260行目**: パーミッション関連関数（filemode, uperm, gperm, operm, issetuid, issetgid, issticky） |
| 3-3 | stat.jl | `base/stat.jl` | **262-300行目**: その他のアクセサ関数（filesize, mtime, ctime, samefile） |

### プログラム呼び出し階層図

```
stat(path::AbstractString)
    └─ ccall(:jl_stat, Cint, (Cstring, Ptr{UInt8}), path, buf)
        └─ StatStruct(buf) ... バイト配列から構造体を構築

isfile(path)
    └─ stat(path)
        └─ S_ISREG(st.mode)

isdir(path)
    └─ stat(path)
        └─ S_ISDIR(st.mode)

islink(path)
    └─ lstat(path)  ... stat ではなく lstat を使用
        └─ S_ISLNK(st.mode)

filesize(path)
    └─ stat(path).size

mtime(path)
    └─ stat(path).mtime

isreadable(path)
    └─ ccall(:jl_fs_access, Cint, ..., R_OK)
```

### データフロー図

```
[入力]              [処理]                            [出力]

パス文字列 ──────▶ stat() ─▶ jl_stat (C)     ──────▶ StatStruct
                                                       ├─ .size  ──▶ filesize
                                                       ├─ .mode  ──▶ isfile/isdir/filemode
                                                       ├─ .mtime ──▶ mtime
                                                       └─ .ctime ──▶ ctime

パス文字列 ──────▶ lstat() ─▶ jl_lstat (C)   ──────▶ StatStruct
                                                       └─ .mode  ──▶ islink
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stat.jl | `base/stat.jl` | ソース | StatStruct 定義および全ファイル情報関数 |
| file.jl | `base/file.jl` | ソース | stat 情報を利用するファイル操作 |
| path.jl | `base/path.jl` | ソース | パス操作（stat の入力パスの前処理） |
| filesystem.jl | `base/filesystem.jl` | ソース | 低レベルファイルシステム操作 |
