# 機能設計書 46-Task（コルーチン）

## 概要

Julia Base ライブラリにおける Task（コルーチン）の作成・スケジューリング・同期を提供する機能の設計書である。

### 本機能の処理概要

**業務上の目的・背景**：Julia の並行処理モデルの基盤である Task は、協調的マルチタスキング（コルーチン）を実現する。I/O 待ちの間に別の計算を進めるなど、非同期処理の効率的な記述を可能にする。Task はスレッド上で実行されるが、一つの Task が yield するまで同一スレッド上で連続実行される協調型スケジューリングを採用する。

**機能の利用シーン**：非同期 I/O 処理、バックグラウンドタスクの実行、`@async` / `@sync` マクロによる並行処理の記述、Channel を使ったプロデューサー・コンシューマーパターンの実装、タスクローカルストレージの利用など。

**主要な処理内容**：
1. `Task(f)` によるタスクの作成
2. `schedule(t)` によるタスクのスケジューリング
3. `wait(t)` / `fetch(t)` によるタスクの完了待ち・結果取得
4. `yield()` / `yieldto(t)` による制御の譲渡
5. `@async` / `@sync` マクロによる並行処理パターン
6. `current_task()` による現在タスクの取得
7. `istaskdone` / `istaskfailed` / `istaskstarted` によるタスク状態の確認
8. `task_local_storage` によるタスクローカルデータの管理
9. `bind(channel, task)` によるチャネル・タスクのライフサイクル連結

**関連システム・外部連携**：Julia ランタイムのスケジューラ（C 実装）、スレッドシステム、I/O イベントループ（libuv）と連携する。

**権限による制御**：Task の作成・実行に権限制御はない。ただし Task 内で実行される処理はそのプロセスの権限に従う。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | CLI / REPL | 主画面 | @async/@sync によるタスクの対話的実行 |

## 機能種別

並行処理基盤 / ランタイム機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| f | Function (引数なし) | Yes | タスクとして実行する関数 | 呼び出し可能であること |
| sticky | Bool | No | スレッド固定フラグ（デフォルト: true） | - |
| val | Any | No | schedule 時にタスクに渡す値 | - |

### 入力データソース

ユーザーコードから関数オブジェクト（クロージャ含む）として渡される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| task | Task | 作成されたタスクオブジェクト |
| result | Any | fetch(t) で取得するタスクの実行結果 |
| state | Symbol | タスクの状態（:runnable, :done, :failed） |

### 出力先

関数の戻り値としてユーザーコードに返される。

## 処理フロー

### 処理シーケンス

```
1. タスクの作成
   └─ Task(f) で関数をラップしたコルーチンオブジェクトを生成
2. タスクのスケジューリング
   └─ schedule(t) でランタイムのキューにタスクを追加
3. タスクの実行
   └─ スケジューラがキューから取り出して実行
4. 制御の譲渡
   └─ yield() で他のタスクに実行権を渡す
5. タスクの完了
   └─ 関数の正常終了 → state = :done
   └─ 例外発生 → state = :failed
6. 結果の取得
   └─ wait(t): 完了を待つ（例外は TaskFailedException としてラップ）
   └─ fetch(t): 完了を待ち結果を返す（例外は再throw）
```

### フローチャート

```mermaid
flowchart TD
    A[Task 作成] --> B[schedule]
    B --> C[スケジューラキュー]
    C --> D[タスク実行開始]
    D --> E{yield?}
    E -->|Yes| F[他タスクに制御譲渡]
    F --> C
    E -->|No| G{完了?}
    G -->|正常終了| H[state = :done]
    G -->|例外| I[state = :failed]
    G -->|未完了| D
    H --> J[wait/fetch で結果取得]
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-46-01 | sticky タスク | sticky=true（デフォルト）のタスクは作成されたスレッドに固定される | Task 作成時 |
| BR-46-02 | @async の暗黙的 schedule | @async は Task 作成と schedule を同時に行う | @async 使用時 |
| BR-46-03 | @sync の待機 | @sync ブロック内の全 @async タスクの完了を待つ | @sync 使用時 |
| BR-46-04 | fetch の例外伝播 | タスクが例外で終了した場合、fetch は例外を再 throw する | fetch 呼出時 |
| BR-46-05 | wait の TaskFailedException | wait は失敗タスクを TaskFailedException でラップする | wait 呼出時 |
| BR-46-06 | task_local_storage の独立性 | 各タスクは独立したローカルストレージを持つ | task_local_storage 使用時 |

### 計算ロジック

スケジューラは FIFO キューベースで、yield されたタスクをキューの末尾に追加する。タスクのマイグレーション（sticky=false の場合）により、空いているスレッドに自動的に移動する。

## データベース操作仕様

### 操作別データベース影響一覧

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TaskFailedException | Exception | タスク内で未捕捉例外が発生 | try-catch でタスク内の例外をハンドル |
| ConcurrencyViolationError | Exception | 同期プリミティブの不正使用 | ロック取得・解放の順序を確認 |
| ErrorException | Exception | 完了済みタスクを再スケジュール | タスクの状態を確認してからスケジュール |

### リトライ仕様

タスクのリトライは自動的には行われない。アプリケーション層で新しいタスクを作成してリトライする。

## トランザクション仕様

Task 自体にはトランザクション機能はない。`@sync` ブロックは全タスクの完了を保証するが、一部失敗時のロールバックは行わない（CompositeException を throw する）。

## パフォーマンス要件

- Task の作成は軽量（数百ナノ秒〜マイクロ秒オーダー）
- yield は高速なコンテキストスイッチ（ナノ秒オーダー）
- 大量のタスク（数百万単位）を作成しても動作可能

## セキュリティ考慮事項

- タスクは同一プロセス内で実行されるため、メモリ空間を共有する
- task_local_storage は他のタスクからアクセス可能（プライベートではない）
- データ競合の防止はユーザーの責任

## 備考

- Task は Julia の並行処理の基本単位であり、`@threads` や Channel もこの上に構築されている
- Task はスタックフルコルーチンとして実装されている（独自のスタック領域を持つ）
- Julia 1.3 以降、Task は sticky=false に設定することでスレッド間マイグレーションが可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Task 構造体の定義は Julia のコア型として定義されている。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | task.jl | `base/task.jl` | **1-80行目**: Task 型の基本的な操作。`current_task()` の定義（ccall :jl_get_current_task） |
| 1-2 | task.jl | `base/task.jl` | **82-120行目**: タスク状態の確認関数 `istaskdone`, `istaskfailed`, `istaskstarted` |

**読解のコツ**: Task 型自体は C ランタイムで定義されている（`src/jltypes.c`）。Julia 側では主に操作関数を定義する。`sticky` フィールドがスレッド固定を制御する重要なフラグ。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | task.jl | `base/task.jl` | **122-170行目**: `schedule(t)` / `schedule(t, val)` - タスクをランタイムキューに追加 |
| 2-2 | task.jl | `base/task.jl` | **172-220行目**: `yield()` / `yieldto(t, val)` - 制御の譲渡 |
| 2-3 | task.jl | `base/task.jl` | **222-280行目**: `wait(t::Task)` - タスクの完了待ち |
| 2-4 | task.jl | `base/task.jl` | **282-340行目**: `fetch(t::Task)` - タスクの結果取得 |

**主要処理フロー**:
1. **122-140行目**: `schedule` は `ccall(:jl_wakeup_thread)` でスレッドにタスク実行を通知
2. **172-190行目**: `yield` は `wait()` を呼び出してスケジューラに戻る
3. **222-260行目**: `wait` は `_wait(t)` でタスクの完了を待ち、failed の場合は `TaskFailedException` をラップ
4. **282-310行目**: `fetch` は `wait` の後に `t.result` を返す

#### Step 3: マクロとヘルパーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | task.jl | `base/task.jl` | **342-400行目**: `@async` マクロ - Task 作成 + schedule をワンステップで |
| 3-2 | task.jl | `base/task.jl` | **402-460行目**: `@sync` マクロ - ブロック内の全 @async タスクの同期 |
| 3-3 | task.jl | `base/task.jl` | **462-520行目**: `task_local_storage` - Dict ベースのタスクローカルストレージ |

### プログラム呼び出し階層図

```
@async expr
    ├─ Task(() -> expr)     ... タスク作成
    ├─ put!(sync_var, task) ... @sync 変数に登録
    └─ schedule(task)       ... スケジュール
        └─ ccall(:jl_wakeup_thread) ... ランタイムに通知

@sync begin ... end
    ├─ sync_var = Channel()
    ├─ [ブロック内の @async が put! で登録]
    └─ sync_end(sync_var)  ... 全タスクの wait
        └─ wait(task) for each task
            └─ _wait(task) ... 完了まで suspend

fetch(task)
    ├─ wait(task)
    └─ task.result          ... 結果取得

yield()
    └─ wait()               ... スケジューラに戻る
        └─ [次のタスクが実行される]
```

### データフロー図

```
[入力]                    [処理]                           [出力]

関数 f ──────────▶ Task(f) ──────────────────────▶ Task オブジェクト
Task   ──────────▶ schedule ──▶ ランタイムキュー   ──▶ タスク実行開始
Task   ──────────▶ wait ──▶ suspend + resume     ──▶ タスク完了通知
Task   ──────────▶ fetch ──▶ wait + result        ──▶ 実行結果
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| task.jl | `base/task.jl` | ソース | Task の Julia 側操作関数・マクロ定義 |
| condition.jl | `base/condition.jl` | ソース | Condition（タスク間同期の基盤） |
| lock.jl | `base/lock.jl` | ソース | ReentrantLock（タスク間の排他制御） |
| channels.jl | `base/channels.jl` | ソース | Channel（タスク間通信） |
| threadingconstructs.jl | `base/threadingconstructs.jl` | ソース | @spawn マクロ（マルチスレッドタスク） |
| task.c | `src/task.c` | ソース(C) | Task のランタイム実装（スタック管理等） |
| scheduler.c | `src/scheduler.c` | ソース(C) | タスクスケジューラの C 実装 |
