# 機能設計書 6-整数演算

## 概要

本ドキュメントは、Julia言語における加減乗除・剰余・べき乗・ビット演算などの整数演算機能に関する機能設計書である。

### 本機能の処理概要

Julia言語の整数型（Int8〜Int128, UInt8〜UInt128, Bool）に対する基本的な算術演算およびビット演算を提供する。CPUネイティブ命令にマッピングされた高速な整数演算基盤である。

**業務上の目的・背景**：すべての数値計算プログラムの基礎となる整数演算を提供する。Julia言語ではオーバーフロー検出なしの高速演算（デフォルト）と、オーバーフロー検出付き演算（checked_xxx系関数）の両方をサポートする。これにより、パフォーマンスと安全性を開発者が選択できる設計となっている。

**機能の利用シーン**：数値計算の基本演算、配列インデックス計算、ループカウンタ、ビットマスク操作、ハッシュ値計算、暗号処理、プロトコルのビットフィールド操作等で使用される。

**主要な処理内容**：
1. 基本算術演算（+, -, *, div, rem, mod, ^）
2. ビット演算（&, |, xor, ~, <<, >>, >>>）
3. 比較演算（<, <=, ==）
4. 符号操作（abs, sign, signbit, copysign, flipsign）
5. 偶奇判定（isodd, iseven）
6. 符号付き・符号なし変換（signed, unsigned）

**関連システム・外部連携**：Core.Intrinsics による低レベル演算プリミティブ（add_int, sub_int, mul_int等）と連携する。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 整数演算は画面に直接関連しないが、REPL上での演算結果表示に使用される |

## 機能種別

計算処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| x | BitInteger | Yes | 第一オペランド | 整数型であること |
| y | BitInteger | Yes（二項演算） | 第二オペランド | 整数型であること、divの場合はゼロでないこと |

### 入力データソース

関数呼び出し引数として直接渡される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 演算結果 | BitInteger | 算術演算の結果（同じ型、またはpromoteされた型） |
| 比較結果 | Bool | 比較演算の結果 |
| ビット演算結果 | BitInteger | ビット演算の結果（同じ型） |

### 出力先

関数の戻り値として返される。

## 処理フロー

### 処理シーケンス

```
1. 引数の型チェック
   └─ 多重ディスパッチにより適切なメソッドが選択
2. 同一型の場合：直接Intrinsics呼び出し
   └─ add_int, sub_int, mul_int等のCPU命令ベース演算
3. 異なる型の場合：型昇格後に演算
   └─ promote(x, y) で共通型に変換後、同一型メソッドを呼び出し
4. 整数除算の場合：float変換
   └─ /(x::Integer, y::Integer) = float(x) / float(y)
```

### フローチャート

```mermaid
flowchart TD
    A[整数演算呼び出し] --> B{同一型?}
    B -->|Yes| C[Intrinsics呼び出し]
    B -->|No| D[型昇格 promote]
    D --> C
    C --> E{演算種別}
    E -->|加算| F[add_int]
    E -->|減算| G[sub_int]
    E -->|乗算| H[mul_int]
    E -->|除算| I[checked_sdiv_int / checked_udiv_int]
    E -->|ビット演算| J[and_int / or_int / xor_int]
    F --> K[結果返却]
    G --> K
    H --> K
    I --> K
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | オーバーフロー | デフォルトの整数演算はオーバーフロー時にラップアラウンドする（モジュラ演算） | 基本算術演算 |
| BR-02 | ゼロ除算 | 整数のゼロ除算はDivideErrorを投げる | div/rem時 |
| BR-03 | 整数除算 | /演算子は整数同士でもfloat結果を返す | /使用時 |
| BR-04 | 型保持 | 同一型の演算結果は同じ型を返す | 同一型演算時 |
| BR-05 | 符号なし減算 | 符号なし整数の減算でアンダーフローするとラップアラウンドする | UInt演算時 |

### 計算ロジック

- `+(x::T, y::T) where {T<:BitInteger} = add_int(x, y)` （int.jl 87行目）
- `-(x::T, y::T) where {T<:BitInteger} = sub_int(x, y)` （int.jl 86行目）
- `*(x::T, y::T) where {T<:BitInteger} = mul_int(x, y)` （int.jl 88行目）
- `div(x::T, y::T) where {T<:BitSigned64} = checked_sdiv_int(x, y)` （int.jl 341行目）

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（データベースを使用しない）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| DivideError | DivideError | ゼロ除算 | ゼロでない値で除算する |
| OverflowError | OverflowError | checked_xxx系関数でのオーバーフロー | BigIntを使用するかオーバーフローを許容する |
| InexactError | InexactError | 変換先型で表現不可能な値 | より広い型を使用する |

### リトライ仕様

算術演算は純粋関数であり、リトライは適用されない。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- 基本整数演算はCPUネイティブ命令に直接マッピングされ、1クロックサイクル程度で実行される
- 整数除算は複数サイクルかかるがハードウェア除算器を使用する

## セキュリティ考慮事項

- 整数オーバーフローによるセキュリティ脆弱性に注意（バッファサイズ計算等）
- 信頼できない入力に対してはchecked_xxx系関数の使用を推奨

## 備考

- int.jlの10-40行目でBitSigned/BitUnsigned/BitInteger等の型エイリアスが定義されている
- intfuncs.jlにgcd, lcm等の数論関数が定義されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

整数型のグループ分けを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | int.jl | `base/int.jl` | **10-24行目**: BitSigned_types, BitUnsigned_types, BitInteger_types等の型タプル定義 |
| 1-2 | int.jl | `base/int.jl` | **26-40行目**: BitSigned, BitUnsigned, BitInteger等のUnion型エイリアス |

**読解のコツ**: 整数型は32ビット以下と64ビット以下の型グループに分類されている。32ビットマシンとの互換性のため、`BitSignedSmall_types` 等のプラットフォーム依存グループも存在する。

#### Step 2: エントリーポイントを理解する

基本算術演算の定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | int.jl | `base/int.jl` | **83行目**: `(<)(x::T, y::T) where {T<:BitSigned} = slt_int(x, y)` - 符号付き比較 |
| 2-2 | int.jl | `base/int.jl` | **85-88行目**: -, +, * 演算子の定義。neg_int, sub_int, add_int, mul_int を使用 |
| 2-3 | int.jl | `base/int.jl` | **94-97行目**: inv, / の定義。整数除算はfloatに変換 |

**主要処理フロー**:
1. **85行目**: `(-)(x::BitInteger) = neg_int(x)` - 単項マイナス
2. **86行目**: `(-)(x::T, y::T) where {T<:BitInteger} = sub_int(x, y)` - 減算
3. **87行目**: `(+)(x::T, y::T) where {T<:BitInteger} = add_int(x, y)` - 加算
4. **88行目**: `(*)(x::T, y::T) where {T<:BitInteger} = mul_int(x, y)` - 乗算
5. **341-344行目**: div/rem の定義。checked_sdiv_int/checked_srem_int を使用

#### Step 3: ビット演算と符号操作を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | int.jl | `base/int.jl` | **177-194行目**: signbit, flipsign, copysign の定義 |
| 3-2 | int.jl | `base/int.jl` | **227-228行目**: abs関数の定義 |
| 3-3 | int.jl | `base/int.jl` | **230行目**: `~(n::Integer) = -n-1` ビット反転 |
| 3-4 | int.jl | `base/int.jl` | **255-267行目**: unsigned/signed 変換関数 |

#### Step 4: 整数論関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | intfuncs.jl | `base/intfuncs.jl` | **44-51行目**: gcd関数（ユークリッドの互除法） |
| 4-2 | intfuncs.jl | `base/intfuncs.jl` | **80-95行目**: _gcd関数（Steinのバイナリアルゴリズム） |

### プログラム呼び出し階層図

```
算術演算子 (+, -, *, div, rem)
    │
    ├─ 同一型メソッド (int.jl:85-88)
    │      └─ Core.Intrinsics (add_int, sub_int, mul_int)
    │             └─ CPUネイティブ命令
    │
    ├─ 異なる型メソッド
    │      └─ promote(x, y) → 同一型メソッド
    │
    └─ 整数除算 (int.jl:341-344)
           ├─ checked_sdiv_int (符号付き)
           └─ checked_udiv_int (符号なし)

ビット演算 (&, |, xor, <<, >>)
    │
    └─ Core.Intrinsics (and_int, or_int, xor_int, shl_int, lshr_int, ashr_int)

数論関数 (gcd, lcm)
    │
    └─ _gcd() (intfuncs.jl:80-95)
           └─ Steinのバイナリアルゴリズム
```

### データフロー図

```
[入力]                     [処理]                          [出力]

整数値 x, y          ───▶ 型ディスパッチ             ───▶ 適切なメソッド選択
                           │
                           ▼
同一型の場合          ───▶ Core.Intrinsics関数       ───▶ CPU命令実行
異なる型の場合        ───▶ promote → Intrinsics      ───▶ 昇格型での結果

整数除算 x/y         ───▶ float(x)/float(y)         ───▶ Float64結果
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| int.jl | `base/int.jl` | ソース | 整数演算の主要定義（算術・ビット・符号・比較） |
| intfuncs.jl | `base/intfuncs.jl` | ソース | 整数論関数（gcd, lcm, powermod等） |
| boot.jl | `base/boot.jl` | ソース | 整数型の定義 |
| promotion.jl | `base/promotion.jl` | ソース | 異なる型間の型昇格 |
| checked.jl | `base/checked.jl` | ソース | オーバーフロー検出付き演算 |
